/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.services;

import android.content.Context;
import android.location.Location;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.work.Data;
import androidx.work.ExistingPeriodicWorkPolicy;
import androidx.work.PeriodicWorkRequest;
import androidx.work.WorkManager;
import androidx.work.Worker;
import androidx.work.WorkerParameters;

import com.google.android.gms.location.FusedLocationProviderClient;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationServices;
import com.google.android.gms.tasks.CancellationToken;
import com.google.android.gms.tasks.OnTokenCanceledListener;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import ly.warp.sdk.BuildConfig;
import ly.warp.sdk.Warply;
import ly.warp.sdk.dexter.listener.DexterError;
import ly.warp.sdk.dexter.listener.PermissionDeniedResponse;
import ly.warp.sdk.io.callbacks.CallbackReceiver;
import ly.warp.sdk.io.models.WarpGeoFence;
import ly.warp.sdk.receivers.LocationChangedReceiver;
import ly.warp.sdk.utils.PermissionsUtil;
import ly.warp.sdk.utils.WarpJSONParser;
import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.WarplyPreferences;
import ly.warp.sdk.utils.constants.WarpConstants;
import ly.warp.sdk.utils.managers.WarplyLocationManager;

public class UpdateUserLocationService extends Worker {

    public static final String TAG = "LOCATION_KEY";

    private int lastForProvider;
    private float lastForTime;
    private double lastForDistance;
    private int lastBacProvider;
    private float lastBacTime;
    private double lastBacDistance;
    private boolean geofencingEnabled = false;

    //Geofences
    private List<WarpGeoFence> geofences = null;

    private static int defaultCheckInterval = 15;
    private final static int defaultInterval = 15;

    public UpdateUserLocationService(@NonNull Context context, @NonNull WorkerParameters workerParams) {
        super(context, workerParams);
    }

    @NonNull
    @Override
    public Result doWork() {
        if (Warply.getWarplyContext() != null) {
            if (new WarplyPreferences(Warply.getWarplyContext()).getAppStatus().equals("foreground")
                /*&& new WarplyPreferences(Warply.getWarplyContext()).getUpdateLocationServiceStatus().equals("on")*/) {
                try {
                    new PermissionsUtil(
                            Warply.getWarplyContext(),
                            mPermissionsCallback,
                            PermissionsUtil.PERMISSION_LOCATION_COARSE,
                            PermissionsUtil.PERMISSION_LOCATION_FINE
                    ).requestPermissions();
                } catch (SecurityException e) {
                    if (BuildConfig.DEBUG) {
                        e.printStackTrace();
                        WorkManager.getInstance(Warply.getWarplyContext()).cancelAllWorkByTag(TAG);
                    }
                }
            } else {
                WorkManager.getInstance(Warply.getWarplyContext()).cancelAllWorkByTag(TAG);
            }
        } else {
            WorkManager.getInstance(getApplicationContext()).cancelAllWorkByTag(TAG);
        }

        regulateLocationReporting();

        return Result.success();
    }

    private void sendLocationData(double lat, double lon) {
        JSONObject jObj = new JSONObject();
        try {
            jObj.putOpt("action", WarpConstants.MICROAPP_GEOFENCING_ACTION);
            jObj.putOpt("lat", lat);
            jObj.putOpt("lon", lon);

        } catch (JSONException e) {
            WarpUtils.log("JSON Exception thrown while creating the object in sendLocationData at WarplyLocationManager", e);
        }

        if (LocationChangedReceiver.wClient != null) {
            Warply.postReceiveMicroappData(WarpConstants.MICROAPP_GEOFENCING, jObj, new CallbackReceiver<JSONObject>() {
                @Override
                public void onSuccess(JSONObject result) {
                    WarpUtils.log("success sending location " + result.toString());
                }

                @Override
                public void onFailure(int errorCode) {
                    WarpUtils.log("failed during sending location data with error " + errorCode);
                }
            });
        } else {
            WarpUtils.log("Could not send user location data. You must registrer to warply first");
        }
    }

    private void sendLocation(Location location) {
        if (isGeofencingEnabled()) {
            if (isInsideFence(location.getLatitude(), location.getLongitude())) {
                sendLocationData(location.getLatitude(), location.getLongitude());
            }
        } else {
            sendLocationData(location.getLatitude(), location.getLongitude());
        }

        if (isGeofencingEnabled() && (getGeofences() == null || getGeofences().isEmpty()))
            retrieveGeofences();
    }

// ------------------- Geofencing -------------------

    private void retrieveGeofences() {
        if (!isGeofencingEnabled())
            return;

        JSONObject jObj = new JSONObject();
        try {
            jObj.putOpt("action", WarpConstants.MICROAPP_GEOFENCING_ACTION_GET_POIS);
        } catch (JSONException e) {
            WarpUtils.log("JSON Exception thrown while creating the object in retrieveGeofences at WarplyLocationManager", e);
            return;
        }

        if (LocationChangedReceiver.wClient != null) {
            Warply.postReceiveMicroappData(WarpConstants.MICROAPP_GEOFENCING, jObj, new CallbackReceiver<JSONObject>() {
                @Override
                public void onSuccess(JSONObject result) {
                    JSONObject json = result.optJSONObject("context");
                    if (json != null) {
                        try {
                            JSONArray geoArray = json.getJSONArray("MAPP_GEOFENCING");
                            geofences = new ArrayList<>();
                            for (int i = 0; i < geoArray.length(); i++) {
                                getGeofences().add(new WarpGeoFence(geoArray.getJSONObject(i)));
                            }
                        } catch (JSONException e) {
                            if (BuildConfig.DEBUG)
                                e.printStackTrace();
                            geofences = null;
                            WarpUtils.log("failed during parsing geofencing data.");
                        }
                    }
                }

                @Override
                public void onFailure(int errorCode) {
                    geofences = null;
                    WarpUtils.log("failed during getting geofencing data with error " + errorCode);
                }
            });
        } else {
            WarpUtils.log("Could not get geofencing data. You must registrer to warply first");
        }
    }

    public boolean isInsideFence(double lat, double lon) {
        if (getGeofences() == null)
            return false;
        for (WarpGeoFence fence : getGeofences()) {
            if (fence.isInsideFence(lat, lon))
                return true;
        }
        return false;
    }


// ------------------- Location Modes -------------------

    private void regulateLocationReporting() {
        Warply.getContext(new CallbackReceiver<JSONObject>() {
            @Override
            public void onSuccess(JSONObject result) {
                JSONObject json = result.optJSONObject("context");
                if (json != null) {
                    JSONObject appDataJson = json.optJSONObject("application_data");
                    if (appDataJson != null) {
                        // location settings
                        int foregroundProvider = WarpJSONParser
                                .getIntFromJSONObject(appDataJson, WarpConstants.LocationSetting.LOCATION_FOREGROUND_MODE.getJsonKey(), 0);
                        long foregroundMinTime = WarpJSONParser
                                .getIntFromJSONObject(appDataJson, WarpConstants.LocationSetting.FOREGROUND_MIN_TIME.getJsonKey(), 60000);
                        float foregroundMinDistance = WarpJSONParser
                                .getIntFromJSONObject(appDataJson, WarpConstants.LocationSetting.FOREGROUND_MIN_DISTANCE.getJsonKey(), 100);
                        int backgroundProvider = WarpJSONParser
                                .getIntFromJSONObject(appDataJson, WarpConstants.LocationSetting.LOCATION_BACKGROUND_MODE.getJsonKey(), 0);
                        long backgroundMinTime = WarpJSONParser
                                .getIntFromJSONObject(appDataJson, WarpConstants.LocationSetting.BACKGROUND_MIN_TIME.getJsonKey(), 60000);
                        float backgroundMinDistance = WarpJSONParser
                                .getIntFromJSONObject(appDataJson, WarpConstants.LocationSetting.BACKGROUND_MIN_DISTANCE.getJsonKey(), 100);
                        int interval = WarpJSONParser.getIntFromJSONObject(appDataJson, WarpConstants.LocationSetting.FEATURES_CHECK_INTERVAL.getJsonKey(), 86400);
                        geofencingEnabled = WarpJSONParser.getBooleanFromJSONObject(appDataJson, WarpConstants.LocationSetting.GEOFENCING_POIS_ENABLED.getJsonKey(), false);

                        Log.i("notification interval", Integer.toString(interval));

                        if (interval != defaultCheckInterval) {
                            defaultCheckInterval = interval;
                        }

                        Context context = Warply.getWarplyContext();
                        String appStatus = new WarplyPreferences(context).getAppStatus();

                        if (foregroundProvider == WarpConstants.LocationModes.OFF.ordinal()
                                && backgroundProvider == WarpConstants.LocationModes.OFF.ordinal()) {

                            WarplyLocationManager.stopReportingLocation(context);
//                            WarplyLocationManager.disableConnectivityChangedReceiver(context);

                        } else if (foregroundProvider == WarpConstants.LocationModes.OFF.ordinal()) {
                            /*
                             * in case foreground provider is off and app is in
                             * foreground mode then app should stop reporting the
                             * location
                             */
                            if (!LocationChangedReceiver.isAppShutDown && appStatus.equals("foreground")) {
                                WarplyLocationManager.stopReportingLocation(context);
                                /*
                                 * in case background provider is on and app is in
                                 * background mode or shut down then app should check for
                                 * settings changes in every fixed time(a day). If none of
                                 * the settings is changed then continue with the previous
                                 * settings.
                                 */
                            } else if (!((lastBacProvider == backgroundProvider)
                                    && (lastBacTime == backgroundMinTime)
                                    && (lastBacDistance == backgroundMinDistance))) {

                                if (WarplyLocationManager.getLocationManager() == null)
                                    WarplyLocationManager.instantiateLocationManager();

                                WarplyLocationManager.startReportingLocation(context,
                                        WarpConstants.LocationModes.values()[backgroundProvider].getJsonKey(),
                                        backgroundMinTime * 1000,
                                        backgroundMinDistance);
                                lastBacProvider = backgroundProvider;
                                lastBacTime = backgroundMinTime;
                                lastBacDistance = backgroundMinDistance;
                            }


                        } else if (backgroundProvider == WarpConstants.LocationModes.OFF.ordinal()) {
                            /*
                             * in case background provider is off and app is in
                             * background mode then app should stop reporting the
                             * location
                             */
                            if (LocationChangedReceiver.isAppShutDown || appStatus.equals("background")) {
                                WarplyLocationManager.stopReportingLocation(context);
                                /*
                                 * in case foreground provider is on and app is in
                                 * foreground mode then app should check for settings
                                 * changes in every fixed time(a day). If none of the
                                 * settings is changed then continue with the previous
                                 * settings.
                                 */
                            } else if (!((lastForProvider == foregroundProvider)
                                    && (lastForTime == foregroundMinTime)
                                    && (lastForDistance == foregroundMinDistance))) {

                                WarplyLocationManager.startReportingLocation(
                                        context,
                                        WarpConstants.LocationModes.values()[foregroundProvider].getJsonKey(),
                                        foregroundMinTime * 1000,
                                        foregroundMinDistance);

                                lastForProvider = foregroundProvider;
                                lastForTime = foregroundMinTime;
                                lastForDistance = foregroundMinDistance;

                            }
                        } else {   // cases foreground and background providers are on
                            if (!LocationChangedReceiver.isAppShutDown && appStatus.equals("foreground")) {
                                if (!((lastForProvider == foregroundProvider)
                                        && (lastForTime == foregroundMinTime)
                                        && (lastForDistance == foregroundMinDistance))) {

                                    WarplyLocationManager.startReportingLocation(
                                            context,
                                            WarpConstants.LocationModes.values()[foregroundProvider].getJsonKey(),
                                            foregroundMinTime * 1000,
                                            foregroundMinDistance);

                                    lastForProvider = foregroundProvider;
                                    lastForTime = foregroundMinTime;
                                    lastForDistance = foregroundMinDistance;

                                }
                            } else if (!((lastBacProvider == backgroundProvider)
                                    && (lastBacTime == backgroundMinTime) && (lastBacDistance == backgroundMinDistance))) {
                                if (WarplyLocationManager.getLocationManager() == null)
                                    WarplyLocationManager.instantiateLocationManager();

                                WarplyLocationManager.startReportingLocation(
                                        context,
                                        WarpConstants.LocationModes.values()[backgroundProvider].getJsonKey(),
                                        backgroundMinTime * 1000,
                                        backgroundMinDistance);

                                lastBacProvider = backgroundProvider;
                                lastBacTime = backgroundMinTime;
                                lastBacDistance = backgroundMinDistance;

                            }
                        } // end else foregroundProvider = on and backgroundProvider =
                        // on
                    }
                }
            }

            @Override
            public void onFailure(int errorCode) {
                WarpUtils.log("Error with code " + errorCode + " while checking location modes");
            }
        });
    }

    public synchronized List<WarpGeoFence> getGeofences() {
        return geofences;
    }

    public synchronized boolean isGeofencingEnabled() {
        return geofencingEnabled;
    }

    private static PeriodicWorkRequest buildWorkRequest(Map<String, Object> location) {
        Data inputData = new Data.Builder().putAll(location).build();
        return new PeriodicWorkRequest.Builder(UpdateUserLocationService.class, defaultInterval, TimeUnit.MINUTES)
                .addTag(TAG)
                .setInputData(inputData)
                .build();
    }

    public static void scheduleWork(Context context, Map<String, Object> location) {
        WorkManager.getInstance(context).enqueueUniquePeriodicWork(
                TAG,
                ExistingPeriodicWorkPolicy.KEEP,
                buildWorkRequest(location));
    }

    private final PermissionsUtil.PermissionCallback mPermissionsCallback = new PermissionsUtil.PermissionCallback() {
        @Override
        public void onError(DexterError error) {
            onPermissionDenied(null);
        }

        @Override
        public void onPermissionDenied(List<PermissionDeniedResponse> denied) {
            if (denied != null && denied.size() < 2)//1 location permission accepted
                onPermissionsGranted();
        }

        @Override
        public void onPermissionsGranted() {
            try {
                FusedLocationProviderClient locationClient = LocationServices.getFusedLocationProviderClient(Warply.getWarplyContext());

                locationClient.getCurrentLocation(LocationRequest.PRIORITY_HIGH_ACCURACY, new CancellationToken() {
                    @NonNull
                    @Override
                    public CancellationToken onCanceledRequested(@NonNull OnTokenCanceledListener onTokenCanceledListener) {
                        return null;
                    }

                    @Override
                    public boolean isCancellationRequested() {
                        return false;
                    }
                })
                        .addOnSuccessListener(location1 -> {
                            if (location1 != null) {
                                sendLocation(location1);
                            }
                        })
                        .addOnFailureListener(e -> {
                            WarpUtils.log("Could not get location data.");
                        });
            } catch (SecurityException e) {
                if (BuildConfig.DEBUG) {
                    e.printStackTrace();
                }
            }
        }
    };
}
