/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.services;

import android.app.AlarmManager;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Build;

import androidx.annotation.NonNull;
import androidx.core.app.NotificationCompat;

import ly.warp.sdk.receivers.WarplyInAppNotificationReceiver;
import ly.warp.sdk.utils.WarplyProperty;

/**
 * Created by Panos Triantafyllou on 14/03/2018.
 */
public class WarpInAppNotificationService extends WarpBaseIntentService {

    public final static String KEY_UID = "uid";
    public final static String KEY_TITLE = "title";
    public final static String KEY_SUBTITLE = "subtitle";
    public final static String KEY_BUTTON = "button";
    public final static String KEY_HEADS_UP = "heads_up";
    private final String TAG_NOTIFICATION = "notification";


    public WarpInAppNotificationService() {
        super();
    }

    @Override
    public void onHandleIntent(Intent intent) {
        int uid = intent.getIntExtra(KEY_UID, 0);
        String mNotificationTitle = intent.getStringExtra(KEY_TITLE);
        String mNotificationSubtitle = intent.getStringExtra(KEY_SUBTITLE);
        boolean mIsHeadsUp = intent.getBooleanExtra(KEY_HEADS_UP, false);

        Intent notifyIntent;
        NotificationCompat.Builder builder = new NotificationCompat.Builder(this);
        PackageManager pm = this.getPackageManager();

        builder.setContentTitle(mNotificationTitle);
        builder.setContentText(mNotificationSubtitle);
        builder.setSmallIcon(WarplyProperty.getPushIconResId(this));
        builder.setVisibility(NotificationCompat.VISIBILITY_PUBLIC);
        builder.setAutoCancel(true);
        builder.setDefaults(Notification.DEFAULT_ALL);

        if (mIsHeadsUp) {
            builder.setPriority(NotificationCompat.PRIORITY_HIGH);
        } else {
            builder.setPriority(NotificationCompat.PRIORITY_DEFAULT);
        }

        notifyIntent = pm.getLaunchIntentForPackage(this.getPackageName())
                .putExtra(KEY_UID, uid)
                .putExtra(KEY_TITLE, mNotificationTitle)
                .putExtra(KEY_SUBTITLE, mNotificationSubtitle);


        PendingIntent pendingIntent;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            pendingIntent = PendingIntent.getBroadcast(this, uid, notifyIntent, PendingIntent.FLAG_UPDATE_CURRENT | PendingIntent.FLAG_MUTABLE);
        } else {
            pendingIntent = PendingIntent.getBroadcast(this, uid, notifyIntent, PendingIntent.FLAG_UPDATE_CURRENT);
        }
        builder.setContentIntent(pendingIntent);
//        builder.setFullScreenIntent(pendingIntent, true); // Heads up notification, always visible OR setPriority to HIGH and setContentIntent
        NotificationManager nm = (NotificationManager) this.getSystemService(NOTIFICATION_SERVICE);
        Notification notification_build = builder.build();
        notification_build.flags = Notification.FLAG_AUTO_CANCEL;
        nm.notify(TAG_NOTIFICATION, uid, notification_build);
    }

    public static void createInAppNotificationAlarm(Context context, @NonNull int uid, long interval, String title, String subtitle, boolean isHeadsUpNotification) {
        Intent notifyIntent = new Intent(context, WarplyInAppNotificationReceiver.class)
                .putExtra(KEY_UID, uid)
                .putExtra(KEY_TITLE, title)
                .putExtra(KEY_SUBTITLE, subtitle)
                .putExtra(KEY_HEADS_UP, isHeadsUpNotification);
        PendingIntent pendingIntent;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            pendingIntent = PendingIntent.getBroadcast(context, uid, notifyIntent, PendingIntent.FLAG_UPDATE_CURRENT | PendingIntent.FLAG_MUTABLE);
        } else {
            pendingIntent = PendingIntent.getBroadcast(context, uid, notifyIntent, PendingIntent.FLAG_UPDATE_CURRENT);
        }
        AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
        alarmManager.setRepeating(AlarmManager.RTC_WAKEUP, System.currentTimeMillis(), interval, pendingIntent);
    }

    public static void cancelInAppNotificationAlarm(Context context, @NonNull int uid) {
        Intent notifyIntent = new Intent(context, WarplyInAppNotificationReceiver.class).putExtra(KEY_UID, uid);
        PendingIntent pendingIntent;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            pendingIntent = PendingIntent.getBroadcast(context, uid, notifyIntent, PendingIntent.FLAG_CANCEL_CURRENT | PendingIntent.FLAG_MUTABLE);
        } else {
            pendingIntent = PendingIntent.getBroadcast(context, uid, notifyIntent, PendingIntent.FLAG_CANCEL_CURRENT);
        }
        AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
        alarmManager.cancel(pendingIntent);
        pendingIntent.cancel();

        // **    OR   **  //
        // If there is already an alarm set up then cancel it
//        if (checkIfAlarmIsActive(context, uid)) {
//            pendingIntent.cancel();
//        }
    }

    public static boolean checkIfAlarmIsActive(Context context, int uid) {
        Intent checkIntent = new Intent(context, WarplyInAppNotificationReceiver.class).putExtra(KEY_UID, uid);

        // If there is an alarm already set up then returns true
        return (PendingIntent.getBroadcast(context, uid, checkIntent, PendingIntent.FLAG_NO_CREATE) != null);
    }

}