package ly.warp.sdk.services;

import android.app.Service;
import android.content.Intent;
import android.os.Binder;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.IBinder;
import android.os.RemoteException;
import android.text.TextUtils;

import org.altbeacon.beacon.Beacon;
import org.altbeacon.beacon.BeaconConsumer;
import org.altbeacon.beacon.BeaconManager;
import org.altbeacon.beacon.RangeNotifier;
import org.altbeacon.beacon.Region;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CountDownLatch;

import ly.warp.sdk.Warply;
import ly.warp.sdk.io.callbacks.CallbackReceiver;
import ly.warp.sdk.io.models.BeaconMessage;
import ly.warp.sdk.io.models.PushCampaign;
import ly.warp.sdk.receivers.WarplyBeaconsApplication;
import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * A {@link Service} used to catch transmitting {@link Beacon}s and show the appropriate
 * Notification. This Service must be specified on the application's AndroidManifest.xml file:
 * <p/>
 * {@code <service android:name="ly.warp.sdk.services.WarplyBeaconsRangingService" />}
 */
public class WarplyBeaconsRangingService extends Service implements BeaconConsumer {

    /**
     * the BeaconConsumer is connected to the application.
     * If application is dead then BeaconConsumer unbound from any app components
     * (or activity or service). I.e. this service can be run but listener will unbind
     * (nothing will received). But AltBeacons library promises to recreate application
     * asap and as result this service too.
     */

    // ===========================================================
    // Constants
    // ===========================================================

    private static final String KEY_STATUS = "status";
    private static final String KEY_CONTEXT = "context";
    private static final String KEY_MAPP_BEACON = "MAPP_BEACON";

    private static final int mResponseStatusOk = 1;

    // ===========================================================
    // Fields
    // ===========================================================

    private final IBinder mBinder = new BeaconsRangingServiceBinder();
    private BeaconManager mBeaconManager;

    private Map<String, Long> mSentBeacons = Collections.synchronizedMap(new HashMap<String, Long>());
    private int mMinDistanceToSend = 0;
    private long mMinTimeToSend = 0;

    private Handler mSendMessageHandler;
    private CountDownLatch mSendMessageLatch;

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public void onCreate() {
        super.onCreate();

        Warply.getInitializer(this).init();
        mBeaconManager = BeaconManager.getInstanceForApplication(this.getApplicationContext());
        startBeaconListener();

        HandlerThread thread = new HandlerThread("WarplyBeaconsRangingServiceThread");
        thread.start();
        mSendMessageHandler = new Handler(thread.getLooper());

        WarpUtils.log(WarplyBeaconsApplication.TAG + "RangingService : onCreate");
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        stopBeaconListener();
        WarpUtils.log(WarplyBeaconsApplication.TAG + "RangingService : onDestroy");
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {

        mBeaconManager.setForegroundScanPeriod(WarplyBeaconsApplication.getBeaconScanPeriod());
        if (mBeaconManager.isBound(this)) {
            mBeaconManager.setBackgroundMode(false);
        }
        mMinDistanceToSend = WarplyBeaconsApplication.getBeaconDistanceToSendWarply();
        mMinTimeToSend = WarplyBeaconsApplication.getBeaconTimeIntervalToSendWarply();

        WarpUtils.log(WarplyBeaconsApplication.TAG + "RangingService : onStartCommand");
        return START_NOT_STICKY;
    }

    @Override
    public IBinder onBind(Intent intent) {
        return mBinder;
    }

    @Override
    public void onBeaconServiceConnect() {

        mBeaconManager.setRangeNotifier(new RangeNotifier() {
            @Override
            public void didRangeBeaconsInRegion(final Collection<Beacon> beacons, Region region) {
                if (beacons.size() > 0) {

                    WarpUtils.log(WarplyBeaconsApplication.TAG + "RangingService : didRangeBeaconsInRegion");
                    // send only first nearby beacon
                    sendBeaconMessageToWarplyServer(beacons.iterator().next());
                }
            }
        });

        try {
            mBeaconManager.startRangingBeaconsInRegion(new Region("RangingRegion", null, null, null));
        } catch (RemoteException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
    }

    // ===========================================================
    // Methods
    // ===========================================================

    private synchronized void sendBeaconMessageToWarplyServer(Beacon beacon) {

        if (beacon != null && beacon.getDistance() <= mMinDistanceToSend) {
            await();
            mSendMessageHandler.post(new SendMessageRunnable(beacon));
        }
    }

    private void await() {

        if (mSendMessageLatch != null) {
            try {
                mSendMessageLatch.await();
            } catch (InterruptedException e) {
                if (WarpConstants.DEBUG)
                    e.printStackTrace();
            }
        }
    }

    private void lock() {

        if (mSendMessageLatch == null || mSendMessageLatch.getCount() == 0) {
            mSendMessageLatch = new CountDownLatch(1);
        }
    }

    private void unlock() {
        if (mSendMessageLatch != null) {
            mSendMessageLatch.countDown();
        }
    }

    private void showNotification(JSONObject response) {

        /*{"status":"1","context":{"MAPP_BEACON":"[{\"sound\": \"default\", " +
                "\"subtitle\": null, " +
				"\"session_uuid\": \"f9dafc0c05f911e5aca30003ff436ff8\", " +
				"\"loyalty-action\": 0, \"message\": \"Hello Alex\", " +
				"\"badge\": \"0\", \"extra_fields\": {}, \"alert\": \"Hello Alex\"}]",
				"events_processed":1,"MAPP_BEACON-status":1}}*/

        if (response != null) {
            if (response.optInt(KEY_STATUS) == mResponseStatusOk) {

                JSONObject json = response.optJSONObject(KEY_CONTEXT);
                if (json != null) {
                    String pushParams = json.optString(KEY_MAPP_BEACON);
                    if (!TextUtils.isEmpty(pushParams)) {
                        try {
                            // cut square brackets
                            pushParams = pushParams.substring(1, pushParams.length() - 1);
                            json = new JSONObject(pushParams);
                            final PushCampaign pc = new PushCampaign(json);
                            if (pc.isFilled()) {
                                new Thread(new Runnable() {
                                    @Override
                                    public void run() {
                                        WarpBaseIntentService.
                                                showCampaignNotification(WarplyBeaconsRangingService.this, pc);
                                    }
                                }).start();
                            }
                        } catch (JSONException e) {
                            if (WarpConstants.DEBUG) {
                                e.printStackTrace();
                            }
                        }
                    }
                }
            }
        }
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    /**
     * service interface for start listening beacons
     */

    public void startBeaconListener() {
        WarpUtils.log(WarplyBeaconsApplication.TAG + "RangingService : startBeaconListener");
        mBeaconManager.bind(this);
    }

    /**
     * service interface for stop listening beacons
     */
    public void stopBeaconListener() {
        WarpUtils.log(WarplyBeaconsApplication.TAG + "RangingService : stopBeaconListener");
        mBeaconManager.unbind(this);
    }

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

    public class BeaconsRangingServiceBinder extends Binder {
        public WarplyBeaconsRangingService getService() {
            // Return this instance of LocalService so clients can call public methods
            return WarplyBeaconsRangingService.this;
        }
    }

    private class SendMessageRunnable implements Runnable {

        private Beacon mSendBeacon;

        public SendMessageRunnable(Beacon beacon) {
            this.mSendBeacon = beacon;
        }

        @Override
        public void run() {

            BeaconMessage beaconMessage = new BeaconMessage(mSendBeacon);
            String messageSignature = beaconMessage.getSignature();
            Long lastSendTimeForBeacon = mSentBeacons.get(messageSignature);
            if (lastSendTimeForBeacon == null) {
                lastSendTimeForBeacon = 0L;
            }

            if (((System.nanoTime() - lastSendTimeForBeacon) / 1000000) > mMinTimeToSend) {

                WarpUtils.log(WarplyBeaconsApplication.TAG + "RangingService : post beacon: "
                        + messageSignature + " " + mSendBeacon.getDistance());

                final String messageSignatureFinal = messageSignature;
                Warply.postReceiveMicroappData(WarpConstants.MICROAPP_BEACON,
                        beaconMessage.toJson(), new CallbackReceiver<JSONObject>() {

                            @Override
                            public void onSuccess(JSONObject jsonObject) {

                                showNotification(jsonObject);
                                mSentBeacons.put(messageSignatureFinal, System.nanoTime());

                                WarpUtils.log(WarplyBeaconsApplication.TAG + "RangingService : post beacon: "
                                        + messageSignatureFinal + " (success)");

                                unlock();
                            }

                            @Override
                            public void onFailure(int i) {

                                WarpUtils.log(WarplyBeaconsApplication.TAG + "RangingService : post beacon: "
                                        + messageSignatureFinal + " (failure)=" + i);

                                unlock();
                            }
                        });

                lock();
            }
        }
    }
}
