package ly.warp.sdk.services;

import android.app.AlarmManager;
import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.os.Build;
import android.os.IBinder;
import android.os.SystemClock;

import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.core.app.NotificationCompat;

import org.greenrobot.eventbus.EventBus;
import org.json.JSONObject;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

import ly.warp.sdk.R;
import ly.warp.sdk.Warply;
import ly.warp.sdk.io.callbacks.CallbackReceiver;
import ly.warp.sdk.io.models.HealthEventModel;
import ly.warp.sdk.io.models.PacingDetails;
import ly.warp.sdk.io.request.PacingCalculateRequest;
import ly.warp.sdk.io.request.PacingDetailsRequest;
import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.WarplyManagerHelper;
import ly.warp.sdk.utils.managers.WarplyEventBusManager;
import ly.warp.sdk.utils.managers.WarplyManager;

/**
 * Created by Panagiotis Triantafyllou on 03/Aug/2022.
 */
public class WarplyHealthService extends Service implements SensorEventListener {

    // ===========================================================
    // Constants
    // ===========================================================

    private static final String STEPS_CHANNEL_ID = "stepschannel";

    // ===========================================================
    // Fields
    // ===========================================================

    private SensorManager mSensorManager;
    private Sensor mSensor;
    private final int ACCEL_RING_SIZE = 50;
    private final int VEL_RING_SIZE = 10;
    // change this threshold according to your sensitivity preferences
    private final float STEP_THRESHOLD = 20f; // default 40, cosmote default 20
    private final int STEP_DELAY_NS = 250000000;
    private int accelRingCounter = 0;
    private float[] accelRingX = new float[ACCEL_RING_SIZE];
    private float[] accelRingY = new float[ACCEL_RING_SIZE];
    private float[] accelRingZ = new float[ACCEL_RING_SIZE];
    private int velRingCounter = 0;
    private float[] velRing = new float[VEL_RING_SIZE];
    private long lastStepTimeNs = 0;
    private float oldVelocityEstimate = 0;
    private double eventMeters = 0.0d, pacingMeters = 0.0d;
    private int sumSteps = 0;

    // ===========================================================
    // Contructors
    // ===========================================================

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================


    @Override
    public void onCreate() {
        super.onCreate();

        mSensorManager = (SensorManager) getSystemService(Context.SENSOR_SERVICE);
        mSensor = mSensorManager.getDefaultSensor(Sensor.TYPE_ACCELEROMETER);
//        registerStepSensor();
        Warply.getInitializer(getApplicationContext()).init();
        WarplyManager.getPacingDetails(new PacingDetailsRequest(), mPacingCallback);
    }

    //    @RequiresApi(api = Build.VERSION_CODES.O)
    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        NotificationManager notificationManager = (NotificationManager) getSystemService(NOTIFICATION_SERVICE);
        PackageManager pm = getPackageManager();
        Intent bIntent = pm.getLaunchIntentForPackage(getPackageName());
        PendingIntent pbIntent = PendingIntent.getActivity(this, 2001, bIntent, PendingIntent.FLAG_IMMUTABLE);

        NotificationCompat.Builder b = new NotificationCompat.Builder(this, STEPS_CHANNEL_ID);
        b.setDefaults(Notification.DEFAULT_VIBRATE | Notification.DEFAULT_LIGHTS);
        b.setChannelId(STEPS_CHANNEL_ID);
        b.setContentTitle(getString(R.string.cos_steps_for_good_notification_title));
//        b.setContentText(getString(R.string.cos_steps_for_good_notification_subtitle));
        b.setStyle(new NotificationCompat.BigTextStyle().bigText(getString(R.string.cos_steps_for_good_notification_subtitle)));
        b.setContentIntent(pbIntent);
        b.setAutoCancel(true);
        b.setOngoing(true);
//        b.setCategory(Notification.CATEGORY_SERVICE);
        b.setSmallIcon(R.drawable.ic_launcher);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O && notificationManager != null) {
            NotificationChannel notificationChannel = new NotificationChannel(STEPS_CHANNEL_ID, "steps_notification_channel", NotificationManager.IMPORTANCE_DEFAULT);
            notificationManager.createNotificationChannel(notificationChannel);
            Notification notification_build = b.setChannelId(STEPS_CHANNEL_ID).build();
            this.startForeground(2001, notification_build);
        } else {
            Notification notification_build = b.build();
            this.startForeground(2001, notification_build);
        }

        new Thread(
                () -> {
                    while (true) {
                        sendSteps();
                        try {
                            Thread.sleep(900000);
                        } catch (InterruptedException e) {
                            e.printStackTrace();
                        }
                    }
                }
        ).start();

//        return super.onStartCommand(intent, flags, startId);
        return Service.START_STICKY;
    }

    @Override
    public void onDestroy() {
        super.onDestroy();

        unregisterStepSensor();
        WarplyManagerHelper.mStepsWebview = 0;
        WarplyManagerHelper.mMetersWebview = 0.0d;
        eventMeters = 0.0d;
        pacingMeters = 0.0d;
        sumSteps = 0;
        sendSteps();
    }

//    @Override
//    public void onTaskRemoved(Intent rootIntent) {
//        Intent restartServiceIntent = new Intent(getApplicationContext(), WarplyHealthService.class);
//        restartServiceIntent.setPackage(getPackageName());
//
//        PendingIntent restartServicePendingIntent = PendingIntent.getService(getApplicationContext(), 2002, restartServiceIntent, PendingIntent.FLAG_ONE_SHOT);
//        AlarmManager alarmService = (AlarmManager) getApplicationContext().getSystemService(Context.ALARM_SERVICE);
//        alarmService.set(
//                AlarmManager.ELAPSED_REALTIME,
//                SystemClock.elapsedRealtime() + 1000,
//                restartServicePendingIntent
//        );
//
//        super.onTaskRemoved(rootIntent);
//    }

    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    @Override
    public void onSensorChanged(SensorEvent sensorEvent) {
        if (sensorEvent.sensor.getType() == Sensor.TYPE_ACCELEROMETER) {
            updateSteps(
                    sensorEvent.timestamp,
                    sensorEvent.values[0],
                    sensorEvent.values[1],
                    sensorEvent.values[2]);
        }
    }

    @Override
    public void onAccuracyChanged(Sensor sensor, int i) {

    }

    // ===========================================================
    // Methods
    // ===========================================================

    public void registerStepSensor() {
        mSensorManager.registerListener(this, mSensor, SensorManager.SENSOR_DELAY_GAME);
    }

    public void unregisterStepSensor() {
        mSensorManager.unregisterListener(this);
    }

    private void updateSteps(long timeNs, float x, float y, float z) {
        float[] currentAccel = new float[3];
        currentAccel[0] = x;
        currentAccel[1] = y;
        currentAccel[2] = z;

        // First step is to update our guess of where the global z vector is.
        accelRingCounter++;
        accelRingX[accelRingCounter % ACCEL_RING_SIZE] = currentAccel[0];
        accelRingY[accelRingCounter % ACCEL_RING_SIZE] = currentAccel[1];
        accelRingZ[accelRingCounter % ACCEL_RING_SIZE] = currentAccel[2];

        float[] worldZ = new float[3];
        worldZ[0] = sum(accelRingX) / Math.min(accelRingCounter, ACCEL_RING_SIZE);
        worldZ[1] = sum(accelRingY) / Math.min(accelRingCounter, ACCEL_RING_SIZE);
        worldZ[2] = sum(accelRingZ) / Math.min(accelRingCounter, ACCEL_RING_SIZE);

        float normalization_factor = norm(worldZ);

        worldZ[0] = worldZ[0] / normalization_factor;
        worldZ[1] = worldZ[1] / normalization_factor;
        worldZ[2] = worldZ[2] / normalization_factor;

        float currentZ = dot(worldZ, currentAccel) - normalization_factor;
        velRingCounter++;
        velRing[velRingCounter % VEL_RING_SIZE] = currentZ;

        float velocityEstimate = sum(velRing);

        if (velocityEstimate > STEP_THRESHOLD
                && oldVelocityEstimate <= STEP_THRESHOLD
                && (timeNs - lastStepTimeNs > STEP_DELAY_NS)) {
            WarplyManagerHelper.mStepsWebview++;
            sumSteps++;
            WarplyManagerHelper.mSteps++;
            WarpUtils.log("COUNT_STEPS: " + String.valueOf(WarplyManagerHelper.mSteps));
            WarpUtils.setStepsCounter(this, WarplyManagerHelper.mSteps);
            WarplyManagerHelper.mMetersWebview = (WarplyManagerHelper.mStepsWebview * 0.762);
            WarpUtils.log("WEBVIEW_METERS: " + String.valueOf(WarplyManagerHelper.mMetersWebview));
            HealthEventModel healthSteps = new HealthEventModel();
//            healthSteps.setMeters((mStepsAll * 0.762)); // 16/09/2022
//            healthSteps.setMeters(WarplyManagerHelper.mMetersWebview); // 20/09/2022
            eventMeters = (sumSteps * 0.762);
            WarpUtils.log("WIDGET_METERS: " + String.valueOf(pacingMeters + eventMeters));
            healthSteps.setMeters(pacingMeters + eventMeters);
            EventBus.getDefault().post(new WarplyEventBusManager(healthSteps));

            lastStepTimeNs = timeNs;
        }
        oldVelocityEstimate = velocityEstimate;
    }

    private float sum(float[] array) {
        float retval = 0;
        for (float v : array) {
            retval += v;
        }
        return retval;
    }

    private float[] cross(float[] arrayA, float[] arrayB) {
        float[] retArray = new float[3];
        retArray[0] = arrayA[1] * arrayB[2] - arrayA[2] * arrayB[1];
        retArray[1] = arrayA[2] * arrayB[0] - arrayA[0] * arrayB[2];
        retArray[2] = arrayA[0] * arrayB[1] - arrayA[1] * arrayB[0];
        return retArray;
    }

    private float norm(float[] array) {
        float retval = 0;
        for (float v : array) {
            retval += v * v;
        }
        return (float) Math.sqrt(retval);
    }

    private float dot(float[] a, float[] b) {
        return a[0] * b[0] + a[1] * b[1] + a[2] * b[2];
    }

    private float[] normalize(float[] a) {
        float[] retval = new float[a.length];
        float norm = norm(a);
        for (int i = 0; i < a.length; i++) {
            retval[i] = a[i] / norm;
        }
        return retval;
    }

    private void sendSteps() {
        String date = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss", Locale.getDefault()).format(new Date());

        if (WarpUtils.getStepsCounter(this) > 0) {
            WarplyManager.setPacingDetails(new PacingCalculateRequest()
                            .setCounter(WarpUtils.getStepsCounter(this))
                            .setDate(date),
                    new CallbackReceiver<JSONObject>() {
                        @Override
                        public void onSuccess(JSONObject result) {
                            int status = result.optInt("status", 2);
                            if (status == 1) {
                                WarplyManagerHelper.mSteps = 0;
                                WarpUtils.setStepsCounter(WarplyHealthService.this, 0);
                            }
                        }

                        @Override
                        public void onFailure(int errorCode) {
                            WarpUtils.log("Warply Health Service error: " + String.valueOf(errorCode));
                        }
                    });
        }
    }

    private CallbackReceiver<PacingDetails> mPacingCallback = new CallbackReceiver<PacingDetails>() {
        @Override
        public void onSuccess(PacingDetails result) {
            pacingMeters = result.getMeters().getDay().getValue();
            registerStepSensor();
        }

        @Override
        public void onFailure(int errorCode) {

        }
    };
}
