package ly.warp.sdk.utils;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.IBinder;
import android.os.IInterface;
import android.os.Looper;
import android.os.Parcel;
import android.os.RemoteException;

import java.lang.reflect.Method;
import java.util.concurrent.LinkedBlockingQueue;

import ly.warp.sdk.io.callbacks.SimpleCallbackReceiver;
import ly.warp.sdk.utils.constants.WarpConstants;

public class GoogleAdInfoClient {

    // ===========================================================
    // Constants
    // ===========================================================

    // ===========================================================
    // Fields
    // ===========================================================


    // ===========================================================
    // Constructors
    // ===========================================================

    // ===========================================================
    // Methods
    // ===========================================================

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public static GoogleAdInfo getGoogleAdvertisingInfo(Context context) {

        if (Looper.myLooper() == Looper.getMainLooper())
            throw new IllegalStateException("Cannot be called from the main thread");

        GoogleAdInfo info = firstTry(context);
        if (info == null) {
            info = secondTry(context);
        }
        return info;
    }

    private static GoogleAdInfo firstTry(Context context) {

        try {
            Class<?> GooglePlayServicesUtil = Class.forName("com.google.android.gms.common.GooglePlayServicesUtil");
            Method isGooglePlayServicesAvailable = GooglePlayServicesUtil.getMethod("isGooglePlayServicesAvailable", Context.class);
            if (isGooglePlayServicesAvailable.invoke(null, context).equals(0)) { // ConnectionResult.SUCCESS
                Class<?> AdvertisingClientId = Class.forName("com.google.android.gms.ads.identifier.AdvertisingIdClient");
                Method getAdvertisingIdInfo = AdvertisingClientId.getMethod("getAdvertisingIdInfo", Context.class);
                Object advertisingTrackingInfo = getAdvertisingIdInfo.invoke(null, context);

                Class<?> Info = Class.forName("com.google.android.gms.ads.identifier.AdvertisingIdClient$Info");
                Method getId = Info.getMethod("getId");
                String id = (String) getId.invoke(advertisingTrackingInfo);

                Method isLimitAdTrackingEnabled = Info.getMethod("isLimitAdTrackingEnabled");
                boolean limitAdTrackingEnabled = (Boolean) isLimitAdTrackingEnabled.invoke(advertisingTrackingInfo);

                return new GoogleAdInfo(id, limitAdTrackingEnabled);
            } else {
                return null;
            }
        } catch (Exception e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
            return null;
        }
    }

    private static GoogleAdInfo secondTry(Context context) {


        AdServiceConnection adServiceConnection = null;
        try {
            adServiceConnection = new AdServiceConnection();
            Intent intent = new Intent("com.google.android.gms.ads.identifier.service.START");
            intent.setPackage("com.google.android.gms");

            if (context.bindService(intent, adServiceConnection, Context.BIND_AUTO_CREATE)) {
                GoogleAdInfoFetcher adInfoFetcher = new GoogleAdInfoFetcher(adServiceConnection.getBinder());
                return adInfoFetcher.fetch();
            }
        } catch (Exception e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        } finally {
            if (adServiceConnection != null) {
                context.unbindService(adServiceConnection);
            }
        }
        return null;
    }

// ===========================================================
// Inner and Anonymous Classes
// ===========================================================

    private static class AdServiceConnection implements ServiceConnection {

        // for prevent multiply binders connection
        boolean retrieved = false;
        private final LinkedBlockingQueue<IBinder> bindersQueue = new LinkedBlockingQueue<>(1);

        @Override
        public void onServiceConnected(ComponentName name, IBinder service) {
            try {
                bindersQueue.put(service);
            } catch (InterruptedException e) {
                if (WarpConstants.DEBUG) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onServiceDisconnected(ComponentName name) {

        }

        public IBinder getBinder() throws InterruptedException {
            if (retrieved) {
                throw new IllegalStateException();
            }
            retrieved = true;
            return bindersQueue.take();
        }
    }

    private static class GoogleAdInfoFetcher implements IInterface {

        private IBinder mPlayServicesBinder;

        public GoogleAdInfoFetcher(IBinder playServicesBinder) {
            mPlayServicesBinder = playServicesBinder;
        }

        @Override
        public IBinder asBinder() {
            return mPlayServicesBinder;
        }

        public GoogleAdInfo fetch() {
            return new GoogleAdInfo(getId(), isLimitAdTrackingEnabled(true));
        }

        private String getId() {

            Parcel data = Parcel.obtain();
            Parcel reply = Parcel.obtain();
            String id = "";
            try {
                data.writeInterfaceToken("com.google.android.gms.ads.identifier.internal.IAdvertisingIdService");
                mPlayServicesBinder.transact(1, data, reply, 0);
                reply.readException();
                id = reply.readString();
            } catch (RemoteException e) {
                if (WarpConstants.DEBUG) {
                    e.printStackTrace();
                }
            } finally {
                reply.recycle();
                data.recycle();
            }
            return id;
        }

        private boolean isLimitAdTrackingEnabled(boolean paramBoolean) {

            Parcel data = Parcel.obtain();
            Parcel reply = Parcel.obtain();
            boolean limitAdTracking = false;
            try {
                data.writeInterfaceToken("com.google.android.gms.ads.identifier.internal.IAdvertisingIdService");
                data.writeInt(paramBoolean ? 1 : 0);
                mPlayServicesBinder.transact(2, data, reply, 0);
                reply.readException();
                limitAdTracking = 0 != reply.readInt();
            } catch (RemoteException e) {
                if (WarpConstants.DEBUG) {
                    e.printStackTrace();
                }
            } finally {
                reply.recycle();
                data.recycle();
            }
            return limitAdTracking;
        }
    }

    public static final class GoogleAdInfo {

        private final String advertisingId;
        private final boolean limitAdTrackingEnabled;

        GoogleAdInfo(String advertisingId, boolean limitAdTrackingEnabled) {
            this.advertisingId = advertisingId;
            this.limitAdTrackingEnabled = limitAdTrackingEnabled;
        }

        public String getId() {
            return this.advertisingId;
        }

        public boolean isLimitAdTrackingEnabled() {
            return this.limitAdTrackingEnabled;
        }
    }

}
