package ly.warp.sdk.utils;


import android.content.Context;
import android.text.TextUtils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import ly.warp.sdk.utils.constants.WarpConstants;

public class ObjectSerializer {

    // ===========================================================
    // Constants
    // ===========================================================

    private static final String WARPLY_SERIALIZE_CACHE_DIR = "WarplyCache";

    // ===========================================================
    // Fields
    // ===========================================================

    private Context mContext;

    // ===========================================================
    // Constructors
    // ===========================================================


    public ObjectSerializer(Context context) {
        this.mContext = context;
    }

    // ===========================================================
    // Methods
    // ===========================================================

    public static boolean externalMemoryAvailable() {
        return android.os.Environment.getExternalStorageState().equals(
                android.os.Environment.MEDIA_MOUNTED);
    }

    public static String externalStorageDirectory(Context context) {
        if (externalMemoryAvailable() && context != null) {
            File dir = context.getExternalFilesDir(null);
            if (dir != null) {
                return dir.getPath();
            }
        }
        return null;
    }
    public String getCachePath() {
        String externalDir = externalStorageDirectory(mContext);
        if (externalDir != null) {
            return externalDir + File.separator + WARPLY_SERIALIZE_CACHE_DIR;
        } else
            return null;
    }

    public String getPathInCache(String pathInsideCache) {
        String cachePath = getCachePath();
        if (cachePath != null && cachePath.length() > 0)
            return cachePath + File.separator + pathInsideCache;
        return null;
    }

    public File getCacheFile(String pathInsideCache) {

        File file = null;
        try {
            String cachePath = getPathInCache(pathInsideCache);
            if (cachePath != null)
                file = new File(cachePath);
        } catch (Exception e) {
            if (WarpConstants.DEBUG)
                e.printStackTrace();
        }
        return file;
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public boolean prepareDirInCache(String path) throws IOException {
        try {

            File f;
            String cachePath = getCachePath();
            if (TextUtils.isEmpty(cachePath))
                return false;

            f = new File(cachePath);
            if (!f.exists())
                f.mkdirs();

            if (path != null && path.length() > 0 && path.length() > cachePath.length()) {

                int lastSeparatorIndex = path.lastIndexOf(File.separator);
                String pathWithoutFile = "";
                if (lastSeparatorIndex > cachePath.length() + 1) {
                    pathWithoutFile = path.substring(cachePath.length() + 1, lastSeparatorIndex);
                }

                if (!new File(cachePath + File.separator + pathWithoutFile).exists()) {

                    String[] folders = pathWithoutFile.split(File.separator);
                    String currPathFolder = cachePath;

                    for (String folder : folders) {
                        currPathFolder += File.separator + folder;
                        f = new File(currPathFolder);
                        if (!f.exists())
                            f.mkdirs();
                    }
                }
            }

        } catch (Exception e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
            return false;
        }
        return true;
    }

    public boolean serialize(Object o, String pathInsideCache) {
        try {
            if (prepareDirInCache(pathInsideCache)) {
                File outputFile = getCacheFile(pathInsideCache);
                if (outputFile != null) {
                    ObjectOutputStream oos = new ObjectOutputStream(
                            new FileOutputStream(outputFile));
                    oos.writeObject(o);
                    oos.flush();
                    oos.close();
                    return true;
                }
            }
        } catch (Exception ex) {
            if (WarpConstants.DEBUG)
                ex.printStackTrace();
        }
        return false;
    }

    public Object deserialize(String pathInsideCache) {
        try {
            File f = getCacheFile(pathInsideCache);
            if (f != null) {
                ObjectInputStream ois = new ObjectInputStream(
                        new FileInputStream(f));
                Object o = ois.readObject();
                ois.close();
                return o;
            }
        } catch (Exception ex) {
            if (WarpConstants.DEBUG)
                ex.printStackTrace();
        }
        return null;
    }

}
