package ly.warp.sdk.utils;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Handler;
import android.os.HandlerThread;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;

import java.lang.ref.WeakReference;
import java.util.List;

import ly.warp.sdk.Warply;
import ly.warp.sdk.dexter.Dexter;
import ly.warp.sdk.dexter.DexterBuilder;
import ly.warp.sdk.dexter.MultiplePermissionsReport;
import ly.warp.sdk.dexter.PermissionToken;
import ly.warp.sdk.dexter.listener.DexterError;
import ly.warp.sdk.dexter.listener.PermissionDeniedResponse;
import ly.warp.sdk.dexter.listener.PermissionRequest;
import ly.warp.sdk.dexter.listener.PermissionRequestErrorListener;
import ly.warp.sdk.dexter.listener.multi.MultiplePermissionsListener;

/**
 * Created by Antonis Moustakos on 3/28/2017.
 */
public class PermissionsUtil{
    // ===========================================================
    // Constants
    // ===========================================================
    private static final int SLEEP_TIME = 5000;
    public static final String PERMISSION_LOCATION_FINE = android.Manifest.permission.ACCESS_FINE_LOCATION,
                                PERMISSION_LOCATION_COARSE = android.Manifest.permission.ACCESS_COARSE_LOCATION,
                                PERMISSION_PHONE_STATE = Manifest.permission.READ_PHONE_STATE,
                                SAFE_PERMISSION_BLUETOOTH = Manifest.permission.BLUETOOTH;
//                                PERMISSION_WRITE_EXTERNAL = Manifest.permission.WRITE_EXTERNAL_STORAGE;

    private static final String[] DANGEROUS_PERMISSIONS = {
//            PERMISSION_LOCATION_FINE,
//            PERMISSION_LOCATION_COARSE,
//            PERMISSION_PHONE_STATE,
//            PERMISSION_WRITE_EXTERNAL
    };

    // ===========================================================
    // Fields
    // ===========================================================
    private WeakReference<Context> context;
    private String[] requestedPerms;
    private PermissionCallback callback;

    private WarplyPreferences prefs;
    private Handler handler;

    // ===========================================================
    // Constructors
    // ===========================================================
    public PermissionsUtil(@NonNull Context context,
                           @Nullable PermissionCallback callback,
                           @Nullable String... requestedPerms) {
        this.prefs = new WarplyPreferences(context);
        this.context = new WeakReference<>(context);
        this.callback = callback;
        this.requestedPerms = requestedPerms;
        if(this.requestedPerms == null)
            this.requestedPerms = DANGEROUS_PERMISSIONS;

        HandlerThread mHandlerThread = new HandlerThread("PermissionsUtil");
        mHandlerThread.start();
        handler = new Handler(mHandlerThread.getLooper());
    }

    // ===========================================================
    // Methods
    // ===========================================================
    public boolean requestNeeded(){
        for(String p : requestedPerms)
            if(!hasPermission(context.get(), p))
                return true;
        return false;
    }
    public static boolean hasPermission(Context con, String permission){
        return ContextCompat.checkSelfPermission(con, permission) == PackageManager.PERMISSION_GRANTED;
    }
    public void requestPermissions(){
        final DexterBuilder builder =
            Dexter.withContext(context.get())
                .withPermissions(requestedPerms)
                .withListener(new MultiplePermissionsListener() {
                    @Override
                    public void onPermissionsChecked(MultiplePermissionsReport report) {
                        if(callback != null)
                            if(report.areAllPermissionsGranted())
                                callback.onPermissionsGranted();
                            else
                                callback.onPermissionDenied(report.getDeniedPermissionResponses());
                    }
                    @Override
                    public void onPermissionRationaleShouldBeShown(List<PermissionRequest> permissions, PermissionToken token) {
                        //TODO: Rationale can be shown here
                        token.continuePermissionRequest();
                    }
                })
                .withErrorListener(new PermissionRequestErrorListener() {
                    @Override
                    public void onError(DexterError error) {
                        if(callback != null)
                            callback.onError(error);
                    }
                })
                .onSameThread();

        handler.post(
            new Runnable() {
                public void run() {
                    while(!prefs.isForeground())
                        try {Thread.sleep(SLEEP_TIME);} catch (InterruptedException ignored) {}
                    builder.check();
                }
            }
        );
    }

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================
    public interface PermissionCallback{
        void onPermissionDenied(List<PermissionDeniedResponse> denied);
        void onPermissionsGranted();
        void onError(DexterError error);
    }

}
