/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.utils;

import androidx.annotation.NonNull;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

import ly.warp.sdk.utils.constants.WarpConstants;

public class WarpJSONParser {

    public static String getStringFromJSON(JSONObject jObject, String string, String defaultValue) {

        String jString = defaultValue;

        try {
            jString = jObject.getString(string);
        } catch (Exception e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
            jString = defaultValue;
        }

        return jString;

    }

    public static JSONObject getJSONFromString(String value) {

        JSONObject jObject = new JSONObject();
        try {
            jObject = new JSONObject(value);
        } catch (Exception e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }

        return jObject;

    }

    public static JSONObject getJObjectFromJSON(JSONObject jObject, String key) {

        JSONObject object = new JSONObject();

        try {
            object = jObject.getJSONObject(key);
        } catch (Exception e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }

        return object;

    }

    public static String getStringFromJSONArray(JSONArray jArray, int index, String defaultValue) {

        String string = defaultValue;

        try {
            string = jArray.getString(index);
        } catch (Exception e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
            string = defaultValue;
        }

        return string;
    }

    public static JSONArray getJSONArrayFromJson(JSONObject jObject, String string) {

        JSONArray jArray = new JSONArray();

        try {
            jArray = jObject.getJSONArray(string);
        } catch (Exception e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }

        return jArray;
    }

    public static JSONObject getJObjectFromJArray(JSONArray jArray, int index) {

        JSONObject jObject = new JSONObject();

        try {
            jObject = jArray.getJSONObject(index);
        } catch (Exception e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }

        return jObject;

    }

    public static int getIntFromJSONObject(JSONObject jObject, String string, int defaultValue) {

        int value = defaultValue;

        try {
            value = jObject.getInt(string);
        } catch (Exception e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
            value = defaultValue;
        }

        return value;

    }

    public static long getLongFromJSONObject(JSONObject jObject, String string, long defaultValue) {

        long value = defaultValue;

        try {
            value = jObject.getLong(string);
        } catch (Exception e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
            value = defaultValue;
        }

        return value;

    }

    public static double getDoubleFromJSONObject(JSONObject jObject, String string, double defaultValue) {

        double value = defaultValue;

        try {
            value = jObject.getDouble(string);
        } catch (Exception e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
            value = defaultValue;
        }

        return value;

    }

    public static float getFloatFromJSONObject(JSONObject jObject, String string, float defaultValue) {

        float value = defaultValue;

        try {
            value = (float) jObject.getDouble(string);
        } catch (Exception e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
            value = defaultValue;
        }

        return value;

    }

    public static boolean getBooleanFromJSONObject(JSONObject jObject, String string, boolean defaultValue) {

        boolean value = defaultValue;

        try {
            value = jObject.getBoolean(string);
        } catch (Exception e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
            value = defaultValue;
        }

        return value;

    }

    public static JSONObject createJSONObjectFromString(String key, String value) {

        JSONObject jObject = new JSONObject();

        try {
            jObject.putOpt(key, value);
        } catch (Exception e) {
            e.printStackTrace();
        }

        return jObject;
    }


    @NonNull
    public static <T> List<T> parseJsonToList(JSONObject json, String arrayName, Class<T> targetClass) {

        JSONArray jArray = null;
        if (json != null) {
            jArray = json.optJSONArray(arrayName);
        }
        return parseJsonToList(jArray, targetClass);
    }

    @NonNull
    public static <T> List<T> parseJsonToList(JSONArray json, Class<T> targetClass) {

        if (json == null)
            return new ArrayList<>(0);

        List<T> values = new ArrayList<>(json.length());

        T model;
        for (int i = 0; i < json.length(); ++i) {
            try {
                model = targetClass.getConstructor(JSONObject.class)
                        .newInstance(json.optJSONObject(i));
                values.add(model);
            } catch (Exception e) {
                try {
                    values.add((T) json.opt(i));
                } catch (Exception e1) {
                }
            }
        }
        return values;
    }

}
