/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.utils;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.drawable.ClipDrawable;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.LayerDrawable;
import android.graphics.drawable.ShapeDrawable;
import android.os.Build;
import android.os.Bundle;
import android.util.Base64;
import android.util.Log;
import android.view.Gravity;
import android.view.View;
import android.view.animation.Animation;
import android.view.animation.ScaleAnimation;

import androidx.security.crypto.EncryptedSharedPreferences;
import androidx.security.crypto.MasterKey;
import androidx.work.OneTimeWorkRequest;
import androidx.work.WorkManager;

import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.params.HttpProtocolParams;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.TimeUnit;
import java.util.zip.GZIPInputStream;

import ly.warp.sdk.services.EventRefreshDeviceTokenService;
import ly.warp.sdk.utils.constants.WarpConstants;

public class WarpUtils {

    private static final String PREFERENCES_PREFIX = "ly.warp.sdk.";

    private static final String PREFERENCES_NAME = PREFERENCES_PREFIX
            + "warply_preferences";
    private static final String PREFERENCES_KEY_REGISTRATION_GCM = PREFERENCES_PREFIX
            + "registration_gcm";
    private static final String PREFERENCES_KEY_WEB_ID = PREFERENCES_PREFIX
            + "web_id";
    private static final String PREFERENCES_KEY_LAST_TRY_GCM = PREFERENCES_PREFIX
            + "last_try_gcm";
    private static final String PREFERENCES_LAST_GCM_SENDER_ID = PREFERENCES_PREFIX
            + "last_gcm_sender_id";
    private static final String PREFERENCES_KEY_API_KEY = PREFERENCES_PREFIX
            + "api_key";
    private static final String PREFERENCES_KEY_DEVICE_INFO = PREFERENCES_PREFIX
            + "device_info";
    private static final String PREFERENCES_KEY_APPLICATION_DATA = PREFERENCES_PREFIX
            + "application_data";
    private static final String PREFERENCES_KEY_GCM_ENABLED = PREFERENCES_PREFIX
            + "gcm_enabled";
    private static final String PREFERENCES_KEY_LAST_APPLICATION_UUID = PREFERENCES_PREFIX
            + "last_appplixation_uuid";
    private static final String PREFERENCES_KEY_TIMESTAMP = PREFERENCES_PREFIX
            + "timestamp";
    private static final String PREFERENCES_KEY_DEVICE_TOKEN = PREFERENCES_PREFIX
            + "device_token";
    private static final String PREFERENCES_KEY_IS_WARPED = PREFERENCES_PREFIX
            + "warped";
    private static final String PREFERENCES_KEY_HAS_APPLICATION_INFO = PREFERENCES_PREFIX
            + "has_application_info";
    private static final String PREFERENCES_KEY_SDK_VERSION = PREFERENCES_PREFIX
            + "sdk_version";
    private static final String PREFERENCES_KEY_APP_VERSION = PREFERENCES_PREFIX
            + "app_version";
    private static final String PREFERENCES_KEY_DEVICE_ID = PREFERENCES_PREFIX
            + "device_id";
    private static final String PREFERENCES_KEY_HAS_DEVICE_ID = PREFERENCES_PREFIX
            + "has_device_id";
    private static final String PREFERENCES_KEY_HAS_DEVICE_INFO = PREFERENCES_PREFIX
            + "has_device_info";
    private static final String PREFERENCES_KEY_OS_VERSION = PREFERENCES_PREFIX
            + "os_version";
    private static final String PREFERENCES_KEY_DEVICE_BRAND = PREFERENCES_PREFIX
            + "device_brand";
    private static final String PREFERENCES_KEY_CARRIER_NAME = PREFERENCES_PREFIX
            + "carrier_name";
    private static final String PREFERENCES_KEY_APP_DATA_ENABLED = PREFERENCES_PREFIX
            + "app_data_enabled";
    private static final String PREFERENCES_KEY_DEVICE_INFO_ENABLED = PREFERENCES_PREFIX
            + "device_info_enabled";
    private static final String PREFERENCES_KEY_APP_DATA = PREFERENCES_PREFIX
            + "application_data";
    private static final String PREFERENCES_KEY_DEVICE_INFO_DATA = PREFERENCES_PREFIX
            + "device_info_data";
    private static final String PREFERENCES_KEY_LIFECYCLE_ANALYTICS_ENABLED = PREFERENCES_PREFIX
            + "lifecycle_analytics_enabled";
    private static final String PREFERENCES_KEY_GIFTS_POPUP = PREFERENCES_PREFIX
            + "gifts_popup";
    private static final String PREFERENCES_KEY_MORE_POPUP = PREFERENCES_PREFIX
            + "more_popup";
    private static final String PREFERENCES_KEY_USER_TAG = PREFERENCES_PREFIX
            + "user_badge";
    private static final String PREFERENCES_KEY_WEBVIEW_PARAMS = PREFERENCES_PREFIX
            + "webview_params";
    private static final String PREFERENCES_STEPS_COUNTER = PREFERENCES_PREFIX
            + "steps_counter";
    private static final String PREFERENCES_KEY_USER_NON_TELCO = PREFERENCES_PREFIX
            + "user_non_telco";
    private static final String PREFERENCES_STEPS_METERS_COUNTER = PREFERENCES_PREFIX
            + "steps_meters_counter";
    private static final String PREFERENCES_TRACKERS_ENABLED = PREFERENCES_PREFIX
            + "trackers_enabled";

    private static SharedPreferences _prefs;

    /* Public methods */

    public static String getUserTag(Context context) {
        return getPreferences(context).getString(PREFERENCES_KEY_USER_TAG, "");
    }

    public static void setUserTag(Context context, String userTag) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_KEY_USER_TAG, userTag);
        editor.apply();
    }

    public static boolean getUserNonTelco(Context context) {
        return getPreferences(context).getBoolean(PREFERENCES_KEY_USER_NON_TELCO, false);
    }

    public static void setUserNonTelco(Context context, boolean userNonTelco) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putBoolean(PREFERENCES_KEY_USER_NON_TELCO, userNonTelco);
        editor.apply();
    }

    /**
     * Method used to return the token of this device for the GCM service
     *
     * @return The token
     */
    public static String getRegistrationGCM(Context context) {
        return GCMRegistrar.getRegistrationId(context);
    }

    public static void setRegistrationGCM(Context context, String registrationGCM) {

        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_KEY_REGISTRATION_GCM, registrationGCM);
        editor.apply();
    }

    public static String getLastApplicationUUID(Context context) {
        return getPreferences(context).getString(
                PREFERENCES_KEY_LAST_APPLICATION_UUID, "");
    }

    public static void setLastApplicationUUID(Context context, String uuid) {
        getPreferences(context).edit()
                .putString(PREFERENCES_KEY_LAST_APPLICATION_UUID, uuid)
                .apply();
    }

    public static String getWebId(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getString(PREFERENCES_KEY_WEB_ID, "");
    }

    public static void setWebId(Context context, String webId) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_KEY_WEB_ID, webId);
        editor.apply();
    }

    public static boolean getGiftsPopup(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getBoolean(PREFERENCES_KEY_GIFTS_POPUP, true);
    }

    public static void setGiftsPopup(Context context, boolean visible) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putBoolean(PREFERENCES_KEY_GIFTS_POPUP, visible);
        editor.apply();
    }

    public static boolean getMorePopup(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getBoolean(PREFERENCES_KEY_MORE_POPUP, true);
    }

    public static void setMorePopup(Context context, boolean visible) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putBoolean(PREFERENCES_KEY_MORE_POPUP, visible);
        editor.apply();
    }

    public static boolean isRegisteredWarply(Context context) {
        return !("".equalsIgnoreCase(getWebId(context)));
    }

    public static void invalidateRegistrationWarply(Context context) {
        setWebId(context, "");
    }

    private static SharedPreferences getPreferences(Context context) {
        if (_prefs == null) {
            try {
                MasterKey masterKey = new
                        MasterKey.Builder(context, MasterKey.DEFAULT_MASTER_KEY_ALIAS).
                        setKeyScheme(MasterKey.KeyScheme.AES256_GCM).build();

                SharedPreferences encryptedSharedPreferences = EncryptedSharedPreferences.create(
                        context,
                        PREFERENCES_NAME,
                        masterKey,
                        EncryptedSharedPreferences.PrefKeyEncryptionScheme.AES256_SIV,
                        EncryptedSharedPreferences.PrefValueEncryptionScheme.AES256_GCM
                );

                _prefs = encryptedSharedPreferences;
            } catch (Exception e) {
                WarpUtils.log("WarpUtils Get Encrypted Shared Preferences Error", e);
                _prefs = context.getSharedPreferences(PREFERENCES_NAME,
                        Context.MODE_PRIVATE);
            }
        }

        return _prefs;
    }

    private static String getStringMetadata(Context context, String key) {

        try {

            Bundle metadata = context.getPackageManager().getApplicationInfo(
                    context.getPackageName(), PackageManager.GET_META_DATA).metaData;

            if (metadata == null) {

                log("Metadata Bundle is null");
                return null;

            } else {

                return metadata.getString(key);
            }

        } catch (NameNotFoundException e) {

            log("Metadata fail", e);
            return null;
        }
    }

    private static boolean getBooleanMetadata(Context context, String key, boolean defValue) {

        try {

            Bundle metadata = context.getPackageManager().getApplicationInfo(
                    context.getPackageName(), PackageManager.GET_META_DATA).metaData;

            if (metadata == null) {

                log("Metadata Bundle is null");
                return defValue;

            } else {

                return metadata.getBoolean(key, defValue);
            }

        } catch (NameNotFoundException e) {

            log("Metadata fail", e);
            return defValue;
        }
    }

    private static int getIntMetadata(Context context, String key, int defValue) {

        try {

            Bundle metadata = context.getPackageManager().getApplicationInfo(
                    context.getPackageName(), PackageManager.GET_META_DATA).metaData;
            if (metadata == null) {
                log("Metadata Bundle is null");
                return defValue;

            } else {

                return metadata.getInt(key, defValue);
            }

        } catch (NameNotFoundException e) {

            log("Metadata fail", e);
            return defValue;
        }
    }

    public static boolean isWaitingGCM(Context context) {

        return (System.currentTimeMillis() - getLastGCMTry(context)) < WarpConstants.GCM_WAIT_LIMIT;
    }

    static long getLastGCMTry(Context context) {

        SharedPreferences prefs = getPreferences(context);
        return prefs.getLong(PREFERENCES_KEY_LAST_TRY_GCM,
                System.currentTimeMillis() - 2 * WarpConstants.GCM_WAIT_LIMIT);
    }

    public static void setLastGCMTry(Context context, long lastTry) {

        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putLong(PREFERENCES_KEY_LAST_TRY_GCM, lastTry);
        editor.apply();
    }

    // read metadata params

    public static void setLastGCMSenderId(Context context, String senderId) {

        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_LAST_GCM_SENDER_ID, senderId);
        editor.apply();
    }

    public static String getLastGCMSenderId(Context context) {

        SharedPreferences prefs = getPreferences(context);
        return prefs.getString(PREFERENCES_LAST_GCM_SENDER_ID, "");
    }

    public static void setApiKey(Context context, String apiKey) {

        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_KEY_API_KEY, apiKey);
        editor.apply();
    }

    public static String getApiKey(Context context) {

        SharedPreferences prefs = getPreferences(context);
        return prefs.getString(PREFERENCES_KEY_API_KEY, "");
    }

    public static String getLastDeviceInfo(Context context) {

        SharedPreferences prefs = getPreferences(context);
        return prefs.getString(PREFERENCES_KEY_DEVICE_INFO, "");
    }

    public static String getLastApplicationData(Context context) {

        SharedPreferences prefs = getPreferences(context);
        return prefs.getString(PREFERENCES_KEY_APPLICATION_DATA, "");
    }

    @SuppressLint("NewApi")
    // For using apply
    public static void setLastDeviceInfo(Context context, String deviceInfo) {

        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_KEY_DEVICE_INFO, deviceInfo);
        editor.apply();
    }

    @SuppressLint("NewApi")
    // For using apply
    public static void setLastApplicationData(Context context, String applicationData) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_KEY_APPLICATION_DATA, applicationData);
        editor.apply();
    }

    public static boolean isGCMEnabled(Context context) {
        return getPreferences(context).getBoolean(PREFERENCES_KEY_GCM_ENABLED,
                true);
    }

    @SuppressLint("NewApi")
    // For using apply
    public static void setGCMEnabled(Context context, boolean value) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putBoolean(PREFERENCES_KEY_APPLICATION_DATA, value);
        editor.apply();
    }

    /**
     * START
     * New checks for sending logs to server
     */
    public static long getCurrentTimeMillis(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getLong(PREFERENCES_KEY_TIMESTAMP, 0);
    }

    public static void setCurrentTimeMillis(Context context, long time) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putLong(PREFERENCES_KEY_TIMESTAMP, time);
        editor.apply();
    }

    public static boolean getIsWarped(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getBoolean(PREFERENCES_KEY_IS_WARPED, false);
    }

    public static void setIsWarped(Context context, boolean warped) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putBoolean(PREFERENCES_KEY_IS_WARPED, warped);
        editor.apply();
    }

    public static String getDeviceToken(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getString(PREFERENCES_KEY_DEVICE_TOKEN, "");
    }

    public static void setDeviceToken(Context context, String token) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_KEY_DEVICE_TOKEN, token);
        editor.apply();

        OneTimeWorkRequest mywork = new OneTimeWorkRequest.Builder(EventRefreshDeviceTokenService.class).setInitialDelay(1, TimeUnit.MINUTES).build();
        WorkManager.getInstance(context).enqueue(mywork);
    }

    public static boolean getHasApplicationInfo(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getBoolean(PREFERENCES_KEY_HAS_APPLICATION_INFO, false);
    }

    public static void setHasApplicationInfo(Context context, boolean hasInfo) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putBoolean(PREFERENCES_KEY_HAS_APPLICATION_INFO, hasInfo);
        editor.apply();
    }

    public static String getSDKVersion(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getString(PREFERENCES_KEY_SDK_VERSION, "");
    }

    public static void setSDKVersion(Context context, String version) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_KEY_SDK_VERSION, version);
        editor.apply();
    }

    public static String getAppVersion(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getString(PREFERENCES_KEY_APP_VERSION, "");
    }

    public static void setAppVersion(Context context, String version) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_KEY_APP_VERSION, version);
        editor.apply();
    }

    public static String getDeviceId(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getString(PREFERENCES_KEY_DEVICE_ID, "");
    }

    public static void setDeviceId(Context context, String id) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_KEY_DEVICE_ID, id);
        editor.apply();
    }

    public static boolean getIsDeviceIdSaved(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getBoolean(PREFERENCES_KEY_HAS_DEVICE_ID, false);
    }

    public static void setIsDeviceIdSaved(Context context, boolean hasID) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putBoolean(PREFERENCES_KEY_HAS_DEVICE_ID, hasID);
        editor.apply();
    }

    public static boolean getIsDeviceInfoSaved(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getBoolean(PREFERENCES_KEY_HAS_DEVICE_INFO, false);
    }

    public static void setIsDeviceInfoSaved(Context context, boolean hasInfo) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putBoolean(PREFERENCES_KEY_HAS_DEVICE_INFO, hasInfo);
        editor.apply();
    }

    public static String getOSVersion(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getString(PREFERENCES_KEY_OS_VERSION, "");
    }

    public static void setOSVersion(Context context, String version) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_KEY_OS_VERSION, version);
        editor.apply();
    }

    public static String getDeviceBrand(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getString(PREFERENCES_KEY_DEVICE_BRAND, "");
    }

    public static void setDeviceBrand(Context context, String brand) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_KEY_DEVICE_BRAND, brand);
        editor.apply();
    }

    public static String getCarrierName(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getString(PREFERENCES_KEY_CARRIER_NAME, "");
    }

    public static void setCarrierName(Context context, String carrier) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_KEY_CARRIER_NAME, carrier);
        editor.apply();
    }

    public static boolean getIsAPPDATAENABLED(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getBoolean(PREFERENCES_KEY_APP_DATA_ENABLED, true);
    }

    public static void setIsAPPDATAENABLED(Context context, boolean appDataEnabled) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putBoolean(PREFERENCES_KEY_APP_DATA_ENABLED, appDataEnabled);
        editor.apply();
    }

    public static boolean getIsDEVICEINFOENABLED(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getBoolean(PREFERENCES_KEY_DEVICE_INFO_ENABLED, true);
    }

    public static void setIsDEVICEINFOENABLED(Context context, boolean deviceInfoEnabled) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putBoolean(PREFERENCES_KEY_DEVICE_INFO_ENABLED, deviceInfoEnabled);
        editor.apply();
    }

    public static boolean getIsLIFECYCLEANALYTICSENABLED(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getBoolean(PREFERENCES_KEY_LIFECYCLE_ANALYTICS_ENABLED, true);
    }

    public static void setIsLIFECYCLEANALYTICSENABLED(Context context, boolean deviceInfoEnabled) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putBoolean(PREFERENCES_KEY_LIFECYCLE_ANALYTICS_ENABLED, deviceInfoEnabled);
        editor.apply();
    }

    public static JSONObject getAppDataObject(Context context) {
        SharedPreferences prefs = getPreferences(context);
        try {
            return new JSONObject(prefs.getString(PREFERENCES_KEY_APP_DATA, ""));
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return new JSONObject();
    }

    public static void setAppDataObject(Context context, JSONObject appData) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_KEY_APP_DATA, appData.toString());
        editor.apply();
    }

    public static void setWebviewParams(Context context, JSONObject params) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_KEY_WEBVIEW_PARAMS, params.toString());
        editor.apply();
    }

    public static String getWebviewParams(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getString(PREFERENCES_KEY_WEBVIEW_PARAMS, "");
    }

    public static void setStepsCounter(Context context, int counter) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putInt(PREFERENCES_STEPS_COUNTER, counter);
        editor.apply();
    }

    public static int getStepsCounter(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getInt(PREFERENCES_STEPS_COUNTER, 0);
    }

    public static void setStepsMetersCounter(Context context, double counter) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putLong(PREFERENCES_STEPS_METERS_COUNTER, Double.doubleToRawLongBits(counter));
        editor.apply();
    }

    public static double getStepsMetersCounter(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return Double.longBitsToDouble(prefs.getLong(PREFERENCES_STEPS_METERS_COUNTER, 0));
    }

    public static void setTrackersEnabled(Context context, boolean isEnabled) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putBoolean(PREFERENCES_TRACKERS_ENABLED, isEnabled);
        editor.apply();
    }

    public static boolean getTrackersEnabled(Context context) {
        SharedPreferences prefs = getPreferences(context);
        return prefs.getBoolean(PREFERENCES_TRACKERS_ENABLED, false);
    }

    public static JSONObject getDeviceInfoObject(Context context) {
        SharedPreferences prefs = getPreferences(context);
        try {
            return new JSONObject(prefs.getString(PREFERENCES_KEY_DEVICE_INFO_DATA, ""));
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return new JSONObject();
    }

    public static void setDeviceInfoObject(Context context, JSONObject deviceInfoData) {
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(PREFERENCES_KEY_DEVICE_INFO_DATA, deviceInfoData.toString());
        editor.apply();
    }

    /**
     * END
     * New checks for sending logs to server
     */


    public static void log(String message, Exception e) {
        if (WarpConstants.DEBUG) {
            Log.d("WARP_DEBUG", message, e);
        }
    }

    public static void warn(String message, Exception e) {
        if (WarpConstants.DEBUG)
            Log.w("WARP_DEBUG", message, e);
    }

    public static void log(String message) {
        if (WarpConstants.DEBUG) {
            Log.d("WARP_DEBUG", message);
        }
    }

    public static void verbose(String message) {
        if (WarpConstants.DEBUG) {
            Log.v("WARP_DEBUG", message);
        }
    }

    public static void verbose(String message, Exception e) {
        if (WarpConstants.DEBUG) {
            Log.v("WARP_DEBUG", message, e);
        }
    }

    public static String produceSignature(String apiKeyAndDate) {

        final String SIGNATURE_ALGORITHM = "SHA-256";

        MessageDigest md = null;

        try {
            md = MessageDigest.getInstance(SIGNATURE_ALGORITHM);
        } catch (NoSuchAlgorithmException e1) {
            if (WarpConstants.DEBUG) {
                e1.printStackTrace();
            }
        }

        md.reset();
        return convertToHex(md.digest(apiKeyAndDate.getBytes()));
    }

    public static String convertToHex(byte buf[]) {
        StringBuilder strbuf = new StringBuilder(buf.length * 2);
        int i;

        for (i = 0; i < buf.length; i++) {
            if (((int) buf[i] & 0xff) < 0x10) {
                strbuf.append("0");
            }

            strbuf.append(Long.toString((int) buf[i] & 0xff, 16));
        }

        return strbuf.toString();
    }

    public static String convertInputStreamToString(InputStream is) {

        InputStreamReader reader = new InputStreamReader(is,
                Charset.forName("utf-8"));
        char[] buffer = new char[1000];
        int read;
        StringBuilder sb = new StringBuilder();

        try {
            while ((read = reader.read(buffer, 0, 1000)) >= 0)
                sb.append(buffer, 0, read);
            is.close();
            return sb.toString();

        } catch (IOException e) {

            if (WarpConstants.DEBUG) {
                log("Exception thrown while converting InputStream to String", e);
            }
            return null;

        }
    }

    public static String convertGZippedInputStreamToString(InputStream is)
            throws IOException {
        String out = convertInputStreamToString(new GZIPInputStream(is));
        is.close();
        return out;
    }

    public static void enableGzip(HttpClient client, HttpRequestBase connection) {
        HttpProtocolParams.setUserAgent(client.getParams(), WarpConstants.GZIP);
        connection.addHeader(WarpConstants.HEADER_ACCEPT_ENCODING,
                WarpConstants.GZIP);
    }

    public static int uintBit() {
        return Byte.SIZE * Integer.SIZE / Byte.SIZE;
    }

    public static int uintRotate(int val, int howmuch) {
        return ((val << howmuch) | (val >> (uintBit() - howmuch)));
    }

    public static Animation getPulseAnimation() {

        Animation anim = new ScaleAnimation(1.0f, 0.7f, 1.0f, 0.7f,
                Animation.RELATIVE_TO_SELF, 0.5f, Animation.RELATIVE_TO_SELF, 0.5f);
        anim.setRepeatCount(Animation.INFINITE);
        anim.setRepeatMode(Animation.REVERSE);
        anim.setDuration(1000);
        return anim;
    }

    public static void animateVisibility(final View view, final int visibility) {

        if (view != null) {

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB_MR1) {
                switch (visibility) {

                    case View.VISIBLE:
                        if (view.getVisibility() != View.VISIBLE) {
                            view.setAlpha(0f);
                            view.animate()
                                    .alpha(1f)
                                    .setDuration(view.getResources().getInteger(
                                            android.R.integer.config_longAnimTime))
                                    .setListener(new AnimatorListenerAdapter() {
                                        @Override
                                        public void onAnimationStart(Animator animation) {
                                            view.setVisibility(visibility);
                                        }
                                    });
                        }
                        break;

                    case View.INVISIBLE:
                    case View.GONE:

                        if (view.getVisibility() == View.VISIBLE) {
                            view.setAlpha(1f);
                            view.animate()
                                    .alpha(0f)
                                    .setDuration(view.getResources().getInteger(
                                            android.R.integer.config_longAnimTime))
                                    .setListener(new AnimatorListenerAdapter() {
                                        @Override
                                        public void onAnimationEnd(Animator animation) {
                                            view.setVisibility(visibility);
                                        }
                                    });
                        }
                        break;

                    default:
                        break;
                }
            } else {
                view.setVisibility(visibility);
            }
        }
    }

    public static Drawable getHorizontalProgressDrawable(int progressColor) {

        ShapeDrawable bgrDrw = new ShapeDrawable();
        bgrDrw.getPaint().setStyle(Paint.Style.FILL);
        bgrDrw.getPaint().setColor(Color.TRANSPARENT);

        ShapeDrawable progressDrw = new ShapeDrawable();
        progressDrw.getPaint().setStyle(Paint.Style.FILL);
        progressDrw.getPaint().setColor(progressColor);
        ClipDrawable clipDrawable = new ClipDrawable(progressDrw, Gravity.START,
                ClipDrawable.HORIZONTAL);

        return new LayerDrawable(new Drawable[]{bgrDrw, clipDrawable});
    }

    public static Bitmap getWarplyLogo() {

        String base64Logo = "iVBORw0KGgoAAAANSUhEUgAAAfQAAAH0CAYAAADL1t+KAAAAGXRFWHRTb2Z0d2FyZQBBZG9iZSBJbWFnZVJlYWR5ccllPAAAMu1JREFUeNrsncFuJLeWpmndu+hd6w06/QRXfgJnGWigFwNYtZxVpZazkrQc4DaUCde+Uk9QUU/gNNBALwZwhXezc/oJHPMG2buZ3ZBVlG9apWREZkacQ/J8H6ArX1dZiiQZ\n" +
                "/PmT5xw6BwAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAEP5KvzPD2/fzv23jzSHCtt///vfv6EZYAp4t1VY+3f63uIH/9f2P977bwuGgM64uwj/6wdf6781" +
                "tIcKV37S5QWAqXhHE4iy818ro2I+R8zV6MK4u9j7FyvaRI0HL+qXNAOM7M7vwoKRlhDl3hukndHP/p7u1xt3/2v+33Z/CLofhEHh17SLCjP/dUczwIhiHhaID7SEKK2fRxuj7nwZ5zGQZ+PFfBP+4eLZHwSXvqN9VLjFpcOIhK12xpMsN0bFfMbiUY3d/rj7k6DHraJH2kiFS8d5J4zjzueOs0xpVnGX0yJs" +
                "tSuOu7DV/qKgR8K2e0c7qbDwk/GMZoAR3DnI0Tmjx5XenYeF45whoELrxfxP4+4LQY8unQA5VrtQpjsnEE4ek4FwXszZVVQed8//xUsO3cXADly6DvO4ZQpwrJgTCKfgkvx8uTH62YnT0CNstW8HCXrkhjbDpQMTLBxk5+wGwgXTsWAIqBDM9otHPAcFPRabaWk7FWYUm4Ej3TkTrDyPhgPh2GrX42Y/EG6o" +
                "Qw/c03Z6LwxpbMAEm69L8mK+NOrOw+cmTkOHxov5QaOdFHQ/YMMefUMbqhDEnGIzMMSdEwin4JKMivnMf7ul+1XY9ZnsiwE/hIh3PW5JY4MeMScQTp5NPJK0yHtHnIYW94e22gcLejwjQtT1XDqTNaQgEC4zl1SxO7925JxrEXLOm76/dDHwh60dJWG1CMVm2E6Fl9x5mFwXtIQoJgPhYs452Td6i8hBRzyD" +
                "BJ2SsFm4MADGhS5bq4Fw7vNOITtBSotI784HLSKHOnQXB3JH26pAsRl47s4JhJPH6lZ7mHsI0FVaRHoxH7yIvDjyh3OWrgfbXfAk5sRWyNMYDoRjJ6iQReRRgh5Lwm5pYxUoNgP7Eyzbn3JYDoRjJ0iPdSrnfAyH7hzFZlQncorNmHfnc0cgnDQro5evzBw7QZqLyKN3xI8WdErCqkKxGWD7U5YQCLc2PNYw" +
                "EDrc9OWcj+XQcem6PFBsxqw7Z/tTYWK1+KFjzvk13a/Cxov5STf4nSTolITVF3WawJyYEwgnzzrOddbEnHvO9TgrXuPijF9MxLseC9LYzMH2p/zEanWOCwvHGUNAhdXQnPNRBZ2SsLh0EHPnYfG2oCVEuTcaCBeOdIjT0SHknJ8Vr3Fx5gNQElaPUGyGMy477hzkaGOKrkWod6HH2fEaZwk6JWGZ6GFyd04g" +
                "nII7t/ihyTlXJWy1nx2vca5DpySsLrM44UOdYk4gnDyWA+EYazoE/RwlNfJipAfiLF2PB4rNVAuBcPITq9W5jHvO9Tgp53wyQackrCoUm6nTnc8dgXDSWA2EC2ONeBwdNseWd5Vw6J9eBvpG1aXPaIbq3DnIEQLhNtY+NPecqzL4nnNxQackLAIAo7lzgpPkuTH6ucNYwwzosBprq30Kh45L1+WaYjNViDnB" +
                "SQoTa6yrYc2dXzHW1GjPzTmfXNApCasOL2f5EAgnSxByy5evgA6T7AhdTPAziXjXY86d6UW787kjEE58YjUaCBe22ud0vwpnlXcVFXRKwuLSAcdUCJsY+2NNzDnW0aPzYr6c6odfTPRzKQmrRyg2s6QZinPnBMLJctatVhUsHDnW0WHS4MtJBJ2SsOrcUmymKDHHMcnzaDQQbu441tGiGTPnXNKhUxJWFwSi" +
                "LB5wTKJ0cX6yCDnnOojsCF1M/PM5S9fjjmIzRbhzrquUx2TOuXfnYRHDnKA05sbOORcX9FgStqUv1SDIij6CP2M1EC4IObt2OoScc5EqhBcCvwOXrgfFZvJ25wtH6pAko5faLAi22g2MuckFnZKw6rAqz1PML3Hn4qyM5pyzcNTjcaqccy2H7gyvinOAYjP5LrQIhJNj68XcXEW4mHPOwlFpzE2Zc64m6DE9" +
                "pKF/9cSDNLas3DmBcPKQcw7SiBvZC8HfFc7SKTajwwwByW6SBTkao4Fwc0fOuRZr78631Qp6dOkUm9GDYjN5uPMwwc5pCTGsV4QDeTqnFAx+Ifz7KAmrB2dp+mJOH8hjNRBu6SglrMW9RM65uqDHF4s0Nj0WFJtRhUA4WVqjgXDhHb+l+1XYSOWc5+DQXXzBOvpdDfJRddw5gXAKTsnwO87CUR71450Lpd+L" +
                "S9djTrEZFdhql2XtzcPW2of27vzaEaOhxUoy5zwbQackLOJizJ0vmGTFnZI50xBzztmB0yGUd1U/3rlQ/N24dD2uKDYjJuYEwslzbzEQzhGjoTrmcngINUGnJKz+y08aG5NsjU4p7gBac+dzR4yGmjnVyDnPzaEHKAmrx4wJYHJ3TiAcc4oU7ALp0LnP6dhZoCrolIRV55Y0NibZmpxSnFOsufOwaCTnXGkB" +
                "qZVznqND//QSOorNaBG2grmNbRp3vnAEwpl1SoJiPuMdVqPxYt7m9EDqgk5JWHUWcWsYxhNzAuHksRoIx+UrOmRZUvgik+egJKz+pADjQSCcLCEQbmPtQ8ec82u6X2cBmdNWe1aCTklYdSg2M547JxBOHnOBcNxzrruA9GLe5PhguTh0SsLqQ0GKcWCSlcVkIJz7vAs0o/tZQGYp6E8vJ2NFjRnFZs5256H9" +
                "5rSEGJ0X86VBd84ukKJGaZd3LUbQKQmr7y4pNnOymLMFilOSgt00HbZezLNeQF5k+Ey4dD0uWfmfDIFwsmxitUlr7pyccz2yv70vO0GnJKw6FJs53p2zBSpLlilDAmJO3Qg91rnlnJfi0AOUhNV16Uwax8FWuyyPRgPhuOdcbwFZxM5xloJOSVh1KDYz3J0vHIFwkmyNBsKFMUbOuZLBzDHnvCSH7hwlYXGd" +
                "+Ys5gXDyWN1qJxBOh5BzXkzRomwFnZKw6oRiMziCNATCydJYDIRzn+MzZnS/OMFQFnX8e5H581ESFpeeqzsnEE5+crXozq8cMS1aZJ1zXpygx5Kw94wrNUKxGUSLxU4Wk6vhy1dAnpBzXtztfbk79KdiMx3jS40His184c4XjkA40ck1loa25s7vGGdqFJlpdVHIc+LS9aDYzJ/FnEA43n8JMSd9VI+Qc75F" +
                "0Kdz6SHKsGWcqbr0Gc3wuS0cgXCik6vRQDjuOdehcwVXK70o6FkpCasvZNbdOYFwspi8VjnmnC/ofhWKyTkvWtDjKn3DeFNjwZ3pbLULc280EI6ccx02JZR3rcWhf3rBGXO4dCV3HhzTnCEgRhsDYq2586Uj51yD4nLOixd0SsKqY7LYDIFwLN6FxHzmONrSYlXyVnupDv3pRafYjB4WhY1AOFlCINzW4Odm" +
                "q12HtsSc8yoEPZ6pURJWD1PFZgiEE6dzNgPhFo4jHU2TWAUXhT43JWGVHauhYjNstQtPrtYC4WLOOeNMh1WpOefVCDolYdUxUfSCQDhx2lhzwhrknOvQeTFf1vSBSnXolITV567mYjMEwqlwY+0Dk3POeEPQ/wEuXd9Z1AqBcLKsYhYL7xBI0JSec16doFMSVp3rGovNEAgnThByi5evLP23K7pfnGqPbC8q" +
                "+AyUhNV3srgmOAeLgXAz/+2Wrlfhpoac8yoFnZKw6sxj8Fgt7jx8ljndKsbGaCBcyDnnSEeekHNe7Xi7qORzcJaOSx9DzAmEk8Vktop359csGtXGW9WBl1UIOiVh1QnFZpaVLExwTXI8WguEiznnVIRTGm/enVc93i4q+iyUhNXltuRiMwTCidN5MV8a/NwsGnXY1pZzXrWgUxJWndKLzbDVLovFnHMWjbqG" +
                "r3ouKvs8lITVpchiMwTCibOJwazWYKtdSRdqzDmvXtApCcukdYKYEwgni9VAuODMyTmXp3OGUptrc+iUhNVnXlixGc40ZVkZDISbOe451+K+1pxzE4L+1ImMY1WKcLwEwomz9WK+Nvo+sGiUZ1NzzrkZQackrDpXhRSbYaudhfbU7jzknF/T9eKYPNq5qPizURJWl6zvTCcQTpzGWiAc95zrzv+155ybEnRK" +
                "wqozc5luZxMIh1uSWtTG9wBkCTnnFo92qnboznGWrk2uxWYIhBN2SwYvXyE+Q48bqx+8akGP0bRrxrca2TlhAuHEaQ0HwoHC4tG78y2CXnEHO4rNaLLIrNgME60sVnPO53S9OOYNXPWCTknYLMii2AyBcOKs/ftnyi3FQDhyznW4sZRzbtWhu7hq6xjvaqgXmyEQTpwwsVrMNOGecx02Vsq7mhf06NJJY9NF" +
                "W0wJhJPl3mAgXFi0knOus3i8oRmc+8rSh/Uu7XdHGokmN7E0r3S/h0C4X2l+MUIg3CtjYn4Zxxjzi8K84t15QzPY2XL/o+Ppcl2XrpTGxla7sDs3+JnvEHOdxSNiblTQY7GZlm5X49IJp4wRCCfOymAgXNgBIhAOk4aga0w4dLsqt1JpbATCidM5m2lDjDGludxieVcE/UuX3tD1qi5dys0QCCeLxUC4hWMH" +
                "SGXx6MV8STPg0HHp+ixioNqU7pyKcLK08ZZDS2LODpAebLUj6H+49M5RElabd4X/fGCC5Z5zHdbknCPoL7l0SsLqMVmxGQLh5N+luEi25M7D+FrQ9eJQUwRBf9GlUxJWn9FLwhIIJ07n36UlYxeEMF/eFUE/DCVhdZlFNz0mBMIJT7DWPrB352EBM6PrxQk55xuaAUFPuXS2b3QZrdgMgXDibGLWiCUxnzly" +
                "zjWgvCuCPkjUG1y6KmMWm2GrXXaCtVgRjq12Hcg5R9AHw8pPl4dzi80QCCfOo8FAOMaYDlsv5mQlIeiDXXrrKAmrLupniDmBcLKYC4Qj5xzDhaCXBWfpuizOSGMjEI4JVmLByRiTJ+Scb2kGBP0Ul97QEmW5dALhxGkMBsLNGWMqdBgtBB2XXi6h2Mz1kf8N26ByWA2EY4zpcE/OOYJ+jksPK0KCLwqZPAmE" +
                "k1/wGrx8Zem/XdH14mzIOUfQx3LprAr1CMVmerc3CYQTZ+vF3NRiN+ac39L14pBzjqCP5tIpCavPw4BiMwQpyWI155wxpmCq2GpH0MeEkrC6JIvNEAgn/z4YDIQLsRxzul6clpxzBH0Kl06AnL5Lnx34M7ba5TD3LsSccyrC6XBPEyDoU4h6g0vXF/UX3PkC5yQ7wVoLhHMc52ixIuccQZ8SAjN0+VOxGQLh" +
                "xGnjwtaSO+c4R4dgnthqR9AndemtoyRsTi4d5yTszg1+ZrbalcwTgXDn8xVNkCYGYP1KS+i+7P5rSz+IEgLhTAm6d+fBmbMDJE/jxZzdUBy6iEsPQtLQEuounYlWDouBcDPHPedaY41AOARdFCLedQmT7ZxmEOPGYCBcWDBynCMP5V0RdHGX3iHqYIQQCGeq5GbMOb+m6+XHmhfzhmZA0DUIEZisJKF6d25M" +
                "zMmc0IHyrgi6qkunJCzUziruRlkinJvP6HpxHr0772gGBF1T1JeOYjNQJ2FcW7t8hZxzHbZezJc0A4KehYuhCaBC7o0GwoHCWKMJEPRcXHrjPudEA9TCxmAgXHDmc7penLV35y3NgKCzwgQYH3N5wDEQjpxznbHGDieCnp1LDytMVplQA48GA+G451wHyrsi6Lh0gInoYqCnJXc+d+Sca7DxYr6hGRD0XF06" +
                "JWGheMdkTMy551wHyrsi6EXAeRAU65ji0ZElQiDcjK6XnyfJOUfQS3DpHaIOOKYi3HnIOScQTp6Qc8495wh6MVASFopzTAYD4cg514Hyrgh6US6dkrBQlGPyY9ZaRbiFI+dcZeHo3Tk1OxD04kR96SgJC2VgMeccdy5PmA/ZakfQy12N0gSQOY3BQDjuOdeBnHMEvWiX3jhKwkK+WAyEm/tvC7penA3lXRH0" +
                "GiDXEnJlZfDyFXLOdRaOBMIh6FW49LAqZWUKuWExEG7pyDlXWTiy1Y6g49IBpsNaRbgg5OScy9OSc46g1+bSKQkLObGOY9ISbLWzcETQYTSIeIccMHddJTnnenMe5V31+QtNMD4ff/559913333FxALK/A/vzv+3ITEP6Wn/6b/+ia4XpfNi/ppmwKHXDCVhQZM2plJaIpybk3MuD1vtCHrdUBIWlLGYc35H" +
                "t4vTkHOOoFsR9aWjJCzIszIYCEd5V3m45xxBtze50gQgSFhAWsw5v6LrxaG8a2Z8RRNMzw9v3/7KhANCvPbufGNIzGf+W3i/ODuXJeScv6IZcOgWYVsKRCZZS2IeeY+Yi0N5VwTdLpSEBSGsVYS7dqSGavBIzjmCzmQLMB0hEM7MJBtzzqkIJ8/Wi/mSZkDQrbv0MNk2tARMQBhb1mpok3OOMQEEXddFOYrN" +
                "wASTrKWrUb07DwGm5JzLs/bufEszIOjwD5dOsRkYk02M0bAEW+3ydI4UXAQdvlzl4tJhJMwV9vDuPDhzUkDluSfnHEGHL126uRuwYDIejQXCzRz3nGuw8WK+oRkQdHhZ1INL72gJOIMulha2RCjvSiCcLJR3RdBhALh0OAeLOefXdLv8PEXOeTlQ+lWRH96+/egojAHH03h3bkbQY855KO86o+tFobwrDh1w" +
                "6TAhFrdAHxBzFdhqR9BhKJSEhVMWgeScg8Q4I+ccQYfjofISDGUbAyotwT3n8nTOXuVBBB1Gcenh5WloCRiAxZzzOd0ubzLIOUfQ4XQoCQt9NJYqwsVAOHLOFcaZF/OWZkDQ4TyXTklYOITFQDjuOWecAYJeLJSEhUPcGwuEmztyzlXGGVvtZfMXmiAPPv788//97rvv/p//x3+jNWCP1ou5GdcUt9r/E3cu" +
                "P868mOPOcegwFpSEhZdck7HPGwLhZnS7OGTbIOgwARSbgSfWfpFnJheYy1f05hzKuyLoMI1LbxzFZuBzPIW1xR33nMuz9WK+pBkQdMClw3TcGAuEWzhyzjXg3LwiuJwlU7i4xTQhEM7MpRgxEO53RyCcNGsC4XDoIOTQaAL63gjccy6PxSMdBB10oCSsWVax762487n/tqDb5ReN5Jwj6CA8uTuKzVgiCLm1" +
                "SzEIhJMn5JxvaAYEHeRdOiVh7WCtItzSkXMuTRhfHOch6KAEJWGNuCYv5mZcEznnapBzjqCDoksneAXXVCNstcsTcs655xxBB2VRpyRs3TwaC4QLF6/M6XZx2GpH0CETyBetk86L+dKQmF/izlUIOedbmgFBhzxcejhfbWkJXFPhhHNzcs6FF42OYzsEHbKDl7IuNn6hZmaRFnPO7+h2+UUjOecIOuTn0sPk" +
                "T/5oHYQJ1toxyju6XX7R6MW8pRkQdMgTztLrwFog3NJ/u6LbxReNBMIZ4i80QVl8/Pnn3XfffTdjciyarRfz/25IzMN4DYFw/0TXi/I/cec4dCjDpXMmVnb/WSKIOYFwsrTknCPoUACx2AwlYcukMRYIR845i0ZA0KEHSsKWh6lAuJhzTiCcPCtyzhF0KM+lswovbKK1dPmK+5xzPqPbRem8mC9pBgQdyhP1" +
                "xlESthS2sYSvFXcegjbJOZeHqHYEHQoGl85EmyOUd5WnIaodQYeyXTolYfNn7fvJzJmmd+fBmZNWKQtHcICgVwIlYfOeaM30D/ecq0F5V0DQK3HpwaFTEjZP7o0FwoWodnLOZQk557z/gKDXJBw0QX4TbQxctOLOQ875Nd0uCuVdAUGv0KV3/ltDS7DIUhJzcs51ePTuvKMZAEGvU0A4R8sDU4Fw7nOK2oxu" +
                "F2VLzjkg6PW6dErC5kFwTJYC4UJEO4FwOgt4AAS9ZmeIS9efaA0GwoHwe07OOSDoNlw6K3c92lgbwIo7D1vtc7pdlM6RqgoIuhlRbxwlYbUwE3EcA+HYapfnnpxzQNCNvfQ0gTirmG1gBe45l2dDzjkg6PZcOiVhZQlCvjb2mck5l4XjNEDQLTtGmkCMG2OBcIGGbpd9n8k5BwTdrksPDp3tuenZxLa2BnUP" +
                "5Ag552uaARB027BFNy1mt0FjYBZ1D2SgvCsg6Lh0SsJOzKOxQLjnUPdgesJW+5ZmAAQdnlw6k+74dF7Ml5YbILp0doEmHGPOXrAlIOiQcOlsjU4D26CfRb1x1D2YbIyRcw4IOjxnzaQ7KlYD4Q6BS59gjFHeFRB0OOTSSWMbB+6g/tKlU/eAMQYIOgiKeoNLH4WVwZzzQe1CE4wG5V0BQYdeWPWfx9aLOUFK" +
                "L7v04NCpe3A+bYxLAEDQIenSw6Tb0hKnOyeagPZh0Q0IOuQCW6On0RAI1+vSO0fdg7PeTcq7AoIOx7p0Jt3jIN/6OJfO+e/xdF7MlzQDIOiAS59YpAiEG+zSqXtwGmy1A4IOJ7n0zlGBaihtzBCA4VAS9sj2IuccEHQ416Uz6Q5w5zTBSS6ddhsGNSIAQYezXTpbowOck28nLsY4TdQbR92DIVDeFRB0GEew" +
                "mHRxThOCS0/Txip7AAg6jOLSEa0DYkQg3NkunZKw6QUjgXCAoMOoot7g0r90TgTCjQYLxgPtQs45IOgwBTgF2mMqlx4cOtvKf2br24UsE0DQYRKXHibdlpb47JxiWh+MB2fpLBgBQQdJIaMJPh094JzGd+mhXRta4hMh55zMCUDQYXKXbn3SJRBuWpduvW07Fs6AoAMufXpCIBxnvdO5dOoecM85IOgg6NKD" +
                "g7C65cy55vRYLgm7IeccEHTQcOnWJl0C4eRcusUAOXLOAUEHFZdubWu08595Sc+LiXrj7NU9WLHVDgg6aGGpJCzOSR5LLr0l5xwQdNB26RYC5DYxuh9kXbqlkrDk4AOCDuqi3lTu0rniUxcLC8YVOeeAoEMu1Lwd/UggnKpLDw695qjvMLbYagcEHbJx6WHSbSv8aFsC4bKg5h0S7jkHBB2yo8atUbba83Dp" +
                "wcU2FX60Ju5AACDokJ1Lr2nSbQiEy25xVZOTJTYDEHTApTPZmnTptdU9oLwrIOiQtUvvKhH1FZevZEktJWHbWDgHAEEHJt0JCYFwRB3n69JL3zmhvCsg6FCMSy99a5St9rxFPTjbruCP8BiD/AAQdChC1JeFTrprAuFYdE3I1ov5ku4DBB1Ko7SzdCtlbGtw6aWWhGX3BxB0KNKlN8GRlDTZEgjHgnFC1uSc" +
                "A4IOJVOKI2njAgTKcelBHEspCcvuDyDoULxLD5NuCa6ErVAWjFNCeVdA0IFJV4AQCMdNV2W69M7lX51wE8/8ARB0KN6lbzOedIMgsBVa/oIxV/dLxUFA0KE6chVNAuHKd+k51z1YkXMOCDrU5tJzdMIhEI6t0DrIsTphyDmn4iAg6MCkKwDlN+ty6bktGBlfgKBDtS49p63RVdw1gHpEPSwYc+nTsNVOoCUg" +
                "6FC1qC8zmHS7uFsA9ZGDS2d8AYIOTLpCEAhXr0tvnH7dA3LOAUEHMy49TLpa25EbAuFYME45vijvCgg6WEMjN5ecYBsuvVVy6dxzDgg6mHTpGpPuI4FwZtAQ1hVb7YCgAy59eroYkAc2XHpYuDWCv7Il5xwQdLDs0iVLwrIVao9wlr5jfAEg6CA36U7NJm7xgz2XLlH3gPKugKADCJSEJVDJNlNXJ+y8mC9p" +
                "ZkDQAaafdFfknJt26VOXhGWxCAg6wJ5Ln6ok7Nb/bAKVEPWpSsI25JwDgg7wpagvJ5h0yTmHJ8Z26dQ0AAQdQGjSbQiEgz2X3rhx6x5Q3hUQdICESw+T7hglYXFPMOWCMeScUz4YEHSAHsYQYgLh4CWX3o7g0smaAAQdYKBLP3fSbQmEgwTnivEjOeeAoAPITLpstUPKpQcxbk78z7fknAOCDnCcSz910l3H" +
                "crIAKU4tCctWOyDoAAKT7tQFRKAul35s3YO1/+9YLAKCDnCiSz9m0r0nEA6OEegjFowdi0VA0AFkJt02prwBDHXpx+zo3JNzDgg6wHkufeiky9kmnCLqQ0rCbsg5BwQdYBxR75t0V3F7HuAUUgtGChRBlfyVJgDlSff9C/8+CDk553COS2/+tf2PN/4f5y+Nu5xzzn94+3bhv81e+KOWsseAoEOuLr3xk9dL" +
                "ky6BcDDWgvH52GrjlnzOHFqIfHp+uhUOwZY75DDpPnchnG3CGC69fUEA2WoHBB1gIpf+fNIlEA7GZH88rcg5BwQdQGbSJRAOxnbpYTw1jrgMQNABRFx6mGzDufmS1oAJCMc6r8k5h9ohKA5yEXXcE0zp0gFw6AAAAICgAwAAAIIOAAAACDoAAACCDgAAAAg6AAAAIOgAAACAoAMAACDoAAAAgKADAAAAgg4A" +
                "AAAIOgAAAIIOAAAACDoAAAAg6AAAAICgAwAAIOgAAACAoAMAAACCDgAAAAg6AAAAgg4AAAAIOgAAACDoAAAAgKADAAAAgg4AAICgAwAAAIIOAAAACDoAAAAg6AAAAAg6AAAAIOgAAACAoAMAAACCDgAAgKADAAAAgg4AAAAIOgAAACDoAAAACDoAAAAg6PAlP7x9e0UrAAAAgl6+mL+jJQAAAEEvm1v/NffC" +
                "PqMpAADgXP5KE6i480v/7XpP2O9plVHade6/zeJX4J/91/6xxi97/9z6r92///3vW4PtFNoktFUYh3+L33f+67f4vc25XeL7cxW/LuO//tveP4dn/6+9fw793PKGAIKe74R0feCPw8u7zvwjXO9NPosSBN23+WJPKJ/TakyY/plCO34bxWlIPMJ8758f4s94EvefThWyRNt0/uc1E3325YE/avzv7A6I4CIu" +
                "IGeJcfn094Owh2f/cGybxF2nxUt/5n/W8oy+/j724eyIfnZ7/bx96msrAp8YJ2rv7cgL+PmBP57s3UPQx2f3NCEf6Oht5gP1du+fL4Mg5Dz44gT9PjUxCC/mbp8tis7lj4khjB3/7dF/bXyf7Ab+928OTCxtFMUpeEj0RffCxPd+gBDuE9r2zn/9nyiExzBLPN9Ssa+fXP3d3oLl8aUFUEV8mxC9MG6/Lviz" +
                "pcb0yqJDL/IMPb6Am54JNufdhatSnndvF8ElVsKtQLuFeIOP/h9/jc9zOdGvuooTxe/B3URnW7JDC2318Ugx33dwa4VnlujrpwVL6Of3FceyPKYWXnF8lOrOU31mzp0XK+gDBuoi4xf09iWHmPmEcpv4s9XEL+4sTu4fE05jCi6jy/w1bveWKubvT/zPg4O9EX7esFv1o0JfL/aEvegF3AvmJxifrkTz00Pq" +
                "uZvKd13qE/ToCrsTXaXWBLsfDHeMaGqLwqFJbjflUUE8//tdeHL/wsX4rx9Lm+zPFPNPC2bJSTEumn5PvB+Swl7kAu7ERfc8ut2SFqqznvn9gzPKRcUDNceVZ+oscJFpGz+kJv0JndqvPb/7JcJZ7zo6y1f+6xsvSl+Fr/j/w9frOG76nMtL/fOxhK3Z+IzvEs57Hdvhm9g238S2uY9tsj01eO3E5w3P+qMb" +
                "vrW+i/23ip/jeV9/86yv2yMe5/JpAVeRS296xvpDYR8pNVd2ljMaSk9b28SJ66WJYJZhsFnKhWcXHKdxThVjDD4eMbkPCmJ79pJvnv2+uUtHfz8R/m7Ygn+Vebrb+wPtF/rr/nk77X2W1n+2xk0Xn/BSf78fuJgdHHX/7M83z3YB3gzcBVjEsfHqiODInHlMLPI+HfkVtE2tdgSIoE+78tzFCegu4dKzEMgD" +
                "wXDZPu+AF2f0c6ojxTwI9Orc1Xic/D85+7iAeXDpLf7L6NRzFfXvDzx/6K+bIe9UFM9cxLx72lE5V1jjefIm7mA8DPjdV3t9XbqoN/EzXyZc+k3uH6LvCNClg6Wrp4ZKcalt35yCzYackWfzvPE5Uk7mw8i/b6iYd9E1vRp7ay38vPBz3eft2m6AqOdYi//uVDEXHl9DxDwIedhKb8YU1LAQje3xtevfjn8S" +
                "9aKD5WL7lRpI/Nz0pExGDbspdgV9QAqb+vlQTzDcKcKvvQAZNc//CDFfxwm+nXhMhZ//Tfx9faKe+yQYdhGyKlwUgx0XPc8c+nk55QQdhf2V69+mrULU43hOteci54ffOx47xdwh6JW49OsMXsRjCmMsMnhxLnue43Hk3/W+p30+pVD5yfdeagUefk/4fe7zNuQuIerHBHNpcJ+Ta9k71jhEWJyLHmfEAMBX" +
                "PWJX/GVKcRykzM9t5ouWlMnYWE1Vq07Qe1LY+sRJeyB+8bwZFHtILUDGTlV759KxBbs4wTdKY6vpmeyHxEZo0WQY8ftjz/O+1liAxHbqE/WwLX1X+HSZ2o14KraTozvv2+U0m6pWo0PvG6i3igPxlAlfO+VOJFUturXFADFXDT6Lv79vsi/tndDiMiHmNwX080PJVeWii20ynnsOLqYSY6eLAY8IekWfZZN4" +
                "EWeKxRNSi4lDZ7RqwXEDUtXWI/2ep632rMX82WRf0q14JW1BbnMJ2hvQz2HcFr313rPQy7Uc7K2EyUDQ81l57npWnuIuvWebaNMzELV2FVLufMwo0ruehcN9bmlhcft9Xcgr8VMhzxnG0+vC+vm6tOpqL7j09sQ5QMNkXCfmir55H0EvmL7gOGnXu3CHt4k+9EToLxRenNA+86lXwnGh0xfgkuVLGgPlcr9Dn" +
                "fVfQ1ZE3me4kBBfblSJ6E7j0nMrfvumZK0ynqlUr6ANS2KRd76Hft3/mcyiYQyM4LjVJtSM65juXLg6Re4GL3Lfe20Je2TbXs88oEqlxOC/cpbc94ySL9NkB9TBMV4ar3aH3uciF4EAML/uhHYEPey9W6uz/jeDzSkaRpj7Xfe4r7jgZ5uyAfynFnSN62c6VuSxYbnsWhB0yXrGg96WwCbreZEWjnv+//1LN" +
                "hJ534dJRpKMIWM95WFfQVnHOzqAEh17KFZepfr4uPOI966tVB9TDIFXNgEPvewlvlQfiS9HHqZWy1FndrdCL832hIvl8Mgx9mOt28baAJnwspJ/D4ijVnqVftZp657TLwV5LmAwEPX9S29hXAltJR60qe6JOJ690NyCKdD3yS1rLC5qjQyhBzLeFLDqGLD7euILJ/GrV28LePQR9okHal8rwRmkg7hJBQAeD" +
                "4wRcQF/E+Shn2nEhdZlYhJU2zlILRy1KiPgtLW84NTavKqjx/qHHpYt/vjhXpApyrR2Yceh9k8Zkg7QnGK7pWSnvThDcc583PGtqx2LMbfB5hSvu3BYiv9BmkxiE1DOXvu3ed2mLRjlYblVD0P/0EnY9L+GdwkDscyZNwgVMVS+8L1WtG/F3/e3Av+8K24ItTUBzYlvoZPzTCeO6pAVLssiVpEuXvBwKQa/H" +
                "pb8RHohDxFG0cpzCi3No56IteIzl9uxd5u31U4X9fOXKp++qYMldiLueBeHWgT1B70lhm6IaUkocPwx43s7JBselXpwpLjw4NPH9VvAYy01Acxf0bcH9fGhnYV7BXNkXdyQZHHfOLieCXjmSKWypYLhm4M+QDI4Te3F6FiOlr7hbB1ULet+zl5yPPnCuFLm0Jf6O2QjzKIJeKalI5NEKt5waDPfCSrlxAsFx" +
                "fS+OG78S2lXBrhLGdbo1LkZmlfRNI2iAcOcI+tGDVGor6c0JrtsduQAYMzgumwsPKijfyJneMNrCn/+/En92WUkfpURz0hoeAzJucOc49N5BevbZdE9w2SlBHJMGx8VFQerF4cKD8SZ6qIfUe1xDYNxTlcFWwAAd+7Mb6rYj6PsOcJNYWZ97Nr04UZxTz9tOtQBx/YVkeHFgCnY8fxGkFvST3C8hfDkUgl65" +
                "Sz931XmbmABOjRSfJDiOHE9Q5DeaoAgD1Cq49L667S09g6A/H6SHnOfs1LOhnmC4k8+iJwyO60tV48U5nn+mCaAy+srBju3SH07cMQCjDr1vYJwqklNGZTYH/v05wXG3vDijc0UTQGUGqHHp7JMxM25SpuicXU4EvXJSKWxH3208QTDcMQuC2xNenPCsl0ovzq7nhYb6+ReaoBoDNOZ9GKm5jLrtCPrBVWdf" +
                "CtviyB+Z+vsfRnjesEJuEwuQy1JenMrLNc4c0E62XHqYe86+DyOaqOsTTQ0Yd+hju96kQI70vKMExw24jlDzxSndoSNUNrB4tJIyJmPch5EyRS0ZNwj6ENd7MIVtaHnDvspwY7ndEYPj+grJSLw4h3Ybir2tiuMCUwu3mssXHyJ1teoY5WBvcecIeg4ufczKcH0ccvuDguPittYigxdnV6HzISDuuAXQrODH" +
                "//aEsV26Aeq7WvXhjLGwcOlUNYLhEPRBgzQ4xe5UkewJhpsi9evcBUhKzLeCqWq/JVb6pU703zKNmFkAXfXMKbWSulr1nFsrcecI+mick8Im6nZHCI7L5cVJTXrXhY6jOdNI/QugeLRy0E1WboD6golPybi5SiyQprgcCkGvnFQKW19KhkQw3HNOCo7rS1WTvI6wx8W8KW0ARWdy6eAYSl24fX/iQtWCAZqf" +
                "EEvSV36aVDUEfdRV5yKxUp9JD8QzguMeMnHnfZPfVYHb7t8zhRzNbMQbAyVZJP7sFwPzZdczXw5ekFN+GkGfilPOpiWD4Ya6/xfP/XsWH1PuJqT46cRVe27ufIxLfaxyW9LD9uxyWXHoffPlMeVgU2LeVl6zAkGfeNW5STiJ6yNWlhJRmccuQPoKyXQKzb7pmRRK2cJmu/0Mt1tQP/e9R1srudIjXq16q2iK" +
                "EHRc+qCV5QeBF6pzA4PjBlRg+qA0KaQ+wyjVp4R44NU5iyL6OS7qcy3IpMFZ5WBje84Spqjh1UDQzxGYIC6HVtjP7/7VCIYbKsTPt4BTz9oqp9mkFhO3ubs3/3x3jupwZy+IComZeJf4M3MXhwy4WrVvoYY7R9BVV523cRKfu3QwXCf0QjWuJzhuQNDJB+VJIfUZwrO/z1jML3Hno/E+54fzfb3sWbg9Go3G" +
                "PmlBHhdw88R/u+aVQNDHYEgKm2Yw3EvP+xJPwXGp891ctrVSW5XXZxSrkBAhzs7HYR53O3IU86uehdvOqgANuLTl+gR3zq1qCPpoAzSVwvbkdg853p1CicK+c/+HXN35s9V46gV+n9uWbIx2JrJ9XN7llsYWF/A/9r2DxgUotav5cKBNFwXMSwh6JaREMnWOJu52Y7TpodSOVNCJy8VVxMnwPvFXPk2quZyn" +
                "R9F5z2syCR9zEfU43j72vENm3flAl/7SpS2pXcNt5aVzEXSFARoG5ylOWyvK9TEhhAcXHzm5ijgppF7kqzjZq4p6FJuPvCWTcZmDqO+Jed9z3LA93Ouqb/tcewZzKIJu2KW/hOZ9valz/5JenJuez6Eq6ntifplwaxTCGM42R1E/QswbbgH7g9Sx2dVTOdiegOIdqWoI+lSOMbjFYwT6g+KzHpsyk2UFprgg" +
                "uu/5a2GS/VV6so/bhikxD7x2lV6dqbCAu4z9fCfcz5/G1wAx3w4Yq5bmy6FXq77BnSPoWqwG/r0cVpbHvAwfMp4YQjv2teVMarIPbs1/haCovoj2G87+ju7rIIqvehZBIVDuo0RQZExN+9X11xXYObbaD7n0Qzxd2rJI7XjQhAj6lAzdylYfiD3BcftkX4HJP9/NwDYNk/2vJ9zudIwr/931R7M3bBWeNW5v" +
                "ev5a6N/fg+BOcdwSxk8YR25YTYEwH7yixvhBl556D1LZAhsrZXMR9HwHaG6O93Gkv1OSqD+dq398IZr2VEd+579+d8PyzJv4rHB6X29cf/yEi4IbhP39GEcuob5BGDdu2Hk5Yj6M1K7mZenzUkn8lSY4ONBSW7vbjF7wMDG+c+mgrWKcZBBKP+GGfxwi1MHFBaf1LrZDuMZyUKBidPhhQv/WHZdbvvY/n3PU" +
                "cfq68f2wdf1xCn/UgvB/v3vW17u+xVocJ9+74y/UQcyH9WPn27kZ+M4+0XFchaBLDtBNYqJ/zOhZd/FZD71Mm9LO/aKo/+bS+f8vTvhxEg/fDk0WM3d6HfYbttlH7+ut769XcWdkiGOexcX2XezrVJbBlTu9ql8YP685Mz/KBB0j6CuabHzYck8P0EOr9k0hz1rsi+Mn0hBs8407LuvgC/f+wtcpYh4E4xvE" +
                "fDpRd58D5U5p38tEX58i5p8KHvlneoWYH92H7RFtTOofgi46QNsDYpKd400Ex7UlB52Ez+W/vo6LEq02X/ln+IZt18n7ehfjEl6fsYg7l01cuHFJyInvysC/R912BD2bAZprIMdjQc967GS/9N++jg5OaiIIv+vr+LtBrq8/iarwIi4s3oMjf03U9dkmaIhLJxhuIjhD71+x7wecbTN2as+ftaupolVc0Yez" +
                "9RCQtnCfi1WMXWjmKYDw8ciJfXvkvx9LhA59hhr6OqSrrWNfhzKis0z6WoJD46aExUbI/pmnxi2Lpun4iiZIEyOonyLe73PejgupPe4fgSnVB3DFwiMhcPFbd/qZ6dPZ3y+U9BytX0JffDwg1l+d+DOvnvX1qYug0N8/EWE9af//nliAveY9w6Frsp/C1hTwrAtnJOgkrvTX8etJ4Pe/XMLpkDZTVl//KU4k" +
                "CvzlAHEPfbwjBiILOsQch57DivPHOCncFPCsofpVS6401OTQoZi+D7soh6rD3RNwiEPPxfmWwspx+xcA6HB74N8XVeAKQa+YkrZm2dICACV3PnOHj0A2pKpND2lrAAAwBqmLbkhVQ9ABAKAAd/5UfvklWoISEXQAACiD1GVWH2geBB0AAMrgzYF/33EHAoIOAAAF8MPbtwt3uO4D7hxBBwCAwt15AHeOoAMA" +
                "QAHuPFTsmx8Sc+q2I+gAAFAGt4k/Y7sdQQcAgALceSpVbctdCQg6AACUQSpVjUIyCDoAABTCoWC4HalqCDoAABRAT6oaYo6gAwBA4e48wHY7gg4AAAW481Sq2oZUNQQdAADK4BZ3jqADAEDZ7nzmDqeqdaSqIegAAFAGi8SfrWgeBB0AAMrg0Hb7zn9taB4EHQAAMiemql0e+OMQDLejlRB0AAAo150H2G5H" +
                "0AEAoAB3Pvffrg78cUuqGoIOAABlQCEZBB0AAAp35zOXTlUjGA5BBwCAAljgzhF0AAAon1QwXEPzIOgAAJA5PalqDalqCDoAAJTvztluR9ABAKAAdz536VS1La2EoAMAQP6kUtU+0DwIOgAA5O/OZy6dqtbQSgg6AADkzy3uvDz+ShMAwMiEs9VXNEPRBNH+KdG/AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" +
                "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAMAr/X4ABAKy5y+BxEmIbAAAAAElFTkSuQmCC";
        byte[] decodedLogo = Base64.decode(base64Logo, Base64.URL_SAFE);
        return BitmapFactory.decodeByteArray(decodedLogo, 0, decodedLogo.length);

    }

    public static Bitmap getWarplyCloseBtn() {

        String base64Logo = "iVBORw0KGgoAAAANSUhEUgAAAEAAAABACAYAAACqaXHeAAAACXBIWXMAAAsTAAALEwEAmpwYA" +
                "AAAGXRFWHRTb2Z0d2FyZQBBZG9iZSBJbWFnZVJlYWR5ccllPAAAA7VJREFUeNrsm8+L00AUx6f1uMLWi3py01UP" +
                "nlpkWUVEC4Kwnrr+AwYvXixdEFERJSAiHoQt/gPJP7D2tEdJYQ/iqbnLbvCkXmzBvbiHOk/flGyatJkm8yOxDx6" +
                "7h5I338+8eZm8TEqj0YjIsJt37hr0D3ideiXmZwPqfep+b3fHlzGukigAKLhJvYG+zHmJIXUXvSsKSKYAqGiYWR" +
                "O9lvFYPeo2OIUx0AoAzrZF/R6RYw7EyyIrUgHAGQfhbaLGOghiIB0AFd/ElFwmag1qhUkhdKUAwFnflpjuPMtii" +
                "zcbuABQ8XWc9RrR0zzMhn7mAFC8q0HKJ1kSjaQQygUTT3CMLo45PYCcieeGUC6geC4I5RnV3s6p+CAEG7VwZ8C2" +
                "xtWex2qoJfldADc5H0ixbDNqszQBANPFz3nqx90ejfBGKWoJWDziWw/uk9cvn5KTS0vSlEAsiAmxOeuBNbUG4FM" +
                "d14PN2TOnyfWr66Tz9pUUCBADYkFMiM1pbdQYmwEW7xXfvHtPvuz75MKqIRwCEw+xICbEnsOsyBqAa/9nFgNrP3l" +
                "Bfh0eChWfMsYpVguCGWDOezUYCAxIVCYIAGxGLQEzzRVFQRCUXeaxJYCF4UC32RK8tKrQUmMZ0MzqqlllgoS60gw" +
                "ugUaWV04LQUZRZZqFAEgDQZL4sebSjY3NzNZ/WkESxY/rAGSAITJC0kxQIB7MAAB10VFmQVAkHqwOACoyIsVBUCg" +
                "erHJi5eKlhogiGGW/j47Ix94eubJ2+a/ga+trZOP2LXK+uqJCPFivTCQby4QD/yupGufIKvV9+r8C8TNbYkIt2IY" +
                "pKeySSAfA1jybeZYJsvoJSgGEC17r8XPykLqsfkIcgIEK8WzNi36UnmEDANBXJV5WP2GK9QGAr1K8Ygg+6wfAMlh" +
                "WJV7RlnjY292psCLo6iBecia4wbuAq4N4yRCOAejqIl4ihO4YAB4383TqBwqG4LEjdsGNkK2TeMEQ7KidoK2beIE" +
                "QJgHgmxJHN/ECIDjBN8Sp3w0+e9SS1swIQ4DYc5gV+zCEhaHDc7Vv33+QvU+fpT3PMwgQE2JzWid8vnhxQGKiR/T" +
                "vByYpnplRx2gj+wF4lsYpkHgn7jD1tIbIVhabIw3MQy18HaHAUhjmfN2b006QT22J4YHjRk4hJDo0PbMnmFMIiU+" +
                "MJ2qK5gxC9sflQxB0Lowej3guACEIOt4iHV7xkTvBpFaUj6bmfjGCAQ3eZ4eMrYPb27k7WosPJxefzi4+nv6/P5" +
                "//I8AAy2M8m2IrbvIAAAAASUVORK5CYII=";

        byte[] decodedLogo = Base64.decode(base64Logo, Base64.DEFAULT);
        return BitmapFactory.decodeByteArray(decodedLogo, 0, decodedLogo.length);

    }

}
