package ly.warp.sdk.utils;

import android.app.Activity;
import android.bluetooth.BluetoothAdapter;
import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.graphics.Point;
import android.os.Build;
import android.provider.Settings;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.util.Base64;
import android.view.Display;
import android.view.WindowManager;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GoogleApiAvailability;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.List;

import ly.warp.sdk.io.callbacks.SimpleCallbackReceiver;
import ly.warp.sdk.utils.constants.WarpConstants;

public class WarplyDeviceInfoCollector {
    // ===========================================================
    // Constants
    // ===========================================================
    // ===========================================================
    // Fields
    // ===========================================================
    private final Context mContext;

    // ===========================================================
    // Constructors
    // ===========================================================

    public WarplyDeviceInfoCollector(Context context) {
        this.mContext = context;
        if (context instanceof Activity)
            new PermissionsUtil((Activity) context, null, PermissionsUtil.PERMISSION_PHONE_STATE)
                    .requestPermissions();
    }

    // ===========================================================
    // Methods
    // ===========================================================

    public void collectToJson(final SimpleCallbackReceiver<JSONObject> callback) {
        JSONObject jObj = new JSONObject();
        new Thread(() -> {
            if (!Thread.currentThread().isInterrupted()) {
                if (GoogleApiAvailability.getInstance().isGooglePlayServicesAvailable(mContext) == ConnectionResult.SUCCESS) {
                    //Google Play Services are available
                    try {
                        if (WarpUtils.getTrackersEnabled(mContext)) {
                            jObj.putOpt("platform", "android");
                            jObj.putOpt("manufacturer", getDeviceManufacturer());
                            jObj.putOpt("android_phone_model", getDeviceModel());
                            jObj.putOpt("os_version", getOsVersion());
                            jObj.putOpt("app_version", getAppVersion());
                        }

                        jObj.putOpt("registration_gcm", getRegistrationGCM());
                        jObj.putOpt("vendor", "google");
                        jObj.putOpt("unique_device_id", getUniqueDeviceId());
                    } catch (JSONException e) {
                        WarpUtils.log("************* WARPLY Warning ********************");
                        WarpUtils.log("Problem when creating Device Info JSON", e);
                        WarpUtils.log("*************************************************");
                    }
                } else {
                    //Google Play Services are not available, or not updated
                    try {
                        if (WarpUtils.getTrackersEnabled(mContext)) {
                            jObj.putOpt("platform", "android");
                            jObj.putOpt("manufacturer", getDeviceManufacturer());
                            jObj.putOpt("android_phone_model", getDeviceModel());
                            jObj.putOpt("os_version", getOsVersion());
                            jObj.putOpt("app_version", getAppVersion());
                        }

                        jObj.putOpt("vendor", "huawei");
                        jObj.putOpt("registration_gcm", getRegistrationGCM());
                        jObj.putOpt("unique_device_id", getUniqueDeviceId());
                    } catch (JSONException e) {
                        WarpUtils.log("************* WARPLY Warning ********************");
                        WarpUtils.log("Problem when creating Device Info JSON", e);
                        WarpUtils.log("*************************************************");
                    }
                }
            }
            callback.onSuccess(jObj);
            Thread.currentThread().interrupt();
        }).start();
    }

    public void getRegistrationParams(final SimpleCallbackReceiver<JSONObject> callback) {
        JSONObject jObj = new JSONObject();
        new Thread(() -> {
            if (!Thread.currentThread().isInterrupted()) {
                // For GoogleApiAvailability we need to add a meta tag in Manifest
                if (GoogleApiAvailability.getInstance().isGooglePlayServicesAvailable(mContext) == ConnectionResult.SUCCESS) {
                    //Google Play Services are available
                    try {
                        if (WarpUtils.getTrackersEnabled(mContext)) {
                            jObj.putOpt("platform", "android");
                            jObj.putOpt("manufacturer", getDeviceManufacturer());
                            jObj.putOpt("android_phone_model", getDeviceModel());
                            jObj.putOpt("os_version", getOsVersion());
                            jObj.putOpt("app_version", getAppVersion());
                        }

                        jObj.putOpt("unique_device_id", getUniqueDeviceId());
                        jObj.putOpt("vendor", "google");
                    } catch (JSONException e) {
                        WarpUtils.log("************* WARPLY Warning ********************");
                        WarpUtils.log("Problem when creating Device Info JSON", e);
                        WarpUtils.log("*************************************************");
                    }
                } else {
                    //Google Play Services are not available, or not updated
                    try {
                        if (WarpUtils.getTrackersEnabled(mContext)) {
                            jObj.putOpt("platform", "android");
                            jObj.putOpt("manufacturer", getDeviceManufacturer());
                            jObj.putOpt("android_phone_model", getDeviceModel());
                            jObj.putOpt("os_version", getOsVersion());
                            jObj.putOpt("app_version", getAppVersion());
                        }

                        jObj.putOpt("unique_device_id", getUniqueDeviceId());
                        jObj.putOpt("vendor", "huawei");
                    } catch (JSONException e) {
                        WarpUtils.log("************* WARPLY Warning ********************");
                        WarpUtils.log("Problem when creating Registration Device Info JSON", e);
                        WarpUtils.log("*************************************************");
                    }
                }
            }
            callback.onSuccess(jObj);
            Thread.currentThread().interrupt();
        }).start();
    }

    private String hashToString(int value) {
        String result = "";
        String string = Integer.toHexString(value);

        int remain = 8 - string.length();
        char[] chars = new char[remain];
        Arrays.fill(chars, '0');
        string = new String(chars) + string;

        int count = 0;
        for (int i = string.length() - 1; i >= 0; i--) {
            count++;
            result = string.substring(i, i + 1) + result;
            if (count == 4) {
                result = "" + result;
                count = 0;
            }
        }
        return result;
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    private String isInDevelopmentMode() {
        return String.valueOf(WarplyProperty.isInDevelopmentMode(mContext));
    }

    private String getCarrierName() {
        TelephonyManager tm = (TelephonyManager)
                mContext.getSystemService(Context.TELEPHONY_SERVICE);
        return tm != null ? tm.getNetworkOperatorName() : "";
    }

    private String getRegistrationGCM() {
        if (!TextUtils.isEmpty(WarpUtils.getDeviceToken(mContext)))
            return WarpUtils.getDeviceToken(mContext);

        WarpUtils.setDeviceToken(mContext, WarpUtils.getRegistrationGCM(mContext));
        return WarpUtils.getRegistrationGCM(mContext);
    }

    private String getAppVersion() {
        try {
            PackageInfo pInfo = mContext.getPackageManager().getPackageInfo(mContext.getPackageName(), 0);
            return pInfo.versionName;
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
            return "";
        }
    }

    private String getHardwareId() {
        return (Build.SERIAL == null ||
                Build.SERIAL.equals(android.os.Build.UNKNOWN)) ? "" : Build.SERIAL;
    }

    /**
     * <p>Requires Permission:
     * {@link android.Manifest.permission#READ_PHONE_STATE READ_PHONE_STATE}
     */
    private String getImei() {
        if (!PermissionsUtil.hasPermission(mContext, PermissionsUtil.PERMISSION_PHONE_STATE))
            return "-";
        TelephonyManager tm = (TelephonyManager) mContext.getSystemService(Context.TELEPHONY_SERVICE);
        try {
            return tm.getDeviceId();
        } catch (Exception e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        return "-";
    }

    private String getAndroidDeviceId() {
        return Settings.Secure.getString(mContext.getContentResolver(), Settings.Secure.ANDROID_ID);
    }

    /*public String getWiFiMac() {
        WifiManager wm = (WifiManager) mContext.getSystemService(Context.WIFI_SERVICE);
        return wm != null ? wm.getConnectionInfo().getMacAddress() : "";
    }*/

    private String getBluetoothMac() {
        BluetoothAdapter ba = BluetoothAdapter.getDefaultAdapter();
        String mac = "";
        if (ba != null && PermissionsUtil.hasPermission(mContext, PermissionsUtil.SAFE_PERMISSION_BLUETOOTH))
            mac = ba.getAddress();
        return mac;
    }

    public String getUniqueDeviceId() {
        String hardwareComplexId = "35" + //we make this look like a valid IMEI
                Build.BOARD.length() % 10
                + Build.BRAND.length() % 10
                + (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP ?
                Build.SUPPORTED_ABIS[0].length() % 10 : Build.CPU_ABI.length() % 10)
                + Build.DEVICE.length() % 10
                + Build.DISPLAY.length() % 10
                + Build.HOST.length() % 10
                + Build.ID.length() % 10
                + Build.MANUFACTURER.length() % 10
                + Build.MODEL.length() % 10
                + Build.PRODUCT.length() % 10
                + Build.TAGS.length() % 10
                + Build.TYPE.length() % 10
                + Build.USER.length() % 10; //13 digits

        String hardwareId = getHardwareId();
//        String imei = getImei();
        String androidId = getAndroidDeviceId();

        String id = hardwareComplexId + hardwareId + /*imei +*/ androidId;
        byte[] hash = id.getBytes();
        try {
            hash = MessageDigest.getInstance("SHA-256").digest(id.getBytes("UTF-8"));
        } catch (NoSuchAlgorithmException | UnsupportedEncodingException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        return Base64.encodeToString(hash, Base64.NO_WRAP);
    }

    public String getAdvertisementId() {
        if (isPackageInstalled("com.android.vending", mContext.getPackageManager())) {
            GoogleAdInfoClient.GoogleAdInfo adInfo = GoogleAdInfoClient.getGoogleAdvertisingInfo(mContext);
            return adInfo != null ? adInfo.getId() : "";
        } else {
            try {
                com.huawei.hms.ads.identifier.AdvertisingIdClient.Info info = com.huawei.hms.ads.identifier.AdvertisingIdClient.getAdvertisingIdInfo(mContext);
                return info != null ? info.getId() : "";
            } catch (IOException e) {
                return "";
            }
        }
    }

    private String getDeviceManufacturer() {
        return Build.MANUFACTURER;
    }

    private String getDeviceModel() {
        if (!TextUtils.isEmpty(android.os.Build.PRODUCT)) {
            return android.os.Build.MODEL +
                    " (" + android.os.Build.PRODUCT + ")";
        } else {
            return android.os.Build.MODEL;
        }
    }

    private String getOsVersion() {
        return android.os.Build.VERSION.RELEASE;
    }

    private String getAndroidSDKVersion() {
        return String.valueOf(android.os.Build.VERSION.SDK_INT);
    }

    private String getDeviceName() {
        return android.os.Build.DEVICE;
    }

    @SuppressWarnings("deprecation")
    public int getDisplayHeight() {
        WindowManager wm = (WindowManager) mContext
                .getSystemService(Context.WINDOW_SERVICE);
        Display display = wm.getDefaultDisplay();
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB_MR2) {
            Point size = new Point();
            wm.getDefaultDisplay().getSize(size);
            return size.y;
        } else {
            return display.getHeight();
        }
    }

    @SuppressWarnings("deprecation")
    public int getDisplayWidth() {
        WindowManager wm = (WindowManager) mContext
                .getSystemService(Context.WINDOW_SERVICE);
        Display display = wm.getDefaultDisplay();
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB_MR2) {
            Point size = new Point();
            wm.getDefaultDisplay().getSize(size);
            return size.x;
        } else {
            return display.getWidth();
        }
    }

    public JSONObject getInstalledPackages(boolean includeSystemPackages) {
        JSONObject json = new JSONObject();
        JSONArray jsonPacks = new JSONArray();
        List<PackageInfo> packs = mContext.getPackageManager().getInstalledPackages(0);

        for (PackageInfo pack : packs) {
            if (!includeSystemPackages) {
                if (pack.versionName == null
                        || pack.packageName.contains("com.android")
                        || pack.packageName.contains("com.google")
                        || pack.packageName.contains(android.os.Build.
                        MANUFACTURER.toLowerCase())) {
                    continue;
                }
            }

            if (mContext.getPackageManager().getLaunchIntentForPackage(pack.packageName) != null) {
                try {
                    JSONObject jsonPack = new JSONObject();
                    jsonPack.putOpt("package_name", pack.packageName);
                    jsonPack.putOpt("app_name", pack.applicationInfo.
                            loadLabel(mContext.getPackageManager()).toString());
                    jsonPack.putOpt("app_version", pack.versionName);
                    jsonPacks.put(jsonPack);

                } catch (JSONException e) {
                    WarpUtils.warn("[WARPLY Packages Installed] " +
                            "Error on JSON creation ", e);
                }
            }
        }

        try {
            json.putOpt("packages_count", jsonPacks.length());
            json.putOpt("packages", jsonPacks);
        } catch (JSONException e) {
            WarpUtils.warn("[WARPLY Packages Installed] " +
                    "Error on JSON creation ", e);
        }

        return json;
    }

    public static boolean isPackageInstalled(String packageName, PackageManager packageManager) {
        try {
            return packageManager.getApplicationInfo(packageName, 0).enabled;
        } catch (PackageManager.NameNotFoundException e) {
            return false;
        }
    }

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

}
