package ly.warp.sdk.utils;

import android.app.Activity;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Handler;
import android.os.Looper;

import androidx.annotation.NonNull;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import ly.warp.sdk.dexter.listener.DexterError;
import ly.warp.sdk.dexter.listener.PermissionDeniedResponse;
import ly.warp.sdk.io.callbacks.WarplyReadyCallback;
import ly.warp.sdk.utils.constants.WarpConstants;

public class WarplyInitializer extends Thread /*implements ActivityCompat.OnRequestPermissionsResultCallback*/ {

    // ===========================================================
    // Constants
    // ===========================================================

    private final long INIT_TIME_OUT = 1000 * 20;
    private final long INIT_ATTEMPTS = 3;
    private final int REQUEST_CODE_ASK_MULTIPLE_PERMISSIONS = 3001;
    private boolean mPermissionsResultRequested = false;

    // ===========================================================
    // Fields
    // ===========================================================

    private Context mContext;
    private Activity mPermissionsActivity;
    private WarplyInitCallback mInitCallback;
    private WarplyReadyCallback mReadyCallback;

    private static volatile Handler mMainThreadHandler;
    private static volatile AtomicBoolean stopThreadLock;

    // ===========================================================
    // Constructors
    // ===========================================================

    public WarplyInitializer(Context context, WarplyReadyCallback readyCallback, WarplyInitCallback initCallback) {

//        this.mPermissionsResultRequested = false;
        this.mContext = context;
        this.mReadyCallback = readyCallback;
        this.mInitCallback = initCallback;

        if (mMainThreadHandler == null) {
            mMainThreadHandler = new Handler(Looper.getMainLooper());
        }

        if (stopThreadLock == null) {
            stopThreadLock = new AtomicBoolean(true);
        }
        stopThreadLock.set(true);
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================
    @Override
    public synchronized void start() {
        init();
    }

    @Override
    public void run() {
        stopThreadLock.set(false);
        long startWarplyInit = System.currentTimeMillis();
        int initAttempts = 0;

        // waiting for warply server side registration
        while (!WarpUtils.isRegisteredWarply(mContext.getApplicationContext())) {
            if (stopThreadLock.get())
                return;

            if ((System.currentTimeMillis() - startWarplyInit) > INIT_TIME_OUT) {
                mInitCallback.onInit(mContext);
                startWarplyInit = System.currentTimeMillis();
                initAttempts++;
            }

            if (initAttempts >= INIT_ATTEMPTS) {
                notifyListener(true);
                return;
            }
        }

        notifyListener(false);
    }

//    @Override
//    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
//        switch (requestCode) {
//            case REQUEST_CODE_ASK_MULTIPLE_PERMISSIONS:
//                if (permissions.length > 0) {
//                    Map<String, Integer> results = new HashMap<>();
//                    for (String permission : PermissionsUtil.DANGEROUS_PERMISSIONS) {
//                        results.put(permission, PackageManager.PERMISSION_GRANTED);
//                    }
//
//                    for (int i = 0; i < permissions.length; i++) {
//                        results.put(permissions[i], grantResults[i]);
//                    }
//
//                    boolean stillNeedRequest = false;
//                    for (Map.Entry<String, Integer> entry : results.entrySet()) {
//                        if (entry.getValue() != PackageManager.PERMISSION_GRANTED) {
//                            stillNeedRequest = true;
//                            break;
//                        }
//                    }
//
//                    if (mReadyCallback != null && mPermissionsActivity != null) {
//                        if (stillNeedRequest) {
//                            mReadyCallback.onWarplyPermissionsDenied();
//                        }
//                    }
//                    mPermissionsResultRequested = true;
//                }
//                break;
//
//            default:
//                break;
//        }
//    }

    // ===========================================================
    // Methods
    // ===========================================================

    private void notifyListener(final boolean timeOut) {
        if (mReadyCallback != null) {
            mMainThreadHandler.post(new Runnable() {
                @Override
                public void run() {
                    if (timeOut) {
                        mReadyCallback.onWarplyInitTimeOut();
                    } else {
                        mReadyCallback.onWarplyReady();
                    }
                }
            });
        }
    }

    // ===========================================================
    // Getters & Setters
    // ===========================================================

    public synchronized void init() {

        stopThreadLock.set(true);
        mInitCallback.onInit(mContext);
        try {
            super.start(); // call run()
        } catch (IllegalThreadStateException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
    }

    public synchronized void initWithPermissions(Activity activity) {
        mContext = activity;
//        this.mPermissionsActivity = activity;
//
//        int result;
//        List<String> listPermissionsNeeded = new ArrayList<>();
//        for (String p : PermissionsUtil.DANGEROUS_PERMISSIONS) {
//            result = ContextCompat.checkSelfPermission(activity, p);
//            if (result != PackageManager.PERMISSION_GRANTED) {
//                listPermissionsNeeded.add(p);
//            }
//        }
//
//        if (!listPermissionsNeeded.isEmpty()) {
//            ActivityCompat.requestPermissions(activity, listPermissionsNeeded.toArray(new String[listPermissionsNeeded.size()]), REQUEST_CODE_ASK_MULTIPLE_PERMISSIONS);
////            init();
//        }
//        init();

        PermissionsUtil permUtil = new PermissionsUtil(
                activity,
                new PermissionsUtil.PermissionCallback() {
                    @Override
                    public void onPermissionDenied(List<PermissionDeniedResponse> denied) {
                        mPermissionsResultRequested = true;
                        if (mReadyCallback != null)
                            mReadyCallback.onWarplyPermissionsDenied();
                    }

                    @Override
                    public void onPermissionsGranted() {
                        mPermissionsResultRequested = true;
                    }

                    @Override
                    public void onError(DexterError error) {
                        onPermissionDenied(null);
                    }
                },
                null //Requests all dangerous permissions
        );
        if (permUtil.requestNeeded() && !mPermissionsResultRequested) {
            permUtil.requestPermissions();
        } else {
            mPermissionsResultRequested = true;
            init();
        }
    }

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

    public interface WarplyInitCallback {
        void onInit(Context context);
    }
}
