/*
 * Copyright 2010-2022 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.utils;

import android.app.ActivityManager;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;

import androidx.appcompat.app.AlertDialog;
import androidx.work.OneTimeWorkRequest;
import androidx.work.WorkManager;

import org.greenrobot.eventbus.EventBus;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import ly.warp.sdk.R;
import ly.warp.sdk.Warply;
import ly.warp.sdk.activities.ContextualActivity;
import ly.warp.sdk.activities.CouponsetInfoActivity;
import ly.warp.sdk.activities.GiftsForYouActivity;
import ly.warp.sdk.activities.TelcoActivity;
import ly.warp.sdk.activities.WarpViewActivity;
import ly.warp.sdk.db.WarplyDBHelper;
import ly.warp.sdk.io.callbacks.CallbackReceiver;
import ly.warp.sdk.io.models.ActiveDFYCouponModel;
import ly.warp.sdk.io.models.Campaign;
import ly.warp.sdk.io.models.CampaignList;
import ly.warp.sdk.io.models.Consumer;
import ly.warp.sdk.io.models.Coupon;
import ly.warp.sdk.io.models.CouponList;
import ly.warp.sdk.io.models.Couponset;
import ly.warp.sdk.io.models.CouponsetsList;
import ly.warp.sdk.io.models.CustomerStateModel;
import ly.warp.sdk.io.models.LoyaltyBadgeModel;
import ly.warp.sdk.io.models.LoyaltyContextualOfferModel;
import ly.warp.sdk.io.models.LoyaltyGiftsForYouPackage;
import ly.warp.sdk.io.models.LoyaltySDKFirebaseEventModel;
import ly.warp.sdk.io.models.Merchant;
import ly.warp.sdk.io.models.MerchantList;
import ly.warp.sdk.io.models.PushCampaign;
import ly.warp.sdk.io.models.UnifiedCoupon;
import ly.warp.sdk.io.models.WarplyCouponsChangedEventModel;
import ly.warp.sdk.io.request.CosmoteRetrieveSharingRequest;
import ly.warp.sdk.io.request.CosmoteSharingRequest;
import ly.warp.sdk.io.request.WarplyConsumerRequest;
import ly.warp.sdk.io.request.WarplyEditConsumerRequest;
import ly.warp.sdk.io.request.WarplyIntegrationRequest;
import ly.warp.sdk.io.request.WarplyUserCouponsRequest;
import ly.warp.sdk.services.EventService;
import ly.warp.sdk.services.FCMBaseMessagingService;
import ly.warp.sdk.services.WarplyHealthService;
import ly.warp.sdk.utils.managers.WarplyAnalyticsManager;
import ly.warp.sdk.utils.managers.WarplyEventBusManager;
import ly.warp.sdk.utils.managers.WarplyManager;

/**
 * Created by Panagiotis Triantafyllou on 18-Apr-22.
 */

public class WarplyManagerHelper {

    // ===========================================================
    // Constants
    // ===========================================================

    // ===========================================================
    // Fields
    // ===========================================================

    private static Consumer mConsumer = new Consumer();
    private static Consumer mConsumerInternal = new Consumer();
    private static HashMap<String, CampaignList> mUniqueCampaignList = new HashMap<String, CampaignList>();
    private static CouponList mCouponList = new CouponList();
    private static CouponList mCouponListExpired = new CouponList();
    private static CouponsetsList mCouponsetsList = new CouponsetsList();
    private static CustomerStateModel mCustomerStateModel = new CustomerStateModel();
    private static ArrayList<Campaign> mCampaignList = new ArrayList<Campaign>();
    private static ArrayList<Campaign> mCampaignListAll = new ArrayList<Campaign>();
    private static ArrayList<Campaign> mCarouselList = new ArrayList<Campaign>();
    private static ArrayList<ActiveDFYCouponModel> mDfyCoupons = new ArrayList();
    private static ArrayList<LoyaltyGiftsForYouPackage> mSeasonalList = new ArrayList<>();
    private static ArrayList<LoyaltyContextualOfferModel> mCCMSList = new ArrayList<>();
    private static AlertDialog mAlertDialogSharing, mAlertDialogAcceptSharing, mAlertDialogRejectSharing,
            mAlertDialogReturnSharing, mAlertDialogErrorSharing, mAlertDialogErrorUsedSharing, mAlertDialogErrorExpiredSharing;
    private static LoyaltyBadgeModel mLoyaltyBadge = new LoyaltyBadgeModel();
    private static float mDealsSum = 0.0f;
    private static MerchantList mMerchantList = new MerchantList();
    public static double mMetersWebview = 0.0d;
    public static int mStepsWebview = 0;
    public static int mSteps = 0;
    private static ArrayList<UnifiedCoupon> mMarketCoupons = new ArrayList<>();

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    // ===========================================================
    // Methods
    // ===========================================================

    /**
     * Open Warply campaign
     */
    public static String constructCampaignUrl(Campaign item) {
        WarplyManager.getSingleCampaign(item.getSessionUUID());
        String url = item.getIndexUrl();
//                + "?web_id=" + WarpUtils.getWebId(Warply.getWarplyContext())
//                + "&app_uuid=" + WarplyProperty.getAppUuid(Warply.getWarplyContext())
//                + "&api_key=" + WarpUtils.getApiKey(Warply.getWarplyContext())
//                + "&session_uuid=" + item.getSessionUUID()
//                + "&access_token=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("access_token")
//                + "&refresh_token=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("refresh_token")
//                + "&client_id=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_id")
//                + "&client_secret=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_secret");

        JSONObject params = new JSONObject();
        try {
            params.putOpt("web_id", WarpUtils.getWebId(Warply.getWarplyContext()));
            params.putOpt("app_uuid", WarplyProperty.getAppUuid(Warply.getWarplyContext()));
            params.putOpt("api_key", WarpUtils.getApiKey(Warply.getWarplyContext()));
            params.putOpt("session_uuid", item.getSessionUUID());
            params.putOpt("access_token", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("access_token"));
            params.putOpt("refresh_token", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("refresh_token"));
            params.putOpt("client_id", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_id"));
            params.putOpt("client_secret", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_secret"));
        } catch (JSONException e) {
            e.printStackTrace();
        }

        WarpUtils.setWebviewParams(Warply.getWarplyContext(), params);

        return url;
    }

    public static String constructCampaignUrl(Campaign item, boolean fromWidget) {
        item.setNew(false);
        String url = item.getIndexUrl();
//                + "?web_id=" + WarpUtils.getWebId(Warply.getWarplyContext())
//                + "&app_uuid=" + WarplyProperty.getAppUuid(Warply.getWarplyContext())
//                + "&api_key=" + WarpUtils.getApiKey(Warply.getWarplyContext())
//                + "&session_uuid=" + item.getSessionUUID()
//                + "&access_token=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("access_token")
//                + "&refresh_token=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("refresh_token")
//                + "&client_id=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_id")
//                + "&client_secret=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_secret");

        JSONObject params = new JSONObject();
        try {
            params.putOpt("web_id", WarpUtils.getWebId(Warply.getWarplyContext()));
            params.putOpt("app_uuid", WarplyProperty.getAppUuid(Warply.getWarplyContext()));
            params.putOpt("api_key", WarpUtils.getApiKey(Warply.getWarplyContext()));
            params.putOpt("session_uuid", item.getSessionUUID());
            params.putOpt("access_token", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("access_token"));
            params.putOpt("refresh_token", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("refresh_token"));
            params.putOpt("client_id", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_id"));
            params.putOpt("client_secret", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_secret"));
            params.putOpt("from_widget", fromWidget);
        } catch (JSONException e) {
            e.printStackTrace();
        }

        WarpUtils.setWebviewParams(Warply.getWarplyContext(), params);

        return url;
    }

    /**
     * Handle sharing flow
     */
    public static void handleSharing(Context context, String sharingId) {
        WarpUtils.log("SHARING_ID: " + sharingId);
        WarplyManager.cosmoteSharing(new CosmoteSharingRequest().setSharingId(sharingId), new CallbackReceiver<JSONObject>() {
            @Override
            public void onSuccess(JSONObject result) {
                int status = result.optInt("status", 2);
                if (status == 1) {
                    JSONObject innerResult = new JSONObject();
                    innerResult = result.optJSONObject("result");
                    if (innerResult != null) {
                        String modalText = innerResult.optString("modal_text", "");
                        String sharingStatus = innerResult.optString("status", "");
                        String couponsetType = innerResult.optString("couponset_type", "");

                        if (!TextUtils.isEmpty(sharingStatus) && sharingStatus.equals("pending")) {
                            initialSharingDialog(context, modalText, sharingId, !TextUtils.isEmpty(couponsetType) && couponsetType.equals("supermarket"));
                            return;
                        }
                        if (!TextUtils.isEmpty(sharingStatus) && sharingStatus.equals("reject")) {
                            returnSharingDialog(context, modalText);
                        }
                    }
                } else if (status == 4) {
                    String modalText4 = result.optString("msg", "");
                    errorExpiredDialog(context, modalText4);
                } else if (status == 5) {
                    String modalText5 = result.optString("msg", "");
                    errorUsedDialog(context, modalText5);
                } else if (status == 7) {
                    String modalText7 = result.optString("msg", "");
                    returnDialog(context, modalText7, sharingId);
                } else
                    errorSharingDialog(context);
            }

            @Override
            public void onFailure(int errorCode) {
                errorSharingDialog(context);
            }
        });
    }

    /**
     * First dialog when we handle the deeplink
     */
    private static void initialSharingDialog(Context context, String alertTitle, String sharingId, boolean isSupermarket) {
        mAlertDialogSharing = new AlertDialog.Builder(context)
                .setTitle(R.string.cos_dlg_success_title)
                .setMessage(alertTitle)
                .setNegativeButton(R.string.cos_dlg_negative_button, (dialogPositive, whichPositive) -> {
                    rejectSharingDialog(context, sharingId);
                })
                .setPositiveButton(R.string.cos_dlg_positive_button, (dialogNegative, whichNegative) -> {
                    WarplyManager.cosmoteRetrieveSharing(new CosmoteRetrieveSharingRequest()
                                    .setSharingId(sharingId)
                                    .setAccept(true),
                            new CallbackReceiver<JSONObject>() {
                                @Override
                                public void onSuccess(JSONObject result) {
                                    int status = result.optInt("status", 2);
                                    if (status == 1) {
                                        WarplyManager.getUserCouponsWithCouponsets(new CallbackReceiver<CouponList>() {
                                            @Override
                                            public void onSuccess(CouponList result) {
                                                WarplyCouponsChangedEventModel couponsChanged = new WarplyCouponsChangedEventModel();
                                                couponsChanged.setChanged(true);
                                                EventBus.getDefault().post(new WarplyEventBusManager(couponsChanged));
                                            }

                                            @Override
                                            public void onFailure(int errorCode) {

                                            }
                                        });
                                        acceptSharingDialog(context, isSupermarket);
                                    } else
                                        errorSharingDialog(context);
                                }

                                @Override
                                public void onFailure(int errorCode) {
                                    errorSharingDialog(context);
                                }
                            });
                })
                .show();
    }

    /**
     * Dialog when we success get the sharing gift
     */
    private static void acceptSharingDialog(Context context, boolean isSupermarket) {
        mAlertDialogAcceptSharing = new AlertDialog.Builder(context)
                .setTitle(R.string.cos_dlg_success_title)
                .setMessage(isSupermarket ? R.string.cos_dlg_positive_button_text2 : R.string.cos_dlg_positive_button_text)
                .setPositiveButton(R.string.cos_dlg_positive_button2, (dialogPositive, whichPositive) -> {
                    dialogPositive.dismiss();
                })
                .show();
    }

    /**
     * Dialog when we reject the sharing gift
     */
    private static void rejectSharingDialog(Context context, String sharingId) {
        mAlertDialogRejectSharing = new AlertDialog.Builder(context)
                .setTitle(R.string.cos_dlg_reject_title)
                .setMessage(R.string.cos_dlg_reject_text)
                .setPositiveButton(R.string.cos_dlg_negative_button, (dialogPositive, whichPositive) -> {
                    WarplyManager.cosmoteRetrieveSharing(new CosmoteRetrieveSharingRequest()
                                    .setSharingId(sharingId)
                                    .setAccept(false),
                            new CallbackReceiver<JSONObject>() {
                                @Override
                                public void onSuccess(JSONObject result) {
                                    int status = result.optInt("status", 2);
                                    if (status == 1)
                                        returnSharingSuccessDialog(context);
                                    else
                                        errorSharingDialog(context);
                                }

                                @Override
                                public void onFailure(int errorCode) {
                                    errorSharingDialog(context);
                                }
                            });
                })
                .setNegativeButton(R.string.cos_dlg_negative_button2, (dialogNegative, whichNegative) -> {
                    dialogNegative.dismiss();
                })
                .show();
    }

    /**
     * Dialog when the gift gets returned
     */
    private static void returnSharingDialog(Context context, String alertTitle) {
        mAlertDialogReturnSharing = new AlertDialog.Builder(context)
                .setTitle(R.string.cod_dlg_return_title)
                .setMessage(alertTitle)
                .setPositiveButton(R.string.cos_dlg_positive_button2, (dialogPositive, whichPositive) -> {
                    dialogPositive.dismiss();
                })
                .show();
    }

    /**
     * Dialog when the gift gets successfully returned
     */
    private static void returnSharingSuccessDialog(Context context) {
        mAlertDialogReturnSharing = new AlertDialog.Builder(context)
                .setTitle(R.string.cod_dlg_reject_title)
                .setMessage(R.string.cod_dlg_reject_subtitle)
                .setPositiveButton(R.string.cos_dlg_positive_button2, (dialogPositive, whichPositive) -> {
                    dialogPositive.dismiss();
                })
                .show();
    }

    /**
     * Error dialog
     */
    private static void errorSharingDialog(Context context) {
        mAlertDialogErrorSharing = new AlertDialog.Builder(context)
                .setTitle(R.string.cos_dlg_error_title)
                .setMessage(R.string.cos_dlg_error_subtitle)
                .setPositiveButton(R.string.cos_dlg_positive_button2, (dialogPositive, whichPositive) -> {
                    dialogPositive.dismiss();
                })
                .show();
    }

    /**
     * Return dialog
     */
    private static void returnDialog(Context context, String message, String sharingId) {
        mAlertDialogErrorSharing = new AlertDialog.Builder(context)
                .setTitle(R.string.cos_dlg_return_title)
                .setMessage(message)
                .setNegativeButton(R.string.cos_dlg_negative_button2, (dialogPositive, whichPositive) -> {
//                    rejectSharingDialog(context, sharingId);
                    dialogPositive.dismiss();
                })
                .setPositiveButton(R.string.cos_dlg_positive_button, (dialogNegative, whichNegative) -> {
                    WarplyManager.cosmoteRetrieveSharing(new CosmoteRetrieveSharingRequest()
                                    .setSharingId(sharingId)
                                    .setAccept(true),
                            new CallbackReceiver<JSONObject>() {
                                @Override
                                public void onSuccess(JSONObject result) {
                                    int status = result.optInt("status", 2);
                                    if (status == 1) {
                                        WarplyManager.getUserCouponsWithCouponsets(new CallbackReceiver<CouponList>() {
                                            @Override
                                            public void onSuccess(CouponList result) {
                                                WarplyCouponsChangedEventModel couponsChanged = new WarplyCouponsChangedEventModel();
                                                couponsChanged.setChanged(true);
                                                EventBus.getDefault().post(new WarplyEventBusManager(couponsChanged));
                                            }

                                            @Override
                                            public void onFailure(int errorCode) {

                                            }
                                        });
//                                        acceptSharingDialog(context);
                                    } else
                                        errorSharingDialog(context);
                                }

                                @Override
                                public void onFailure(int errorCode) {
                                    errorSharingDialog(context);
                                }
                            });
                })
                .show();
    }

    /**
     * Expired dialog
     */
    private static void errorExpiredDialog(Context context, String message) {
        mAlertDialogErrorExpiredSharing = new AlertDialog.Builder(context)
                .setTitle(R.string.cos_dlg_error_title)
                .setMessage(message)
                .setPositiveButton(R.string.cos_dlg_positive_button2, (dialogPositive, whichPositive) -> {
                    dialogPositive.dismiss();
                })
                .show();
    }

    /**
     * Used dialog
     */
    private static void errorUsedDialog(Context context, String message) {
        mAlertDialogErrorUsedSharing = new AlertDialog.Builder(context)
                .setTitle(R.string.cos_dlg_error_title)
                .setMessage(message)
                .setPositiveButton(R.string.cos_dlg_positive_button2, (dialogPositive, whichPositive) -> {
                    dialogPositive.dismiss();
                })
                .show();
    }

    /**
     * Open CCMS campaign
     */
    public static String constructCcmsUrl(Campaign item, LoyaltyContextualOfferModel ccmsItem) {
        WarplyManager.getSingleCampaign(item.getSessionUUID());
        Log.v("CCMS_PRESSED", ccmsItem.getId() + " " + ccmsItem.getOfferName());
        item.setNew(false);
//        String tempAudienceLevel = ccmsItem.getEligibleAssets() != null && ccmsItem.getEligibleAssets().size() > 0 ? "MSISDN" : "GUID";
        String assets = "";
        if (ccmsItem.getEligibleAssets() != null && ccmsItem.getEligibleAssets().size() > 0) {
            for (String singleAsset : ccmsItem.getEligibleAssets()) {
                assets = assets + singleAsset + ",";
            }

            if (!TextUtils.isEmpty(assets)) {
                assets = assets.substring(0, assets.length() - 1);
            }
        }
        String url = item.getIndexUrl();
//                + "?web_id=" + WarpUtils.getWebId(Warply.getWarplyContext())
//                + "&app_uuid=" + WarplyProperty.getAppUuid(Warply.getWarplyContext())
//                + "&api_key=" + WarpUtils.getApiKey(Warply.getWarplyContext())
//                + "&session_uuid=" + item.getSessionUUID()
//                + "&access_token=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("access_token")
//                + "&refresh_token=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("refresh_token")
//                + "&client_id=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_id")
//                + "&client_secret=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_secret")
//                + "&Session_ID=" + ccmsItem.getSessionId()
//                + "&businessService=" + ccmsItem.getBusinessService()
//                + "&offerName=" + ccmsItem.getOfferName()
//                + "&productType=" + ccmsItem.getProductType()
//                + "&provDuration=" + ccmsItem.getProvDuration()
//                + "&noOfRecurrance=" + ccmsItem.getNoOfRecurrance()
//                + "&price=" + ccmsItem.getPrice()
//                + "&discount=" + ccmsItem.getDiscount()
//                + "&voiceCategory=" + ccmsItem.getVoiceCategory()
//                + "&dataCategory=" + ccmsItem.getDataCategory()
//                + "&minsValue=" + ccmsItem.getMinsValue()
//                + "&dataValue=" + ccmsItem.getDataValue()
//                + "&provStepValueMins=" + ccmsItem.getProvStepValueMins()
//                + "&UACIOfferTrackingCode=" + ccmsItem.getUACIOfferTrackingCode()
//                + "&OFFERCODE1=" + ccmsItem.getOfferCode1()
//                + "&SCORE=" + ccmsItem.getScore()
//                + "&WAVE=" + ccmsItem.getWave()
//                + "&VALIDITY=" + ccmsItem.getValidity()
//                + "&TREATMENT_CODE=" + ccmsItem.getTreatmentCode()
//                + "&OfferAudienceLevel=" + ccmsItem.getOfferAudienceLevel()
//                + "&ASSETS=" + assets
//                + "&ZONE=" + ccmsItem.getZone();

        JSONObject params = new JSONObject();
        try {
            params.putOpt("web_id", WarpUtils.getWebId(Warply.getWarplyContext()));
            params.putOpt("app_uuid", WarplyProperty.getAppUuid(Warply.getWarplyContext()));
            params.putOpt("api_key", WarpUtils.getApiKey(Warply.getWarplyContext()));
            params.putOpt("session_uuid", item.getSessionUUID());
            params.putOpt("access_token", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("access_token"));
            params.putOpt("refresh_token", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("refresh_token"));
            params.putOpt("client_id", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_id"));
            params.putOpt("client_secret", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_secret"));
            params.putOpt("Session_ID", ccmsItem.getSessionId());
            params.putOpt("businessService", ccmsItem.getBusinessService());
            params.putOpt("offerName", ccmsItem.getOfferName());
            params.putOpt("productType", ccmsItem.getProductType());
            params.putOpt("provDuration", ccmsItem.getProvDuration());
            params.putOpt("noOfRecurrance", ccmsItem.getNoOfRecurrance());
            params.putOpt("price", ccmsItem.getPrice());
            params.putOpt("discount", ccmsItem.getDiscount());
            params.putOpt("voiceCategory", ccmsItem.getVoiceCategory());
            params.putOpt("dataCategory", ccmsItem.getDataCategory());
            params.putOpt("minsValue", ccmsItem.getMinsValue());
            params.putOpt("dataValue", ccmsItem.getDataValue());
            params.putOpt("provStepValueMins", ccmsItem.getProvStepValueMins());
            params.putOpt("UACIOfferTrackingCode", ccmsItem.getUACIOfferTrackingCode());
            params.putOpt("OFFERCODE1", ccmsItem.getOfferCode1());
            params.putOpt("SCORE", ccmsItem.getScore());
            params.putOpt("WAVE", ccmsItem.getWave());
            params.putOpt("VALIDITY", ccmsItem.getValidity());
            params.putOpt("TREATMENT_CODE", ccmsItem.getTreatmentCode());
            params.putOpt("OfferAudienceLevel", ccmsItem.getOfferAudienceLevel());
            params.putOpt("ASSETS", assets);
            params.putOpt("ZONE", ccmsItem.getZone());
        } catch (JSONException e) {
            e.printStackTrace();
        }

        WarpUtils.setWebviewParams(Warply.getWarplyContext(), params);

        return url;
    }

    /**
     * Open CCMS campaign
     */
    public static String constructCcmsUrl(LoyaltyContextualOfferModel item) {
        String url = "";
        if (item != null) {
            if (mCampaignList != null && mCampaignList.size() > 0) {
                for (Campaign camp : mCampaignList) {
                    try {
                        JSONObject extraFields = WarpJSONParser.getJSONFromString(camp.getExtraFields());
                        if (extraFields != null) {
                            if (extraFields.has("ccms_offer") && extraFields.optString("ccms_offer").equals("true")) {
                                if (item.getLoyaltyCampaignId().equals(camp.getSessionUUID())) {
                                    WarplyManager.getSingleCampaign(camp.getSessionUUID());
//                                    String tempAudienceLevel = item.getEligibleAssets() != null && item.getEligibleAssets().size() > 0 ? "MSISDN" : "GUID";
                                    String assets = "";
                                    if (item.getEligibleAssets() != null && item.getEligibleAssets().size() > 0) {
                                        for (String singleAsset : item.getEligibleAssets()) {
                                            assets = assets + singleAsset + ",";
                                        }

                                        if (!TextUtils.isEmpty(assets)) {
                                            assets = assets.substring(0, assets.length() - 1);
                                        }
                                    }

                                    url = camp.getIndexUrl();
//                                            + "?web_id=" + WarpUtils.getWebId(Warply.getWarplyContext())
//                                            + "&app_uuid=" + WarplyProperty.getAppUuid(Warply.getWarplyContext())
//                                            + "&api_key=" + WarpUtils.getApiKey(Warply.getWarplyContext())
//                                            + "&session_uuid=" + camp.getSessionUUID()
//                                            + "&access_token=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("access_token")
//                                            + "&refresh_token=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("refresh_token")
//                                            + "&client_id=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_id")
//                                            + "&client_secret=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_secret")
//                                            + "&Session_ID=" + item.getSessionId()
//                                            + "&businessService=" + item.getBusinessService()
//                                            + "&offerName=" + item.getOfferName()
//                                            + "&productType=" + item.getProductType()
//                                            + "&provDuration=" + item.getProvDuration()
//                                            + "&noOfRecurrance=" + item.getNoOfRecurrance()
//                                            + "&price=" + item.getPrice()
//                                            + "&discount=" + item.getDiscount()
//                                            + "&voiceCategory=" + item.getVoiceCategory()
//                                            + "&dataCategory=" + item.getDataCategory()
//                                            + "&minsValue=" + item.getMinsValue()
//                                            + "&dataValue=" + item.getDataValue()
//                                            + "&provStepValueMins=" + item.getProvStepValueMins()
//                                            + "&UACIOfferTrackingCode=" + item.getUACIOfferTrackingCode()
//                                            + "&OFFERCODE1=" + item.getOfferCode1()
//                                            + "&SCORE=" + item.getScore()
//                                            + "&WAVE=" + item.getWave()
//                                            + "&VALIDITY=" + item.getValidity()
//                                            + "&TREATMENT_CODE=" + item.getTreatmentCode()
//                                            + "&OfferAudienceLevel=" + item.getOfferAudienceLevel()
//                                            + "&ASSETS=" + assets
//                                            + "&ZONE=" + item.getZone();

                                    JSONObject params = new JSONObject();
                                    try {
                                        params.putOpt("web_id", WarpUtils.getWebId(Warply.getWarplyContext()));
                                        params.putOpt("app_uuid", WarplyProperty.getAppUuid(Warply.getWarplyContext()));
                                        params.putOpt("api_key", WarpUtils.getApiKey(Warply.getWarplyContext()));
                                        params.putOpt("session_uuid", camp.getSessionUUID());
                                        params.putOpt("access_token", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("access_token"));
                                        params.putOpt("refresh_token", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("refresh_token"));
                                        params.putOpt("client_id", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_id"));
                                        params.putOpt("client_secret", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_secret"));
                                        params.putOpt("Session_ID", item.getSessionId());
                                        params.putOpt("businessService", item.getBusinessService());
                                        params.putOpt("offerName", item.getOfferName());
                                        params.putOpt("productType", item.getProductType());
                                        params.putOpt("provDuration", item.getProvDuration());
                                        params.putOpt("noOfRecurrance", item.getNoOfRecurrance());
                                        params.putOpt("price", item.getPrice());
                                        params.putOpt("discount", item.getDiscount());
                                        params.putOpt("voiceCategory", item.getVoiceCategory());
                                        params.putOpt("dataCategory", item.getDataCategory());
                                        params.putOpt("minsValue", item.getMinsValue());
                                        params.putOpt("dataValue", item.getDataValue());
                                        params.putOpt("provStepValueMins", item.getProvStepValueMins());
                                        params.putOpt("UACIOfferTrackingCode", item.getUACIOfferTrackingCode());
                                        params.putOpt("OFFERCODE1", item.getOfferCode1());
                                        params.putOpt("SCORE", item.getScore());
                                        params.putOpt("WAVE", item.getWave());
                                        params.putOpt("VALIDITY", item.getValidity());
                                        params.putOpt("TREATMENT_CODE", item.getTreatmentCode());
                                        params.putOpt("OfferAudienceLevel", item.getOfferAudienceLevel());
                                        params.putOpt("ASSETS", assets);
                                        params.putOpt("ZONE", item.getZone());
                                    } catch (JSONException e) {
                                        e.printStackTrace();
                                    }

                                    WarpUtils.setWebviewParams(Warply.getWarplyContext(), params);
                                    break;
                                }
                            }
                        }
                    } catch (Exception exception) {
                        Log.v("WarplyManagerHelper_CCMS_Error", exception.toString());
                    }
                }
            }
        }

        return url;
    }

    public static void handleContextualCampaigns(Context context, LoyaltyContextualOfferModel item) {
        if (item != null) {
            if (TextUtils.isEmpty(item.getLoyaltyCampaignId()) || item.getLoyaltyCampaignId().equals("NA")) {
                LoyaltySDKFirebaseEventModel analyticsEvent = new LoyaltySDKFirebaseEventModel();
                analyticsEvent.setEventName("loyalty_sdk_offer_selected");
                analyticsEvent.setParameter("name", item.getTitle());
                analyticsEvent.setParameter("type", "gift");
                EventBus.getDefault().post(new WarplyEventBusManager(analyticsEvent));

                WarplyAnalyticsManager.logTrackersEvent(context, "click", ("Contextual").concat(":").concat(item.getSessionId()));
                Intent intent = new Intent(context, ContextualActivity.class);
                intent.putExtra("ccms", item);
                context.startActivity(intent);
            } else if (mCampaignListAll != null && mCampaignListAll.size() > 0) {
                for (Campaign camp : mCampaignListAll) {
                    try {
                        JSONObject extraFields = WarpJSONParser.getJSONFromString(camp.getExtraFields());
                        if (extraFields != null) {
                            if (extraFields.has("ccms_offer") && extraFields.optString("ccms_offer").equals("true")) {
                                if (item.getLoyaltyCampaignId().equals(camp.getSessionUUID())) {
                                    WarplyManager.getSingleCampaign(camp.getSessionUUID());
                                    String assets = "";
                                    if (item.getEligibleAssets() != null && item.getEligibleAssets().size() > 0) {
                                        for (String singleAsset : item.getEligibleAssets()) {
                                            assets = assets + singleAsset + ",";
                                        }

                                        if (!TextUtils.isEmpty(assets)) {
                                            assets = assets.substring(0, assets.length() - 1);
                                        }
                                    }

                                    String url = camp.getIndexUrl();
//                                            + "?web_id=" + WarpUtils.getWebId(Warply.getWarplyContext())
//                                            + "&app_uuid=" + WarplyProperty.getAppUuid(Warply.getWarplyContext())
//                                            + "&api_key=" + WarpUtils.getApiKey(Warply.getWarplyContext())
//                                            + "&session_uuid=" + camp.getSessionUUID()
//                                            + "&access_token=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("access_token")
//                                            + "&refresh_token=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("refresh_token")
//                                            + "&client_id=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_id")
//                                            + "&client_secret=" + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_secret")
//                                            + "&Session_ID=" + item.getSessionId()
//                                            + "&businessService=" + item.getBusinessService()
//                                            + "&offerName=" + item.getOfferName()
//                                            + "&productType=" + item.getProductType()
//                                            + "&provDuration=" + item.getProvDuration()
//                                            + "&noOfRecurrance=" + item.getNoOfRecurrance()
//                                            + "&price=" + item.getPrice()
//                                            + "&discount=" + item.getDiscount()
//                                            + "&voiceCategory=" + item.getVoiceCategory()
//                                            + "&dataCategory=" + item.getDataCategory()
//                                            + "&minsValue=" + item.getMinsValue()
//                                            + "&dataValue=" + item.getDataValue()
//                                            + "&provStepValueMins=" + item.getProvStepValueMins()
//                                            + "&UACIOfferTrackingCode=" + item.getUACIOfferTrackingCode()
//                                            + "&OFFERCODE1=" + item.getOfferCode1()
//                                            + "&SCORE=" + item.getScore()
//                                            + "&WAVE=" + item.getWave()
//                                            + "&VALIDITY=" + item.getValidity()
//                                            + "&TREATMENT_CODE=" + item.getTreatmentCode()
//                                            + "&OfferAudienceLevel=" + item.getOfferAudienceLevel()
//                                            + "&ASSETS=" + assets
//                                            + "&ZONE=" + item.getZone();

                                    JSONObject params = new JSONObject();
                                    try {
                                        params.putOpt("web_id", WarpUtils.getWebId(Warply.getWarplyContext()));
                                        params.putOpt("app_uuid", WarplyProperty.getAppUuid(Warply.getWarplyContext()));
                                        params.putOpt("api_key", WarpUtils.getApiKey(Warply.getWarplyContext()));
                                        params.putOpt("session_uuid", camp.getSessionUUID());
                                        params.putOpt("access_token", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("access_token"));
                                        params.putOpt("refresh_token", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("refresh_token"));
                                        params.putOpt("client_id", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_id"));
                                        params.putOpt("client_secret", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_secret"));
                                        params.putOpt("Session_ID", item.getSessionId());
                                        params.putOpt("businessService", item.getBusinessService());
                                        params.putOpt("offerName", item.getOfferName());
                                        params.putOpt("productType", item.getProductType());
                                        params.putOpt("provDuration", item.getProvDuration());
                                        params.putOpt("noOfRecurrance", item.getNoOfRecurrance());
                                        params.putOpt("price", item.getPrice());
                                        params.putOpt("discount", item.getDiscount());
                                        params.putOpt("voiceCategory", item.getVoiceCategory());
                                        params.putOpt("dataCategory", item.getDataCategory());
                                        params.putOpt("minsValue", item.getMinsValue());
                                        params.putOpt("dataValue", item.getDataValue());
                                        params.putOpt("provStepValueMins", item.getProvStepValueMins());
                                        params.putOpt("UACIOfferTrackingCode", item.getUACIOfferTrackingCode());
                                        params.putOpt("OFFERCODE1", item.getOfferCode1());
                                        params.putOpt("SCORE", item.getScore());
                                        params.putOpt("WAVE", item.getWave());
                                        params.putOpt("VALIDITY", item.getValidity());
                                        params.putOpt("TREATMENT_CODE", item.getTreatmentCode());
                                        params.putOpt("OfferAudienceLevel", item.getOfferAudienceLevel());
                                        params.putOpt("ASSETS", assets);
                                        params.putOpt("ZONE", item.getZone());
                                    } catch (JSONException e) {
                                        e.printStackTrace();
                                    }

                                    WarpUtils.setWebviewParams(Warply.getWarplyContext(), params);
                                    WarplyAnalyticsManager.logTrackersEvent(context, "click", ("Contextual").concat(":").concat(item.getSessionId()));
                                    context.startActivity(WarpViewActivity.createIntentFromURL(context, url));
                                    break;
                                }
                            } else if (extraFields.has("type") && extraFields.optString("type").equals("telco")) {
                                if (item.getLoyaltyCampaignId().equals(camp.getSessionUUID())) {
                                    LoyaltySDKFirebaseEventModel analyticsEvent = new LoyaltySDKFirebaseEventModel();
                                    analyticsEvent.setEventName("loyalty_sdk_offer_selected");
                                    analyticsEvent.setParameter("name", item.getTitle());
                                    analyticsEvent.setParameter("type", "gift");
                                    EventBus.getDefault().post(new WarplyEventBusManager(analyticsEvent));

                                    WarplyAnalyticsManager.logTrackersEvent(context, "click", ("Telco").concat(":").concat(item.getSessionId()));
                                    WarplyManager.getSingleCampaign(camp.getSessionUUID());
                                    Intent intent = new Intent(context, TelcoActivity.class);
                                    intent.putExtra("ccms", item);
                                    intent.putExtra("loyalty", (Serializable) camp);
                                    context.startActivity(intent);
                                    break;
                                }
                            }
                        }
                    } catch (Exception exception) {
                        Log.v("handleContextual_Error", exception.toString());
                    }
                }
            }
        }
    }

    public static void handleLoyaltyCampaigns(Context context, Campaign item) {
        if (item != null) {
            if (item.getType().equals("coupon")) {
                if (WarplyManagerHelper.getCCMSLoyaltyCampaigns() != null && WarplyManagerHelper.getCCMSLoyaltyCampaigns().size() > 0) {
                    LoyaltyContextualOfferModel ccmsItem = null;
                    for (LoyaltyContextualOfferModel ccms : WarplyManagerHelper.getCCMSLoyaltyCampaigns()) {
                        if (ccms.getLoyaltyCampaignId().equals(item.getSessionUUID())) {
                            ccmsItem = ccms;
                            break;
                        }
                    }

                    if (ccmsItem != null) {
                        try {
                            JSONObject campaignSettings = WarpJSONParser.getJSONFromString(item.getCampaignTypeSettings());
                            if (campaignSettings != null) {
                                if (campaignSettings.has("couponset")) {
                                    String cpnUuid = campaignSettings.optString("couponset");
                                    for (Couponset cpns : WarplyManagerHelper.getCouponsets()) {
                                        if (cpns.getUuid().equals(cpnUuid)) {
                                            LoyaltySDKFirebaseEventModel analyticsEvent = new LoyaltySDKFirebaseEventModel();
                                            analyticsEvent.setEventName("loyalty_sdk_offer_selected");
                                            analyticsEvent.setParameter("name", item.getTitle());
                                            analyticsEvent.setParameter("type", "coupon");
                                            EventBus.getDefault().post(new WarplyEventBusManager(analyticsEvent));

                                            WarplyAnalyticsManager.logTrackersEvent(context, "click", ("Loyalty").concat(":").concat(item.getSessionUUID()));
                                            WarplyManager.getSingleCampaign(item.getSessionUUID());
                                            Intent intent = new Intent(context, CouponsetInfoActivity.class);
                                            intent.putExtra("couponset", (Serializable) cpns);
                                            intent.putExtra("loyalty", (Serializable) item);
                                            intent.putExtra("ccms", ccmsItem);
                                            context.startActivity(intent);
                                            break;
                                        }
                                    }
                                } else {
                                    WarplyAnalyticsManager.logTrackersEvent(context, "click", ("Loyalty").concat(":").concat(item.getSessionUUID()));
                                    context.startActivity(WarpViewActivity.createIntentFromURL(context, WarplyManagerHelper.constructCampaignUrl(item)));
                                }
                            }
                        } catch (Exception exception) {
                            exception.printStackTrace();
                        }

                        return;
                    }
                }

                try {
                    JSONObject campaignSettings = WarpJSONParser.getJSONFromString(item.getCampaignTypeSettings());
                    if (campaignSettings != null) {
                        if (campaignSettings.has("couponset")) {
                            String cpnUuid = campaignSettings.optString("couponset");
                            for (Couponset cpns : WarplyManagerHelper.getCouponsets()) {
                                if (cpns.getUuid().equals(cpnUuid)) {
                                    LoyaltySDKFirebaseEventModel analyticsEvent = new LoyaltySDKFirebaseEventModel();
                                    analyticsEvent.setEventName("loyalty_sdk_offer_selected");
                                    analyticsEvent.setParameter("name", item.getTitle());
                                    analyticsEvent.setParameter("type", "coupon");
                                    EventBus.getDefault().post(new WarplyEventBusManager(analyticsEvent));

                                    WarplyAnalyticsManager.logTrackersEvent(context, "click", ("Loyalty").concat(":").concat(item.getSessionUUID()));
                                    WarplyManager.getSingleCampaign(item.getSessionUUID());
                                    Intent intent = new Intent(context, CouponsetInfoActivity.class);
                                    intent.putExtra("couponset", (Serializable) cpns);
                                    intent.putExtra("loyalty", (Serializable) item);
                                    context.startActivity(intent);
                                    break;
                                }
                            }
                        } else {
                            WarplyAnalyticsManager.logTrackersEvent(context, "click", ("Loyalty").concat(":").concat(item.getSessionUUID()));
                            context.startActivity(WarpViewActivity.createIntentFromURL(context, WarplyManagerHelper.constructCampaignUrl(item)));
                        }
                    }
                } catch (Exception exception) {
                    Log.v("handleLoyalty_Error", exception.toString());
                }

                return;
            }

            WarplyAnalyticsManager.logTrackersEvent(context, "click", ("Loyalty").concat(":").concat(item.getSessionUUID()));
            context.startActivity(WarpViewActivity.createIntentFromURL(context, WarplyManagerHelper.constructCampaignUrl(item)));
        }
    }

    /**
     * Set the List with CCMS Campaigns
     */
    public static void setCCMSLoyaltyCampaigns(ArrayList<LoyaltyContextualOfferModel> list) {
        mCCMSList.clear();
//        mCCMSList.addAll(list);
        mCCMSList = list;

        OneTimeWorkRequest mywork = new OneTimeWorkRequest.Builder(EventService.class).build();
        WorkManager.getInstance(Warply.getWarplyContext()).enqueue(mywork);
    }

    public static void clearCCMSLoyaltyCampaigns() {
        mCCMSList.clear();

        OneTimeWorkRequest mywork = new OneTimeWorkRequest.Builder(EventService.class).build();
        WorkManager.getInstance(Warply.getWarplyContext()).enqueue(mywork);
    }

    /**
     * Get the List with CCMS Campaigns
     */
    public static ArrayList<LoyaltyContextualOfferModel> getCCMSLoyaltyCampaigns() {
        return mCCMSList;
    }

    /**
     * Save the LoyaltyGiftsForYouPackage list
     */
    public static void setSeasonalList(ArrayList<LoyaltyGiftsForYouPackage> seasonalCoupons) {
        mSeasonalList.clear();
        mSeasonalList = seasonalCoupons;
    }

    /**
     * Get the LoyaltyGiftsForYouPackage list
     */
    public static ArrayList<LoyaltyGiftsForYouPackage> getSeasonalList() {
        return mSeasonalList;
    }

    /**
     * Open Questionnaire
     */
    public static String openQuestionnaire() {
        if (mCampaignListAll != null && mCampaignListAll.size() > 0) {
            for (Campaign camp : mCampaignListAll) {
                if (camp.getOfferCategory().equals("questionnaire")) {
                    return constructCampaignUrl(camp);
                }
            }
        }

        return "";
    }

    /**
     * Open Pacing
     */
    public static String openPacing() {
        if (mCampaignListAll != null && mCampaignListAll.size() > 0) {
            for (Campaign camp : mCampaignListAll) {
                if (camp.getOfferCategory().equals("pacing")) {
                    String pacingUrl = constructCampaignUrl(camp, true);
                    WarpUtils.log("PACING_LOG: " + pacingUrl);
                    return pacingUrl;
                }
            }
        }

        return "";
    }

    /**
     * Open Couponset
     */
    public static void openCouponset(Context context, Campaign item) {
        try {
            JSONObject campaignSettings = WarpJSONParser.getJSONFromString(item.getCampaignTypeSettings());
            if (campaignSettings != null) {
                if (campaignSettings.has("couponset")) {
                    String cpnUuid = campaignSettings.optString("couponset");
                    for (Couponset cpns : mCouponsetsList) {
                        if (cpns.getUuid().equals(cpnUuid)) {
                            Intent intent = new Intent(context, CouponsetInfoActivity.class);
                            intent.putExtra("couponset", (Serializable) cpns);
                            intent.putExtra("loyalty", (Serializable) item);
                            context.startActivity(intent);
                            break;
                        }
                    }
                }
            }
        } catch (Exception exception) {
            exception.printStackTrace();
        }
    }

    public static void openTelco(Context context, Campaign item) {
        for (LoyaltyContextualOfferModel ccms : mCCMSList) {
            if (ccms.getLoyaltyCampaignId().equals(item.getSessionUUID())) {
                Intent intent = new Intent(context, TelcoActivity.class);
                intent.putExtra("ccms", ccms);
                intent.putExtra("loyalty", (Serializable) item);
                context.startActivity(intent);
                break;
            }
        }
    }

    /**
     * Save if customer is nonTelco and if he/she has accepted the consents,
     * And his msisdn list
     */
    public static void saveCustomerState(CustomerStateModel customerState) {
        mCustomerStateModel = customerState;

        WarplyIntegrationRequest request = new WarplyIntegrationRequest();
        request.setIsNoTelco(customerState.isNonTelco())
                .setHasAcceptedConsent(customerState.isAcceptedConsent())
                .setGuid(customerState.getGuid())
                .setMsisdnList(customerState.getMsisdnList());
        WarplyManager.consumerIntegration(request, new CallbackReceiver<JSONObject>() {
            @Override
            public void onSuccess(JSONObject result) {
                WarpUtils.log("CUSTOMER STATE SUCCESS");
            }

            @Override
            public void onFailure(int errorCode) {
                WarpUtils.log("CUSTOMER STATE ERROR");
            }
        });
    }

    public static void setTrackersEnabled(Context context, boolean isEnabled) {
        WarpUtils.setTrackersEnabled(context, isEnabled);

//        JSONObject obj = new JSONObject();
//        try {
//            obj.putOpt("trackers_enabled", isEnabled);
//        } catch (JSONException e) {
//            e.printStackTrace();
//        }
//
//        WarplyManager.editConsumer(new WarplyEditConsumerRequest()
//                .setHasProfileMetadata(true)
//                .setProfileMetadata(obj), new CallbackReceiver<JSONObject>() {
//            @Override
//            public void onSuccess(JSONObject result) {
//                WarpUtils.log("EDIT CONSUMER SUCCESS");
//                WarplyManager.getConsumer(new WarplyConsumerRequest(), new CallbackReceiver<Consumer>() {
//                    @Override
//                    public void onSuccess(Consumer result) {
//                        WarplyManagerHelper.setConsumerInternal(result);
//
//                        if (result != null) {
//                            JSONObject profMetadata = WarpJSONParser.getJSONFromString(result.getProfileMetadata());
//                            if (profMetadata != null && profMetadata.has("nonTelco")) {
//                                WarpUtils.setUserNonTelco(Warply.getWarplyContext(), profMetadata.optBoolean("nonTelco"));
//                            } else {
//                                WarpUtils.setUserNonTelco(Warply.getWarplyContext(), false);
//                            }
//                        }
//
//                        if (result != null) {
//                            JSONObject profMetadata = WarpJSONParser.getJSONFromString(result.getProfileMetadata());
//                            if (profMetadata != null) {
//                                if (profMetadata.has("badge")) {
//                                    WarpUtils.setUserTag(Warply.getWarplyContext(), profMetadata.optString("badge"));
//                                }
//                            }
//                        }
//                    }
//
//                    @Override
//                    public void onFailure(int errorCode) {
//
//                    }
//                });
//            }
//
//            @Override
//            public void onFailure(int errorCode) {
//                WarpUtils.log("EDIT CONSUMER FAILED");
//            }
//        });
    }

    /**
     * Get if customer is nonTelco and if he/she has accepted the consents
     * And his msisdn list
     */
    public static CustomerStateModel loadCustomerState() {
        return mCustomerStateModel;
    }

    /**
     * Get user badge tag
     */
    public static String getUserTag(Context context) {
        return WarpUtils.getUserTag(context);
    }

    /**
     * Set the List with active D4Y coupons
     */
    public static void setActiveDFYCoupons(ArrayList<ActiveDFYCouponModel> dfyCoupons) {
        mDfyCoupons.clear();
        mDfyCoupons = dfyCoupons;
    }

    /**
     * Get the List with active D4Y coupons
     */
    public static ArrayList<ActiveDFYCouponModel> getActiveDFYCoupons() {
        return mDfyCoupons;
    }

    public static Consumer getConsumerInternal() {
        return mConsumerInternal;
    }

    public static void setConsumerInternal(Consumer consumer) {
        mConsumerInternal = consumer;
    }

    public static Consumer getConsumer() {
        return mConsumer;
    }

    public static void setConsumer(Consumer consumer) {
        mConsumer = consumer;
    }

    public static HashMap<String, CampaignList> getUniqueCampaignList() {
        return mUniqueCampaignList;
    }

    public static CouponList getCouponList() {
        if (mCouponList != null && mCouponList.size() > 0) {
            WarpUtils.log("GET_COUPON_LIST size: " + String.valueOf(mCouponList.size()));
        } else {
            WarpUtils.log("GET_COUPON_LIST size: 0");
        }
        return mCouponList;
    }

    public static CouponList getCouponExpiredList() {
        if (mCouponListExpired != null && mCouponListExpired.size() > 0) {
            WarpUtils.log("GET_COUPON_EXPIRED_LIST size: " + String.valueOf(mCouponListExpired.size()));
        } else {
            WarpUtils.log("GET_COUPON_EXPIRED_LIST size: 0");
        }
        return mCouponListExpired;
    }

    public static void setUniqueCampaignList(ArrayList<Campaign> campaignList) {
        mCampaignList.clear();
        mCampaignList.addAll(campaignList);


//        if (campaignList != null) {
//            for (Campaign campaign : campaignList) {
//                if (mUniqueCampaignList.containsKey(campaign.getOfferCategory().trim())) {
//                    CampaignList tempCampaignList = (CampaignList) mUniqueCampaignList.get(campaign.getOfferCategory().trim());
//                    tempCampaignList.add(campaign);
//                    mUniqueCampaignList.put(campaign.getOfferCategory().trim(), tempCampaignList);
//                } else {
//                    CampaignList tempCampaignList = new CampaignList();
//                    tempCampaignList.add(campaign);
//                    mUniqueCampaignList.put(campaign.getOfferCategory().trim(), tempCampaignList);
//                }
//            }
//        }
    }

    public static void setCampaignList(ArrayList<Campaign> campaignList) {
//        mCampaignList.clear();
//        mCampaignList.addAll(campaignList);

//        Set<Campaign> set = new LinkedHashSet<>(campaignList);
//        mCampaignListAll.clear();
//        mCampaignListAll.addAll(set);

        mCampaignListAll.clear();
        mCampaignListAll = campaignList;

        mCampaignList.clear();
        for (Campaign camp : campaignList) {
            try {
                JSONObject extraFields = WarpJSONParser.getJSONFromString(camp.getExtraFields());
                if (extraFields != null) {
                    if (extraFields.length() == 0 || !(extraFields.has("ccms_offer") || extraFields.has("type"))) {
                        mCampaignList.add(camp);
                    }
                }
            } catch (Exception exception) {
                mCampaignList.add(camp);
            }
        }
    }

    public static void setCarouselList(ArrayList<Campaign> carouselList) {
        WarpUtils.log("SET_CAROUSEL_LIST_PARAM_COUNT: " + String.valueOf(carouselList.size()));
        WarpUtils.log("SET_CAROUSEL_LIST_PARAM_JSON: " + String.valueOf(carouselList.toString()));
        mCarouselList.clear();
        mCarouselList.addAll(carouselList);
        WarpUtils.log("SET_CAROUSEL_LIST_VAR_COUNT: " + String.valueOf(mCarouselList.size()));
        WarpUtils.log("SET_CAROUSEL_LIST_VAR_JSON: " + String.valueOf(mCarouselList.toString()));
    }

    public static ArrayList<Campaign> getCarouselList() {
        return mCarouselList;
    }

    public static ArrayList<Campaign> getCampaignList() {
        return mCampaignList;
    }

    public static ArrayList<Campaign> getCampaignListAll() {
        return mCampaignListAll;
    }

    public static void setCouponList(CouponList couponList) {
        LoyaltyBadgeModel badge = new LoyaltyBadgeModel();
        CouponList cpnlist = new CouponList();
        CouponList cpnlistExpired = new CouponList();
        float couponValue = 0.0f;
        int couponCount = 0;
        for (Coupon coupon : couponList) {
            if (coupon.getStatus() == 1) {
                cpnlist.add(coupon);
            } else if (coupon.getStatus() == 0) {
                cpnlistExpired.add(coupon);
                try {
                    couponCount += 1;
//                    couponValue += Float.parseFloat(coupon.getDiscount());
                    couponValue += (float) coupon.getFinal_price();
                } catch (NumberFormatException e) {
                    couponCount -= 1;
                }
            }
        }
        badge.setCouponCount(couponCount);
        badge.setValue(couponValue);
        mLoyaltyBadge = badge;
        mCouponList.clear();
        mCouponList.addAll(cpnlist);

        final ExecutorService executorShops = Executors.newFixedThreadPool(1);
        executorShops.submit(() -> {
            mCouponListExpired.clear();
            Collections.sort(cpnlistExpired, (coupon1, coupon2) -> coupon2.getRedeemDate().compareTo(coupon1.getRedeemDate()));
            mCouponListExpired.addAll(cpnlistExpired);
            executorShops.shutdownNow();
        });
    }

    public static LoyaltyBadgeModel getLoyaltyBadge() {
        return mLoyaltyBadge;
    }

    public static CouponsetsList getCouponsets() {
        return mCouponsetsList;
    }

    public static void setCouponsets(CouponsetsList couponsets) {
        mCouponsetsList = couponsets;
    }

    public static void setMerchantList(MerchantList merchantList) {
        mMerchantList.clear();
        mMerchantList = merchantList;
    }

    public static MerchantList getMerchantList() {
        return mMerchantList;
    }

    public static float getDealsCouponsSum() {
        return mDealsSum;
    }

    public static void setDealsCouponsSum(float sum) {
        mDealsSum = sum;
    }

    public static ArrayList<UnifiedCoupon> getMarketCoupons() {
        return mMarketCoupons;
    }

    public static void setMarketCoupons(ArrayList<UnifiedCoupon> marketCoupons) {
        mMarketCoupons = marketCoupons;
    }

    public static boolean checkForLoyaltySDKNotification(Context context, Map<String, String> pushPayload) {
        Bundle data = convertToBundle(pushPayload);
        if (data == null || !data.containsKey("loyalty-action"))
            return false;
        PushCampaign pc = new PushCampaign(data);

        WarpUtils.log("Received push with action: " + pc.getAction());
        if (pc.getAction() == 0 && !pc.hasActions())
            FCMBaseMessagingService.showCampaignNotification(context, pc);
        else
            showDefaultNotification(context, data);

        return true;
    }

    private static Bundle convertToBundle(Map<String, String> data) {
        Bundle bundle = new Bundle();
        for (Map.Entry<String, String> entry : data.entrySet()) {
            bundle.putString(entry.getKey(), entry.getValue());
        }

        return bundle;
    }

    private static void showDefaultNotification(Context context, Bundle data) {
        PushCampaign pc = new PushCampaign(data);

        if (pc != null) {
            PackageManager pm = context.getPackageManager();
            Intent newIntent = pm.getLaunchIntentForPackage(context.getPackageName());

            FCMBaseMessagingService.showNotification(context, pc, newIntent, null, null, null, 0);
        }

    }

    public static void logoutUser() {
        if (isMyServiceRunning(WarplyHealthService.class)) {
            Intent stepsServiceIntent = new Intent(Warply.getWarplyContext(), WarplyHealthService.class);
            Warply.getWarplyContext().stopService(stepsServiceIntent);
        }
    }

    private static boolean isMyServiceRunning(Class<?> serviceClass) {
        ActivityManager manager = (ActivityManager) Warply.getWarplyContext().getSystemService(Context.ACTIVITY_SERVICE);
        for (ActivityManager.RunningServiceInfo service : manager.getRunningServices(Integer.MAX_VALUE)) {
            if (serviceClass.getName().equals(service.service.getClassName())) {
                return true;
            }
        }
        return false;
    }

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================
}