package ly.warp.sdk.utils;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.ServiceInfo;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.os.Bundle;
import android.provider.Settings;

import androidx.annotation.DrawableRes;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;

import android.text.TextUtils;
import android.util.TypedValue;

import java.io.IOException;
import java.util.Properties;

import ly.warp.sdk.R;
import ly.warp.sdk.utils.constants.WarpConstants;

public class WarplyProperty {

    // ===========================================================
    // Constants
    // ===========================================================

    public static final String PROPERTIES_FILE = "warply.properties";

    public static final String KEY_UUID = "Uuid";
    public static final String KEY_DEBUG = "Debug";
    public static final String KEY_SEND_PACKAGES = "SendPackages";
    public static final String KEY_WEB_ACTION_HANDLER = "WebActionHandler";
    public static final String KEY_PROGRESS_COLOR = "ProgressColor";
    public static final String KEY_PROGRESS_DRAWABLE = "ProgressDrawable";
    public static final String KEY_PUSH_COLOR = "PushColor";
    public static final String KEY_PUSH_ICON = "PushIcon";
    public static final String KEY_PUSH_SOUND = "PushSound";
    public static final String KEY_LANGUAGE = "Language";
    public static final String KEY_MERCHANT_ID = "MerchantId";
    public static final String KEY_LOGIN_TYPE = "LoginType";
    public static final String KEY_DL_URL_SCHEME = "DL_URL_SCHEME";
    public static final String KEY_BASE_URL = "BaseURL";
    public static final String KEY_VERIFY_URL = "VerifyURL";

    // ===========================================================
    // Methods
    // ===========================================================

    private static String getWarplyProperty(Context context, String propertyKey) {
        if(context == null) {
            return "";
        }
        try {
            Properties properties = new Properties();
            properties.load(context.getResources().getAssets().open(PROPERTIES_FILE));
            return properties.getProperty(propertyKey);
        } catch (IOException | NullPointerException e) {
            WarpUtils.log(e.getMessage());
            return "";
        }
//        return "";
    }

    @Nullable
    private static Bundle getMetaData(Context context) {

        try {
            return context.getPackageManager()
                    .getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA).metaData;
        } catch (PackageManager.NameNotFoundException e) {
            WarpUtils.log("Metadata fail", e);
            return null;
        }
    }

    private static String getStringMetadata(Context context, String key) {
        Bundle data = getMetaData(context);
        return data != null ? data.getString(key) : "";
    }

    private static boolean getBooleanMetadata(Context context, String key, boolean defValue) {
        Bundle data = getMetaData(context);
        return data != null ? data.getBoolean(key, defValue) : defValue;
    }

    private static int getIntMetadata(Context context, String key, int defValue) {
        Bundle data = getMetaData(context);
        return data != null ? data.getInt(key, defValue) : defValue;
    }

    @NonNull
    private static Drawable getLauncherIcon(Context context, Drawable defaultDrawable) {
        try {
            int iconRes = context.getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA).icon;
            return ContextCompat.getDrawable(context, iconRes);
        } catch (PackageManager.NameNotFoundException e) {
            WarpUtils.log("Metadata fail", e);
        }
        return defaultDrawable;
    }

    private static int getPrimaryColor(Context context, int defValue) {

        try {
            TypedValue typedValue = new TypedValue();
            TypedArray typedArray = context.obtainStyledAttributes(typedValue.data, new int[]{R.attr.colorPrimary});
            int color = typedArray.getColor(0, defValue);
            typedArray.recycle();
            return color;
        } catch (UnsupportedOperationException e) {
            WarpUtils.log(e.getMessage());
        }
        return defValue;
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public static int getAppVersionCode(Context context) {

        PackageInfo packInfo = null;
        try {
            packInfo = context.getApplicationContext().getPackageManager()
                    .getPackageInfo(context.getApplicationContext().getPackageName(), 0);
        } catch (PackageManager.NameNotFoundException e) {
            WarpUtils.log(e.getMessage());
        }
        return packInfo != null ? packInfo.versionCode : 0;
    }

    public static boolean isInDevelopmentMode(Context context) {

        boolean debuggable = false;
        PackageManager pm = context.getPackageManager();
        try {
            ApplicationInfo appInfo = pm.getApplicationInfo(
                    context.getPackageName(), 0);
            debuggable = (0 != (appInfo.flags &= ApplicationInfo.FLAG_DEBUGGABLE));
        } catch (PackageManager.NameNotFoundException e) {
            WarpUtils.log(e.getMessage());
        }
        return debuggable;
    }

    public static String getCustomPushServiceClassName(Context context) {

        String name = null;
        try {
            PackageInfo packInfo = context.getPackageManager().getPackageInfo(context.getPackageName(),
                    PackageManager.GET_SERVICES);
            if (packInfo != null && packInfo.services != null && packInfo.services.length > 0) {

                for (ServiceInfo sInfo : packInfo.services) {

                    if (!TextUtils.isEmpty(sInfo.nonLocalizedLabel) && WarpConstants.METADATA_KEY_CUSTOM_GCM_INTENT_SERVICE
                            .equals(sInfo.nonLocalizedLabel)) {
                        name = sInfo.name;
                        break;
                    }
                }
            }

        } catch (PackageManager.NameNotFoundException e) {
            WarpUtils.log(e.getMessage());
        }
        return name;
    }

    public static String getAppUuid(Context context) {
        return getWarplyProperty(context, KEY_UUID);
    }

    public static String getLanguage(Context context) {
        return getWarplyProperty(context, KEY_LANGUAGE);
    }

    public static String getMerchantId(Context context) {
        return String.valueOf(getWarplyProperty(context, KEY_MERCHANT_ID));
    }

    public static String getLoginType(Context context) {
        return getWarplyProperty(context, KEY_LOGIN_TYPE);
    }

    public static String getDlUrlScheme(Context context) {
        return getWarplyProperty(context, KEY_DL_URL_SCHEME);
    }

    public static boolean isDebugMode(Context context) {
        return Boolean.parseBoolean(getWarplyProperty(context, KEY_DEBUG));
    }

    public static String getBaseUrl(Context context) {
        return getWarplyProperty(context, KEY_BASE_URL);
    }

    public static String getVerifyUrl(Context context) {
        return getWarplyProperty(context, KEY_VERIFY_URL);
    }

    public static boolean isSendPackages(Context context) {
        return Boolean.parseBoolean(getWarplyProperty(context, KEY_SEND_PACKAGES));
    }

    public static String getWebActionListenerClassName(Context context) {
        return getWarplyProperty(context, KEY_WEB_ACTION_HANDLER);
    }

    public static int getProgressColor(Context context) {

        String resName = getWarplyProperty(context, KEY_PROGRESS_COLOR);
        if (!TextUtils.isEmpty(resName)) {
            int resId = context.getResources().getIdentifier(resName, "color", context.getPackageName());
            if (resId > 0) {
                return ContextCompat.getColor(context, resId);
            }
        }
        int color = Color.parseColor("#384047");
        int primaryColor = getPrimaryColor(context, color);
        return primaryColor != Color.WHITE && primaryColor != Color.TRANSPARENT ? primaryColor : color;
    }

    @NonNull
    public static Drawable getProgressDrawable(Context context) {

        String resName = getWarplyProperty(context, KEY_PROGRESS_DRAWABLE);
        if (!TextUtils.isEmpty(resName)) {
            int resId = context.getResources().getIdentifier(resName, "drawable", context.getPackageName());
            if (resId > 0) {
                return ContextCompat.getDrawable(context, resId);
            }
        }
        return getLauncherIcon(context, new BitmapDrawable(context.getResources(), WarpUtils.getWarplyLogo()));
    }

    public static int getPushColor(Context context) {

        String resName = getWarplyProperty(context, KEY_PUSH_COLOR);
        if (!TextUtils.isEmpty(resName)) {
            int resId = context.getResources().getIdentifier(resName, "color", context.getPackageName());
            if (resId > 0) {
                return ContextCompat.getColor(context, resId);
            }
        }
        int color = Color.parseColor("#384047");
        int primaryColor = getPrimaryColor(context, color);
        return primaryColor != Color.WHITE && primaryColor != Color.TRANSPARENT ? primaryColor : color;
    }

    @DrawableRes
    public static int getPushIconResId(Context context) {

        String resName = getWarplyProperty(context, KEY_PUSH_ICON);
        if (!TextUtils.isEmpty(resName)) {
            int resId = context.getResources().getIdentifier(resName, "drawable", context.getPackageName());
            if (resId > 0) {
                return resId;
            }
        }
        return android.R.drawable.btn_star;
    }

    public static Uri getPushSoundUri(Context context) {

        String resName = getWarplyProperty(context, KEY_PUSH_SOUND);
        if (!TextUtils.isEmpty(resName)) {
            int resId = context.getResources().getIdentifier(resName, "raw", context.getPackageName());
            if (resId > 0) {
                return Uri.parse("android.resource://" + context.getPackageName() + "/raw/" + resId);
            }
        }
        return Settings.System.DEFAULT_NOTIFICATION_URI;
    }
}
