/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.utils.managers;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import org.json.JSONException;
import org.json.JSONObject;

import ly.warp.sdk.Warply;
import ly.warp.sdk.io.callbacks.CallbackReceiver;
import ly.warp.sdk.io.models.CampaignList;
import ly.warp.sdk.io.models.PushCampaign;
import ly.warp.sdk.io.request.WarplyInboxRequest;
import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.constants.WarpConstants;

import static ly.warp.sdk.Warply.INSTANCE;
import static ly.warp.sdk.Warply.getWarplyContext;

public class WarplyAnalyticsManager {

    /**
     * Default method for logging events with low priority happening in a page
     *
     * @param eventPage The name of the page that the event happened
     * @param eventId   The identifier of the event that happened, can contain letters
     * @param metadata  JSON object, containing metadata for this event
     */
    public static void logEvent(@Nullable String eventPage, String eventId, @Nullable JSONObject metadata) {
        sendEvent(eventPage, eventId, metadata, false);
    }

    public static void logTrackersEvent(Context context, String eventType, String eventName) {
        if (WarpUtils.getTrackersEnabled(context)) {
//            Warply.getInitializer(context).init();
            sendEvent(context, null, eventType.concat(":").concat(eventName), null, false);
        }
    }

    /**
     * Log screen events so after we can check if there are in app campaigns to show
     * (per screen or all screens). After we send the event with priority we make
     * a get_inbox request with extra parameters.
     *
     * @param eventPage
     * @param eventId
     * @param metadata
     * @param force
     * @param context
     */
    public static void logEventInApp(Context context, @Nullable String eventPage,
                                     @NonNull String eventId, @Nullable JSONObject metadata,
                                     boolean force) {
        sendEventInApp(context, eventPage, eventId, metadata, force);
    }

    /**
     * Default method for logging events with high priority happening in a page
     *
     * @param eventPage The name of the page that the event happened
     * @param eventId   The identifier of the event that happened, can contain letters
     * @param metadata  JSON object, containing metadata for this event
     */
    public static void logUrgentEvent(String eventPage, String eventId,
                                      JSONObject metadata) {
        sendEvent(eventPage, eventId, metadata, true);
    }

    public static void logAppLifeCycleEvent() {
        sendLifecycleEvent(WarpConstants.MICROAPP_APPLICATION_PAGE_ID, "session");
    }

    public static void logAppLifeCycleEvent(boolean foreground) {
        sendLifecycleEvent(WarpConstants.MICROAPP_APPLICATION_PAGE_ID, "session"
                /*foreground ? "NB_WillEnterForeground" : "NB_DidEnterBackground"*/);
    }

    public static void logUserReceivedPush(String sessionUUID) {
        JSONObject metadata = new JSONObject();
        try {
            metadata.putOpt("session_uuid", sessionUUID);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        sendEvent(WarpConstants.MICROAPP_APPLICATION_PAGE_ID,
                "NB_PushReceived", metadata, true);
    }

    public static void logUserReceivedPush(PushCampaign pc) {
        JSONObject metadata = new JSONObject();
        try {
            metadata.putOpt("session_uuid", pc.getSessionUUID());
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        sendPushReceivedEvent(WarpConstants.MICROAPP_APPLICATION_PAGE_ID,
                "NB_PushReceived", metadata, false);
    }

//    public static void logUserEngagedPush(JSONObject pushSessionUUID) {
//        sendEvent(WarpConstants.MICROAPP_APPLICATION_PAGE_ID, "NB_PushAck",
//                pushSessionUUID, true);
//    }

    public static void logUserEngagedPush(JSONObject pushSessionUUID) {
        sendPushAckEvent(WarpConstants.MICROAPP_APPLICATION_PAGE_ID, "NB_PushAck",
                pushSessionUUID, false);
    }

    public static void logUserReceivedPush(JSONObject pushSessionUUID) {
        sendEvent(WarpConstants.MICROAPP_APPLICATION_PAGE_ID,
                "NB_PushReceived", pushSessionUUID, true);
    }

    /**
     * Send an event to the Warply server
     *
     * @param eventPage Event page ID
     * @param eventId   Event ID
     * @param metadata  Event metadata
     * @param force     If this event should be forced or not
     */
    private static void sendEvent(@Nullable String eventPage, String eventId, @Nullable JSONObject metadata, boolean force) {
        boolean isInAppAnalyticsMicroAppActive = WarplyServerPreferencesManager.isMicroAppActive(WarpConstants.MicroApp.CUSTOM_ANALYTICS);
        if (isInAppAnalyticsMicroAppActive)
            try {
                JSONObject jObj = new JSONObject();
                if (eventPage != null) {
                    jObj.putOpt("page_id", eventPage);
                }
                jObj.putOpt("event_id", eventId);
                jObj.putOpt("time_submitted", System.currentTimeMillis() / 1000);
                if (metadata != null) {
                    jObj.putOpt("action_metadata", metadata);
                }
                Warply.postMicroappData(WarpConstants.MICROAPP_INAPP_ANALYTICS, jObj, force);
            } catch (JSONException e) {
                if (WarpConstants.DEBUG) {
                    e.printStackTrace();
                }
            }
        else
            WarpUtils.log("analytics micro-app is not active");
    }

    private static void sendEvent(Context context, @Nullable String eventPage, String eventId, @Nullable JSONObject metadata, boolean force) {
        boolean isInAppAnalyticsMicroAppActive = WarplyServerPreferencesManager.isMicroAppActive(context, WarpConstants.MicroApp.CUSTOM_ANALYTICS);
        if (isInAppAnalyticsMicroAppActive)
            try {
                JSONObject jObj = new JSONObject();
                if (eventPage != null) {
                    jObj.putOpt("page_id", eventPage);
                }
                jObj.putOpt("event_id", eventId);
                jObj.putOpt("time_submitted", System.currentTimeMillis() / 1000);
                if (metadata != null) {
                    jObj.putOpt("action_metadata", metadata);
                }
                Warply.postMicroappData(context, WarpConstants.MICROAPP_INAPP_ANALYTICS, jObj, force);
            } catch (JSONException e) {
                if (WarpConstants.DEBUG) {
                    e.printStackTrace();
                }
            }
        else
            WarpUtils.log("analytics micro-app is not active");
    }

    /**
     * Send an in app event to the Warply server and show a campaign per screen
     *
     * @param eventPage Event page ID
     * @param eventId   Event ID
     * @param metadata  Event metadata
     * @param force     If this event should be forced or not
     */
    private static void sendEventInApp(final Context context, @Nullable String eventPage,
                                       @NonNull final String eventId, @Nullable JSONObject metadata, boolean force) {
        boolean isInAppAnalyticsMicroAppActive = WarplyServerPreferencesManager.isMicroAppActive(WarpConstants.MicroApp.CUSTOM_ANALYTICS);
        if (isInAppAnalyticsMicroAppActive) {
            try {
                JSONObject jObj = new JSONObject();
                if (eventPage != null) {
                    jObj.putOpt("page_id", eventPage);
                }
                jObj.putOpt("event_id", eventId);
                jObj.putOpt("time_submitted", System.currentTimeMillis() / 1000);
                if (metadata != null) {
                    jObj.putOpt("action_metadata", metadata);
                }
                Warply.postMicroappData(WarpConstants.MICROAPP_INAPP_ANALYTICS, jObj, force);
            } catch (JSONException e) {
                if (WarpConstants.DEBUG) {
                    e.printStackTrace();
                }
            }
        } else
            WarpUtils.log("analytics micro-app is not active");


        // TODO: check for the interval else INSTANCE.getInAppCampaigns()
//        if() {
        WarplyInboxRequest inboxFilterRequest = new WarplyInboxRequest().setUseCache(false);

        Warply.getInboxInApp(inboxFilterRequest, new CallbackReceiver<CampaignList>() {
            @Override
            public void onSuccess(CampaignList result) {
                INSTANCE.mInAppCampaigns = result;
                Warply.showInAppCampaign(context, eventId);
            }

            @Override
            public void onFailure(int errorCode) {
            }
        });
//        } else {
//Warply.showInAppCampaign(context, eventId);
//        }
    }

    private static void sendPushReceivedEvent(@Nullable String eventPage, String eventId, @Nullable JSONObject metadata, boolean force) {
        boolean isInAppAnalyticsMicroAppActive = WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.CUSTOM_ANALYTICS);
        if (isInAppAnalyticsMicroAppActive)
            try {
                JSONObject jObj = new JSONObject();
                if (eventPage != null) {
                    jObj.putOpt("page_id", eventPage);
                }
                jObj.putOpt("event_id", eventId);
                jObj.putOpt("time_submitted", System.currentTimeMillis() / 1000);
                if (metadata != null) {
                    jObj.putOpt("action_metadata", metadata);
                }
                Warply.postMicroappPush(WarpConstants.MICROAPP_INAPP_ANALYTICS, jObj, force);
            } catch (JSONException e) {
                if (WarpConstants.DEBUG) {
                    e.printStackTrace();
                }
            }
        else
            WarpUtils.log("analytics micro-app is not active");
    }

    private static void sendPushAckEvent(@Nullable String eventPage, String eventId, @Nullable JSONObject metadata, boolean force) {
        boolean isInAppAnalyticsMicroAppActive = WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.CUSTOM_ANALYTICS);
        if (isInAppAnalyticsMicroAppActive)
            try {
                JSONObject jObj = new JSONObject();
                if (eventPage != null) {
                    jObj.putOpt("page_id", eventPage);
                }
                jObj.putOpt("event_id", eventId);
                jObj.putOpt("time_submitted", System.currentTimeMillis() / 1000);
                if (metadata != null) {
                    jObj.putOpt("action_metadata", metadata);
                }
                Warply.postMicroappPushAck(WarpConstants.MICROAPP_INAPP_ANALYTICS, jObj, force);
            } catch (JSONException e) {
                if (WarpConstants.DEBUG) {
                    e.printStackTrace();
                }
            }
        else
            WarpUtils.log("analytics micro-app is not active");
    }

    /**
     * Send a lifecycle event to the Warply server
     *
     * @param eventPage Event page ID
     * @param eventId   Event ID
     */
    private static void sendLifecycleEvent(String eventPage, String eventId) {
        if (getWarplyContext() == null)
            return;
        boolean isLifecycleAnalyticsMicroAppActive = WarpUtils.getIsLIFECYCLEANALYTICSENABLED(getWarplyContext());
        if (!isLifecycleAnalyticsMicroAppActive) {
            WarpUtils.log("session analytics micro-app is not active");
            return;
        }
        sendEvent(eventPage, eventId, null, true);
    }
}
