package ly.warp.sdk.utils.managers;

import android.content.Context;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;

import ly.warp.sdk.io.callbacks.WarplyHealthCallback;

/**
 * Created by Panagiotis Triantafyllou on 27/Απρ/2022.
 *
 * @guide http://www.gadgetsaint.com/android/create-pedometer-step-counter-android/#.Weg2KmiCyM8
 */
public class WarplyHealthManager implements SensorEventListener {

    // ===========================================================
    // Constants
    // ===========================================================

    // ===========================================================
    // Fields
    // ===========================================================

    private long mSteps = 0;
    private static SensorManager mSensorManager;
    private static Sensor mSensor;
    private static WarplyHealthManager mWarplyHealthManagerInstance;
    private static final int ACCEL_RING_SIZE = 50;
    private static final int VEL_RING_SIZE = 10;
    // change this threshold according to your sensitivity preferences
    private static final float STEP_THRESHOLD = 20f;
    private static final int STEP_DELAY_NS = 250000000;
    private int accelRingCounter = 0;
    private float[] accelRingX = new float[ACCEL_RING_SIZE];
    private float[] accelRingY = new float[ACCEL_RING_SIZE];
    private float[] accelRingZ = new float[ACCEL_RING_SIZE];
    private int velRingCounter = 0;
    private float[] velRing = new float[VEL_RING_SIZE];
    private long lastStepTimeNs = 0;
    private float oldVelocityEstimate = 0;

    //TODO: if we want to send the steps back to an activity/fragment/service etc
    private static WarplyHealthCallback mHealthCallback;

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public void onSensorChanged(SensorEvent sensorEvent) {
        if (sensorEvent.sensor.getType() == Sensor.TYPE_ACCELEROMETER) {
            updateSteps(
                    sensorEvent.timestamp,
                    sensorEvent.values[0],
                    sensorEvent.values[1],
                    sensorEvent.values[2]);
        }
    }

    @Override
    public void onAccuracyChanged(Sensor sensor, int i) {

    }

    // ===========================================================
    // Methods
    // ===========================================================

    public static WarplyHealthManager getInstance(Context context) {
        if (mWarplyHealthManagerInstance == null) {
            mWarplyHealthManagerInstance = new WarplyHealthManager();
            mSensorManager = (SensorManager) context.getSystemService(Context.SENSOR_SERVICE);
            mSensor = mSensorManager.getDefaultSensor(Sensor.TYPE_ACCELEROMETER);
        }
        return mWarplyHealthManagerInstance;
    }

    public static WarplyHealthManager getInstance(Context context, WarplyHealthCallback callback) {
        if (mWarplyHealthManagerInstance == null) {
            mWarplyHealthManagerInstance = new WarplyHealthManager();
            mSensorManager = (SensorManager) context.getSystemService(Context.SENSOR_SERVICE);
            mSensor = mSensorManager.getDefaultSensor(Sensor.TYPE_ACCELEROMETER);
            mHealthCallback = callback;
        }
        return mWarplyHealthManagerInstance;
    }

    public void registerStepSensor() {
        mSensorManager.registerListener(mWarplyHealthManagerInstance, mSensor, SensorManager.SENSOR_DELAY_GAME);
    }

    public void unregisterStepSensor() {
        mSensorManager.unregisterListener(mWarplyHealthManagerInstance);
    }

    private void updateSteps(long timeNs, float x, float y, float z) {
        float[] currentAccel = new float[3];
        currentAccel[0] = x;
        currentAccel[1] = y;
        currentAccel[2] = z;

        // First step is to update our guess of where the global z vector is.
        accelRingCounter++;
        accelRingX[accelRingCounter % ACCEL_RING_SIZE] = currentAccel[0];
        accelRingY[accelRingCounter % ACCEL_RING_SIZE] = currentAccel[1];
        accelRingZ[accelRingCounter % ACCEL_RING_SIZE] = currentAccel[2];

        float[] worldZ = new float[3];
        worldZ[0] = sum(accelRingX) / Math.min(accelRingCounter, ACCEL_RING_SIZE);
        worldZ[1] = sum(accelRingY) / Math.min(accelRingCounter, ACCEL_RING_SIZE);
        worldZ[2] = sum(accelRingZ) / Math.min(accelRingCounter, ACCEL_RING_SIZE);

        float normalization_factor = norm(worldZ);

        worldZ[0] = worldZ[0] / normalization_factor;
        worldZ[1] = worldZ[1] / normalization_factor;
        worldZ[2] = worldZ[2] / normalization_factor;

        float currentZ = dot(worldZ, currentAccel) - normalization_factor;
        velRingCounter++;
        velRing[velRingCounter % VEL_RING_SIZE] = currentZ;

        float velocityEstimate = sum(velRing);

        if (velocityEstimate > STEP_THRESHOLD
                && oldVelocityEstimate <= STEP_THRESHOLD
                && (timeNs - lastStepTimeNs > STEP_DELAY_NS)) {
            mSteps++;
            // if we want to send the steps back to an activity/fragment/service etc
            mHealthCallback.onStepCount(mSteps);
            lastStepTimeNs = timeNs;
        }
        oldVelocityEstimate = velocityEstimate;
    }

    private float sum(float[] array) {
        float retval = 0;
        for (float v : array) {
            retval += v;
        }
        return retval;
    }

    private float[] cross(float[] arrayA, float[] arrayB) {
        float[] retArray = new float[3];
        retArray[0] = arrayA[1] * arrayB[2] - arrayA[2] * arrayB[1];
        retArray[1] = arrayA[2] * arrayB[0] - arrayA[0] * arrayB[2];
        retArray[2] = arrayA[0] * arrayB[1] - arrayA[1] * arrayB[0];
        return retArray;
    }

    private float norm(float[] array) {
        float retval = 0;
        for (float v : array) {
            retval += v * v;
        }
        return (float) Math.sqrt(retval);
    }

    private float dot(float[] a, float[] b) {
        return a[0] * b[0] + a[1] * b[1] + a[2] * b[2];
    }

    private float[] normalize(float[] a) {
        float[] retval = new float[a.length];
        float norm = norm(a);
        for (int i = 0; i < a.length; i++) {
            retval[i] = a[i] / norm;
        }
        return retval;
    }

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================
}
