/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.utils.managers;

import android.app.PendingIntent;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.location.Criteria;
import android.location.LocationManager;
import android.os.Build;

import androidx.work.WorkManager;

import java.util.List;

import ly.warp.sdk.BuildConfig;
import ly.warp.sdk.Warply;
import ly.warp.sdk.dexter.listener.DexterError;
import ly.warp.sdk.dexter.listener.PermissionDeniedResponse;
import ly.warp.sdk.receivers.ConnectivityChangedReceiver;
import ly.warp.sdk.receivers.LocationChangedReceiver;
import ly.warp.sdk.services.UpdateUserLocationService;
import ly.warp.sdk.utils.PermissionsUtil;
import ly.warp.sdk.utils.WarplyPreferences;

public class WarplyLocationManager {

    private static LocationManager locationManager;
    private static String locationProvider;
    public static Context waprplyContext;

    public static void instantiateLocationManager() {

        waprplyContext = Warply.getWarplyContext();

        /**
         * Acquire a reference to the system Location Manager LocationManager is
         * the main class through which application can access location services
         * on Android
         */
        locationManager = (LocationManager) Warply.getWarplyContext()
                .getSystemService(Context.LOCATION_SERVICE);

        locationProvider = LocationManager.PASSIVE_PROVIDER;

    }

    /**
     * This method allows Location Manager to report current location back to
     * the Warply service.
     */
    public static void startDefaultReportingLocation(Context act) {

        instantiateLocationManager();

        enableConnectivityChangedReceiver();
        enableLocationChangedReceiver();

        // Log.i("locationProvider", locationProvider);

        if (locationProvider != null) {

            final long defaultMinTime = 60000;
            final float defaultMinDistance = 100;

            Intent activeIntent = new Intent(waprplyContext, LocationChangedReceiver.class);
            final PendingIntent locationListenerPendingIntent = PendingIntent
                    .getBroadcast(waprplyContext, 0, activeIntent,
                            PendingIntent.FLAG_UPDATE_CURRENT);

            // Register the listener with the Location Manager to receive
            // location updates
            PermissionsUtil.PermissionCallback callback = new PermissionsUtil.PermissionCallback() {
                @Override
                public void onError(DexterError error) {
                    onPermissionDenied(null);
                }

                @Override
                public void onPermissionDenied(List<PermissionDeniedResponse> denied) {
                    if (denied != null && denied.size() < 2)//1 location permission accepted
                        onPermissionsGranted();
                }

                @Override
                public void onPermissionsGranted() {
                    /*
                     * second and third parameters control the frequency at which
                     * listener receives updates (when onLocationChanged() callback will
                     * be invoked). The second is the minimum time interval between
                     * notifications and the third is the minimum change in distance
                     * between notifications&#xfffd;setting both to zero requests location
                     * notifications as frequently as possible.
                     */
                    try {
                        locationManager.requestLocationUpdates(locationProvider,
                                defaultMinTime, defaultMinDistance,
                                locationListenerPendingIntent);
                    } catch (SecurityException e) {
                        if (BuildConfig.DEBUG) {
                            e.printStackTrace();
                        }
                    }
                }
            };

            try {
                new PermissionsUtil(
                        act,
                        callback,
                        PermissionsUtil.PERMISSION_LOCATION_COARSE,
                        PermissionsUtil.PERMISSION_LOCATION_FINE
                ).requestPermissions();
            } catch (Exception e) {
                if (BuildConfig.DEBUG)
                    e.printStackTrace();
            }
        }

//         MicroAppIndexes geofencingIndex = MicroAppIndexes.GEOFENCING_INDEX;
//         boolean isGeofencingMicroAppEnable = WarpClient.MICROAPP_STATUSES_MAP.get(WarpConstants.MICROAPPS[geofencingIndex.ordinal()]);
//
//         if (isGeofencingMicroAppEnable) {
//             enableConnectivityChangedReceiver();
//             enableLocationChangedReceiver();
//
//             if(locationProvider != null){
//                 long defaultMinTime = 60000;
//                 float defaultMinDistance = 100;
//
//                 Intent activeIntent = new Intent(waprplyContext,
//                 LocationChangedReceiver.class);
//                 PendingIntent locationListenerPendingIntent =
//                 PendingIntent.getBroadcast(waprplyContext, 0, activeIntent,
//                 PendingIntent.FLAG_UPDATE_CURRENT);
//
//                 // Register the listener with the Location Manager to receive
//                 //location updates
//                 /**
//                 * second and third parameters control the frequency at which listener
//                 * receives updates (when onLocationChanged() callback will be
//                 invoked).
//                 * The second is the minimum time interval between notifications and
//                 * the third is the minimum change in distance between
//                 notifications&#xfffd;setting
//                 * both to zero requests location notifications as frequently as
//                 possible.
//                 */
//                 locationManager.requestLocationUpdates(locationProvider, defaultMinTime, defaultMinDistance, locationListenerPendingIntent);
//            }
//         } else {
//             stopReportingLocation(waprplyContext);
//             WarpUtils.log("geofencing micro-app is not active");
//         }

    }

    /**
     * This method disallows Location Manager to report current location back to
     * the Warply service.
     */
    public static void stopReportingLocation(Context wContext) {

        if (locationManager != null) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
                locationManager.removeUpdates(PendingIntent.getBroadcast(Warply.getWarplyContext(), 0,
                        new Intent(Warply.getWarplyContext(), LocationChangedReceiver.class),
                        PendingIntent.FLAG_UPDATE_CURRENT | PendingIntent.FLAG_MUTABLE));
            } else {
                locationManager.removeUpdates(PendingIntent.getBroadcast(Warply.getWarplyContext(), 0,
                        new Intent(Warply.getWarplyContext(), LocationChangedReceiver.class),
                        PendingIntent.FLAG_UPDATE_CURRENT));
            }
        }

        disableLocationChangedReceiver(wContext);
        disableUpdateLocationService(wContext);
    }

    public static void startReportingLocation(Context wContext, final String newProvider, final long minTime, final float minDistance) {

        if ((new WarplyPreferences(wContext).getLocationChangedReceiverStatus()).equals("off"))
            enableLocationChangedReceiver();

        if (wContext != null) {
            final PendingIntent locationListenerPendingIntent;
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
                locationListenerPendingIntent = PendingIntent.getBroadcast(wContext, 0,
                        new Intent(wContext, LocationChangedReceiver.class),
                        PendingIntent.FLAG_UPDATE_CURRENT | PendingIntent.FLAG_MUTABLE);
            } else {
                locationListenerPendingIntent = PendingIntent.getBroadcast(wContext, 0,
                        new Intent(wContext, LocationChangedReceiver.class),
                        PendingIntent.FLAG_UPDATE_CURRENT);
            }

            final LocationManager lm = (LocationManager) wContext.getSystemService(Context.LOCATION_SERVICE);

            if (lm != null) {
                lm.removeUpdates(locationListenerPendingIntent);

                try {
                    PermissionsUtil.PermissionCallback callback = new PermissionsUtil.PermissionCallback() {
                        @Override
                        public void onError(DexterError error) {
                            onPermissionDenied(null);
                        }

                        @Override
                        public void onPermissionDenied(List<PermissionDeniedResponse> denied) {
                            if (denied != null && denied.size() < 2)//1 location permission accepted
                                onPermissionsGranted();
                        }

                        @Override
                        public void onPermissionsGranted() {
                            try {
                                if (lm.isProviderEnabled(newProvider)) {
                                    lm.requestLocationUpdates(newProvider, minTime, minDistance,
                                            locationListenerPendingIntent);
                                } else {
                                    Criteria criteria = new Criteria();
                                    criteria.setAltitudeRequired(false);
                                    criteria.setBearingRequired(false);
                                    criteria.setSpeedRequired(false);
                                    String bestProvider;
                                    if (newProvider.equals(LocationManager.GPS_PROVIDER)) {
                                        criteria.setAccuracy(Criteria.ACCURACY_FINE);
                                    } else {
                                        criteria.setAccuracy(Criteria.ACCURACY_COARSE);
                                    }
                                    bestProvider = lm.getBestProvider(criteria, true);
                                    if (bestProvider == null || (criteria.getAccuracy() == Criteria.ACCURACY_COARSE
                                            && bestProvider.equals(LocationManager.GPS_PROVIDER))) {
                                        lm.requestLocationUpdates(LocationManager.PASSIVE_PROVIDER, minTime, minDistance, locationListenerPendingIntent);
                                    } else
                                        lm.requestLocationUpdates(bestProvider, minTime, minDistance, locationListenerPendingIntent);
                                }
                            } catch (SecurityException e) {
                                if (BuildConfig.DEBUG) {
                                    e.printStackTrace();
                                }
                            }
                        }
                    };

                    new PermissionsUtil(
                            wContext,
                            callback,
                            PermissionsUtil.PERMISSION_LOCATION_COARSE,
                            PermissionsUtil.PERMISSION_LOCATION_FINE
                    ).requestPermissions();
                } catch (SecurityException e) {
                    if (BuildConfig.DEBUG) {
                        e.printStackTrace();
                    }
                }
            }
        }

        // MicroAppIndexes geofencingIndex = MicroAppIndexes.GEOFENCING_INDEX;
        //
        // boolean isGeofencingMicroAppEnable = WarpClient.MICROAPP_STATUSES_MAP
        // .get(WarpConstants.MICROAPPS[geofencingIndex.ordinal()]);
        //
        // if (isGeofencingMicroAppEnable) {
        //
        // if( (new
        // WarplyPreferences(wContext).getLocationChangedReceiverStatus()).equals("off"))
        // enableLocationChangedReceiver();
        //
        // if(wContext != null){
        //
        // Intent activeIntent = new Intent(wContext,
        // LocationChangedReceiver.class);
        // PendingIntent locationListenerPendingIntent =
        // PendingIntent.getBroadcast(wContext, 0, activeIntent,
        // PendingIntent.FLAG_UPDATE_CURRENT);
        //
        // if(locationManager != null)
        // locationManager.removeUpdates(locationListenerPendingIntent);
        //
        // locationManager.requestLocationUpdates(newProvider, minTime,
        // minDistance, locationListenerPendingIntent);
        //
        // }
        //
        // } else {
        //
        // stopReportingLocation(wContext);
        // WarpUtils.log("geofencing micro-app is not active");
        //
        // }

    }

    public static void regulateLocationWithNewSettings(Context wContext, String newProvider, long minTime, float minDistance) {
        if (wContext != null) {
            if (newProvider.equals("off"))
                stopReportingLocation(wContext);
            else
                startReportingLocation(wContext, newProvider, minTime, minDistance);
        }
    }

    private static void enableLocationChangedReceiver() {
        Warply.getWarplyContext()
                .getPackageManager()
                .setComponentEnabledSetting(new ComponentName(Warply.getWarplyContext(), LocationChangedReceiver.class),
                        PackageManager.COMPONENT_ENABLED_STATE_ENABLED,
                        PackageManager.DONT_KILL_APP);

        new WarplyPreferences(Warply.getWarplyContext()).saveLocationChangedReceiverStatus("on");
    }

    private static void disableLocationChangedReceiver(Context wContext) {
        wContext.getPackageManager().setComponentEnabledSetting(new ComponentName(wContext, LocationChangedReceiver.class),
                PackageManager.COMPONENT_ENABLED_STATE_DISABLED,
                PackageManager.DONT_KILL_APP);
        new WarplyPreferences(wContext).saveLocationChangedReceiverStatus("off");
        WorkManager.getInstance(Warply.getWarplyContext()).cancelAllWorkByTag(UpdateUserLocationService.TAG);
    }

    private static void enableConnectivityChangedReceiver() {
        waprplyContext.getPackageManager()
                .setComponentEnabledSetting(new ComponentName(waprplyContext, ConnectivityChangedReceiver.class),
                        PackageManager.COMPONENT_ENABLED_STATE_ENABLED,
                        PackageManager.DONT_KILL_APP);

    }

    public static void disableConnectivityChangedReceiver(Context wContext) {
        wContext.getPackageManager()
                .setComponentEnabledSetting(new ComponentName(wContext, ConnectivityChangedReceiver.class),
                        PackageManager.COMPONENT_ENABLED_STATE_DISABLED,
                        PackageManager.DONT_KILL_APP);
    }

    private static void disableUpdateLocationService(Context wContext) {
        wContext.stopService(new Intent(wContext, UpdateUserLocationService.class));
        new WarplyPreferences(wContext).saveUpdateLocationServiceStatus("off");
    }

    public static LocationManager getLocationManager() {
        return locationManager;
    }
}
