/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package ly.warp.sdk.utils.managers;

import java.util.HashSet;

import ly.warp.sdk.utils.constants.WarpConstants;
import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.Warply;

import org.json.JSONException;
import org.json.JSONObject;

import android.app.Activity;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;

public class WarplySessionManager {
//    private final static HashSet<Activity> _activities = new HashSet<Activity>();
//    private static ConnectivityReceiver _receiver;

    /**
     * Must be called in <b>every</b> Activity's onCreate method, in order to
     * make sure that application Open state is going to be
     * persisted correctly. <br />
     *
     * @param activity The Activity that was started
     */
    public static void onCreateActivity(Activity activity) {
//        synchronized (_activities) {
            Warply.getInitializer(activity).init();
//            _activities.add(activity);
//            if (_activities.size() == 1)
                onApplicationStart();
//        }
    }

    /**
     * Must be called in <b>every</b> Activity's onStart method, in order to
     * make sure that application Foreground / Background state is going to be
     * persisted correctly. <br />
     * Make sure you call {@link #onStopActivity(Activity)
     * Warply.onStopActivity} too in onStop. <br />
     * Typically something like WarplySessionManager.onStartActivity(this)
     * should be called.
     *
     * @param activity The Activity that was started
     */
    public static void onStartActivity(Activity activity) {
//        synchronized (_activities) {
            Warply.getInitializer(activity).init();
//            _activities.add(activity);
//            if (_activities.size() == 1)
                onApplicationEnterForeground();
//        }
    }

    /**
     * Must be called in <b>every</b> Activity's onStop method, in order to make
     * sure that application Foreground / Background state is going to be
     * persisted correctly. <br />
     * Make sure you call {@link #onStartActivity(Activity)
     * Warply.onStartActivity} too in onStart. <br />
     * Typically something like WarplySessionManager.onStopActivity(this) should
     * be called.
     *
     * @param activity The Activity that was stopped
     */
    public static void onStopActivity(Activity activity) {
//        synchronized (_activities) {
        Warply.getInitializer(activity).init();
//            _activities.remove(activity);
//            if (_activities.size() == 0)
                onApplicationEnterBackground();
//        }
    }

    private static void onApplicationStart() {
        Warply.onApplicationStarted();
    }

    private static void onApplicationEnterForeground() {
        Warply.onApplicationEnterForeground();
//        _receiver = new ConnectivityReceiver();
//        _receiver.startTracking();
    }

    private static void onApplicationEnterBackground() {
        Warply.onApplicationEnterBackground();
//        if (_receiver != null)
//            _receiver.stopTracking();
    }

    private static class ConnectivityReceiver extends BroadcastReceiver {

        private static final int TYPE_NOTHING = 0;
        private static final int TYPE_MOBILE = 1;
        private static final int TYPE_WIFI = 2;

        private int mType = 0;
        private long mStart = -1;
        private long mWifi = 0;
        private long mWWan = 0;

        @Override
        public void onReceive(Context context, Intent intent) {
            synchronized (this) {
                saveState();

                ConnectivityManager manager = (ConnectivityManager) context
                        .getSystemService(Context.CONNECTIVITY_SERVICE);
                NetworkInfo info = manager.getActiveNetworkInfo();
                if (info == null)
                    return;

                boolean isConneted = info.isConnectedOrConnecting();
                if (!isConneted)
                    return;

                int type = info.getType();
                switch (type) {
                    case ConnectivityManager.TYPE_MOBILE:
                    case ConnectivityManager.TYPE_MOBILE_DUN:
                    case ConnectivityManager.TYPE_MOBILE_HIPRI:
                    case ConnectivityManager.TYPE_MOBILE_MMS:
                    case ConnectivityManager.TYPE_MOBILE_SUPL:
                        onMobileConnectionChange();
                        break;
                    case ConnectivityManager.TYPE_ETHERNET:
                    case ConnectivityManager.TYPE_WIFI:
                    case ConnectivityManager.TYPE_WIMAX:
                        onWifiConnectionChange();
                        break;
                    default:
                        mType = TYPE_NOTHING;
                        break;
                }
            }
        }

        private void saveState() {
            switch (mType) {
                case TYPE_MOBILE:
                    mWWan += System.currentTimeMillis() - mStart;
                    break;
                case TYPE_WIFI:
                    mWifi += System.currentTimeMillis() - mStart;
                    break;
            }
            mType = TYPE_NOTHING;
        }

        private void onMobileConnectionChange() {
            mStart = System.currentTimeMillis();
            mType = TYPE_MOBILE;
        }

        private void onWifiConnectionChange() {
            mStart = System.currentTimeMillis();
            mType = TYPE_WIFI;
        }

        private void startTracking() {
            mStart = 0;
            mWifi = 0;
            mWWan = 0;
            mType = TYPE_NOTHING;
            Context context = Warply.getWarplyContext();
            IntentFilter filter = new IntentFilter();
            filter.addAction(ConnectivityManager.CONNECTIVITY_ACTION);
            context.registerReceiver(this, filter);
            onReceive(context, null);
        }

        private void stopTracking() {
            Context context = Warply.getWarplyContext();
            context.unregisterReceiver(this);
            saveState();

            long total = mWifi + mWWan;
            if (total <= 0)
                return;
            double wifiPerc = (double) mWifi / (double) total;
            double wWanPerc = (double) mWWan / (double) total;

            try {
                JSONObject json = new JSONObject();
                JSONObject usage = new JSONObject();
                JSONObject time = new JSONObject();

                usage.putOpt("wifi", wifiPerc);
                usage.putOpt("wwan", wWanPerc);

                time.putOpt("wifi", mWifi / 1000);
                time.putOpt("wwan", mWWan / 1000);

                json.putOpt("usage", usage);
                json.putOpt("time", time);
                if (Warply.isInitialized())
                    Warply.postMicroappData(WarpConstants.MICROAPP_DEVICE_INFO, json);
            } catch (JSONException e) {
                // Should never reach
                WarpUtils.warn("Unexpected JSON Exception", e);
            }
        }
    }
}
