/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.utils.managers;

import ly.warp.sdk.db.WarplyDBHelper;
import ly.warp.sdk.io.callbacks.CallbackReceiver;
import ly.warp.sdk.io.callbacks.SimpleCallbackReceiver;
import ly.warp.sdk.utils.constants.WarpConstants;
import ly.warp.sdk.utils.WarpJSONParser;
import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.Warply;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class WarplyUserManager {

    private static final String ACTION = "action";

    /**
     * @param userName data is data which will be matched with user web_id for
     *                 targeting purposes.
     */

    public static void sendName(String userName) {
        boolean isConsumerDataMicroAppActive = WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.CONSUMER);
        if (isConsumerDataMicroAppActive)
            sendLoginData("user_name", userName);
        else
            WarpUtils.log("consumer data micro-app is not active");
    }

    public static void sendEmail(String email) {
        boolean isConsumerDataMicroAppActive = WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.CONSUMER);
        if (isConsumerDataMicroAppActive)
            sendLoginData("user_email", email);
        else
            WarpUtils.log("consumer data micro-app is not active");
    }

    public static void sendMsisdn(String msisdn) {
        boolean isConsumerDataMicroAppActive = WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.CONSUMER);
        if (isConsumerDataMicroAppActive)
            sendLoginData("msisdn", msisdn);
        else
            WarpUtils.log("consumer data micro-app is not active");
    }

    /**
     * @param customData : data to for matching user with custom information other than
     *                   name, email or msisdn
     */
    public static void sendCustomData(JSONObject customData) {
        boolean isConsumerDataMicroAppActive = WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.CONSUMER);
        if (isConsumerDataMicroAppActive)
            sendConsumerDataWithJSON("custom_data", customData);
        else
            WarpUtils.log("consumer data micro-app is not active");
    }

    public static void getCustomData(final CallbackReceiver<JSONObject> receiver) {
        boolean isConsumerDataMicroAppActive = WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.CONSUMER);
        if (isConsumerDataMicroAppActive) {
            Warply.getMicroappData(WarpConstants.MICROAPP_CONSUMER_DATA,
                    new CallbackReceiver<JSONObject>() {
                        @Override
                        public void onSuccess(JSONObject result) {
                            JSONObject contextData = WarpJSONParser
                                    .getJObjectFromJSON(result, "context");
                            JSONObject consumerData = null;
                            if (contextData != null
                                    && contextData.has("consumer_data")) {
                                consumerData = WarpJSONParser
                                        .getJObjectFromJSON(
                                                contextData,
                                                WarpConstants.MICROAPP_CONSUMER_DATA);
                            }

                            if (consumerData != null
                                    && consumerData.has("custom_data")) {
                                JSONObject customData = WarpJSONParser
                                        .getJObjectFromJSON(consumerData,
                                                "custom_data");
                                receiver.onSuccess(customData);
                            } else {
                                receiver.onSuccess(null);
                            }
                        }

                        @Override
                        public void onFailure(int errorCode) {
                            receiver.onFailure(errorCode);
                            WarpUtils
                                    .log("get data from WarplyUserManager returned error: "
                                            + errorCode);
                        }
                    });
        }
    }

    public static void getName(final CallbackReceiver<String> receiver) {
        getLoginData(new CallbackReceiver<String>() {

            @Override
            public void onSuccess(String result) {
                receiver.onSuccess(result);
            }

            @Override
            public void onFailure(int errorCode) {
                receiver.onFailure(errorCode);

            }
        }, "user_name");
    }

    public static void getEmail(final CallbackReceiver<String> receiver) {
        getLoginData(new CallbackReceiver<String>() {

            @Override
            public void onSuccess(String result) {
                receiver.onSuccess(result);
            }

            @Override
            public void onFailure(int errorCode) {
                receiver.onFailure(errorCode);

            }
        }, "user_email");
    }

    private static void getLoginData(final CallbackReceiver<String> receiver,
                                     final String key) {
        boolean isConsumerDataMicroAppActive = WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.CONSUMER);
        if (isConsumerDataMicroAppActive) {
            Warply.getMicroappData(WarpConstants.MICROAPP_CONSUMER_DATA,
                    new CallbackReceiver<JSONObject>() {
                        @Override
                        public void onSuccess(JSONObject result) {
                            JSONObject contextData = WarpJSONParser
                                    .getJObjectFromJSON(result, "context");

                            JSONObject consumerData = null;
                            if (contextData != null
                                    && contextData.has("consumer_data")) {
                                consumerData = WarpJSONParser
                                        .getJObjectFromJSON(
                                                contextData,
                                                WarpConstants.MICROAPP_CONSUMER_DATA);
                            }

                            String loginValue = null;
                            if (consumerData != null && consumerData.has(key)) {
                                loginValue = WarpJSONParser.getStringFromJSON(
                                        consumerData, key, null);
                                receiver.onSuccess(loginValue);
                            } else {
                                receiver.onSuccess(null);
                            }
                        }

                        @Override
                        public void onFailure(int errorCode) {
                            receiver.onFailure(errorCode);
                            WarpUtils
                                    .log("get data from WarplyUserManager returned error: "
                                            + errorCode);
                        }
                    });
        }

    }

    public static void getMsisdn(final CallbackReceiver<String> receiver) {
        getLoginData(new CallbackReceiver<String>() {

            @Override
            public void onSuccess(String result) {
                receiver.onSuccess(result);
            }

            @Override
            public void onFailure(int errorCode) {
                receiver.onFailure(errorCode);

            }
        }, "msisdn");
    }

    private static void sendLoginData(String jsonLabel, String userData) {
        JSONObject jObj = new JSONObject();
        try {
            jObj.putOpt(jsonLabel, userData);
        } catch (JSONException e) {
            WarpUtils
                    .log("JSON Exception thrown while creating the object in sendData at WarplyUserManager",
                            e);
        }
        Warply.postMicroappData(WarpConstants.MICROAPP_LOGIN_DATA, jObj, true);
    }

    private static void sendConsumerDataWithJSON(String jsonLabel,
                                                 JSONObject jsonObject) {
        JSONObject jObj = new JSONObject();
        try {
            jObj.putOpt(jsonLabel, jsonObject);
        } catch (JSONException e) {
            WarpUtils
                    .log("JSON Exception thrown while creating the object in sendData at WarplyUserManager",
                            e);
        }
        Warply.postMicroappData(WarpConstants.MICROAPP_CONSUMER_DATA, jObj,
                true);
    }

    // ----- user tagging specific methods

    private static final String ACTION_ADD_TAGS = "add_tags";
    private static final String ACTION_REMOVE_TAGS = "remove_tags";
    private static final String ACTION_REMOVE_ALL_TAGS = "remove_all_tags";
    private static final String ACTION_REWRITE_TAGS = "rewrite_tags";
    private static final String TAGS = "tags";

    /**
     * Add a tag for this user. For adding multiple tags see
     * {@link #addTags(String[]) addTags} method.
     *
     * @param tag The tag to be added
     */
    public static void addTag(String tag) {
        addTags(new String[]{tag});
    }

    /**
     * Add multiple tags for this user
     *
     * @param tags The tags to be added
     */
    public static void addTags(String[] tags) {

        if (WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.USER_TAGGING)) {

            if (tags != null && tags.length > 0) {
                try {
                    JSONObject jObj = new JSONObject();
                    JSONArray jArray = new JSONArray();
                    jObj.putOpt(ACTION, ACTION_ADD_TAGS);
                    for (String tag : tags) {
                        jArray.put(tag);
                    }
                    jObj.putOpt(TAGS, jArray);

                    WarplyDBHelper.getInstance(Warply.getWarplyContext()).saveTags(tags);
                    Warply.postMicroappData(WarpConstants.MICROAPP_USER_TAG,
                            jObj, true);

                } catch (JSONException e) {
                    WarpUtils.log("JSON Exception thrown while creating the object" +
                            " in addTags at WarplyUserManager", e);
                }
            }

        } else {
            WarpUtils.log("user tagging micro-app is not active");
        }
    }

    /**
     * Remove one tag from this user. For removing multiple tags see
     * {@link #removeTags(String[]) removeTags} method
     *
     * @param tag The tag to be removed
     */
    public static void removeTag(String tag) {
        removeTags(new String[]{tag});
    }

    /**
     * Remove multiple tags from this user. For removing all tags see
     * {@link #removeAllTags() removeAllTags} method
     *
     * @param tags The tags to be removed
     */
    public static void removeTags(String[] tags) {

        if (WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.USER_TAGGING)) {

            if (tags != null && tags.length > 0) {
                try {
                    JSONObject jObj = new JSONObject();
                    JSONArray jArray = new JSONArray();
                    jObj.putOpt(ACTION, ACTION_REMOVE_TAGS);
                    for (String tag : tags) {
                        jArray.put(tag);
                    }
                    jObj.putOpt(TAGS, jArray);

                    WarplyDBHelper.getInstance(Warply.getWarplyContext()).removeTags(tags);
                    Warply.postMicroappData(WarpConstants.MICROAPP_USER_TAG, jObj,
                            true);
                } catch (JSONException e) {
                    WarpUtils.log("JSON Exception thrown while creating the object " +
                            "for in removeTags at WarplyUserManager", e);
                }
            }
        } else {
            WarpUtils.log("user tagging micro-app is not active");
        }
    }

    /**
     * Remove every tag from this user
     */
    public static void removeAllTags() {

        if (WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.USER_TAGGING)) {
            try {
                WarplyDBHelper.getInstance(Warply.getWarplyContext()).removeAllTags();
                Warply.postMicroappData(WarpConstants.MICROAPP_USER_TAG,
                        new JSONObject().putOpt(ACTION, ACTION_REMOVE_ALL_TAGS), true);
            } catch (JSONException e) {
                WarpUtils.log("JSON Exception thrown while creating the " +
                        "object for removeAllTags at WarplyUserManager", e);
            }
        } else {
            WarpUtils.log("user tagging micro-app is not active");
        }
    }

    /**
     * Resend all local tags to server
     */
    public static void rewriteTags() {

        if (WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.USER_TAGGING)) {

            String[] localTags = WarplyDBHelper.getInstance(Warply
                    .getWarplyContext()).getTags();

            if (localTags != null && localTags.length > 0) {
                try {
                    JSONObject jObj = new JSONObject();
                    jObj.putOpt(ACTION, ACTION_REWRITE_TAGS);
                    JSONArray jArray = new JSONArray();
                    for (String tag : localTags) {
                        jArray.put(tag);
                    }
                    jObj.putOpt(TAGS, jArray);
                    Warply.postMicroappData(WarpConstants.MICROAPP_USER_TAG, jObj, true);
                } catch (JSONException e) {
                    WarpUtils.log("JSON Exception thrown while creating the " +
                            "object for rewriteTags at WarplyUserManager", e);
                }
            }
        } else {
            WarpUtils.log("user tagging micro-app is not active");
        }
    }

    /**
     * @param callbackReceiver to get user's tags from service
     */
    public static void getTags(final SimpleCallbackReceiver<String[]> callbackReceiver) {

        if (WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.USER_TAGGING))

            Warply.getContext(new CallbackReceiver<JSONObject>() {
                @Override
                public void onSuccess(JSONObject result) {

                    // parse tags
                    String[] tags = null;
                    JSONObject json = WarpJSONParser.getJObjectFromJSON(result, "context");
                    if (json != null) {
                        JSONArray jArray = WarpJSONParser.getJSONArrayFromJson(json, TAGS);
                        if (jArray != null) {
                            tags = new String[jArray.length()];
                            for (int i = 0; i < jArray.length(); i++) {
                                tags[i] = WarpJSONParser.getStringFromJSONArray(jArray, i, null);
                            }
                        }
                    }
                    callbackReceiver.onSuccess(tags);
                }

                @Override
                public void onFailure(int errorCode) {
                    callbackReceiver.onFailure(errorCode);
                }
            });
        else {
            WarpUtils.log("user tagging micro-app is not active");
            callbackReceiver.onFailure(WarpConstants.RESULT_CODE_ERROR);
        }
    }

    public static void getTagsLocal(SimpleCallbackReceiver<String[]> callbackReceiver) {

        if (WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.USER_TAGGING))
            callbackReceiver.onSuccess(WarplyDBHelper.getInstance(Warply
                    .getWarplyContext()).getTags());
        else {
            WarpUtils.log("user tagging micro-app is not active");
            callbackReceiver.onFailure(WarpConstants.RESULT_CODE_ERROR);
        }
    }

    // ------ user session specific methods

    public static void sessionLogin(String authToken) {

        boolean isUserSessionMicroAppActive = WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.USER_SESSION);
        if (isUserSessionMicroAppActive)
            postUserSessionAction("login", authToken);
        else
            WarpUtils.log("user session micro-app is not active");
    }

    public static void sessionLogout(String authToken) {
        boolean isUserSessionMicroAppActive = WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.USER_SESSION);
        if (isUserSessionMicroAppActive)
            postUserSessionAction("logout", authToken);
        else
            WarpUtils.log("user session micro-app is not active");
    }

    private static void postUserSessionAction(final String action,
                                              final String authToken) {
        Warply.postReceiveMicroappData(WarpConstants.MICROAPP_USER_SESSION,
                createUserSessionJSON(action, authToken),
                new CallbackReceiver<JSONObject>() {
                    @Override
                    public void onSuccess(JSONObject result) {
                    }

                    @Override
                    public void onFailure(int errorCode) {
                        if (action.equals("login"))
                            postUserSessionAction("register", authToken);
                    }
                });
        Warply.postMicroappData(WarpConstants.MICROAPP_USER_SESSION,
                createUserSessionJSON(action, authToken), false);
    }

    private static JSONObject createUserSessionJSON(String action,
                                                    String authToken) {
        JSONObject actionObj = new JSONObject();
        JSONObject authTokenObj = new JSONObject();
        try {
            authTokenObj.putOpt("auth_token", authToken);
            actionObj.putOpt(action, authTokenObj);
        } catch (JSONException e) {
            WarpUtils
                    .log("JSON Exception thrown while creating the object in createUserSessionJSON at WarplyUserManager",
                            e);
        }
        return actionObj;
    }

    // ------- push notifications methods

    /**
     * Method that lets the user enable / disable Warply notifications for this device.
     *
     * @param enable   A boolean value. It should be true if the user wants to enable Warply push notifications and false if they want to disable them.
     * @param receiver A callback receiver with a Boolean success parameter that has the same value as the enable parameter of the method.
     */
    public static void setWarplyNotificationsEnabled(final boolean enable, final CallbackReceiver<Boolean> receiver) {
        JSONObject postJSON = new JSONObject();
        try {
            postJSON.put(WarpConstants.WARP_ENABLED, Boolean.valueOf(enable));
            Warply.postReceiveMicroappData(WarpConstants.MICROAPP_APPLICATION_DATA, postJSON, new CallbackReceiver<JSONObject>() {

                @Override
                public void onSuccess(JSONObject result) {
                    receiver.onSuccess(enable);
                }

                @Override
                public void onFailure(int errorCode) {
                    receiver.onFailure(errorCode);
                }
            });
        } catch (JSONException e) {
            e.printStackTrace();
        }

    }

    /**
     * Method that lets the user know if the specific device has Warply notifications enabled / disabled.
     *
     * @param receiver A callback receiver with a Boolean success parameter. If the success callback parameter
     *                 of the receiver is true it means that the device has Warply notifications enabled, otherwise disabled.
     */
    public static void getWarplyNotificationsEnabled(final CallbackReceiver<Boolean> receiver) {
        Warply.getMicroappData(WarpConstants.DEVICE_STATUS, new CallbackReceiver<JSONObject>() {

            @Override
            public void onSuccess(JSONObject response) {
                boolean isEnabled;
                try {
                    JSONObject context = response.getJSONObject("context");
                    JSONObject deviceStatus = context.getJSONObject(WarpConstants.DEVICE_STATUS);
                    isEnabled = deviceStatus.getBoolean(WarpConstants.WARP_ENABLED);
                    receiver.onSuccess(isEnabled);
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }

            @Override
            public void onFailure(int errorCode) {
                receiver.onFailure(errorCode);
            }
        });
    }


}
