package ly.warp.sdk.views;

import android.content.Context;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentActivity;
import androidx.fragment.app.FragmentManager;
import androidx.fragment.app.FragmentTransaction;
import android.text.TextUtils;

import ly.warp.sdk.BuildConfig;
import ly.warp.sdk.Warply;
import ly.warp.sdk.fragments.WarpViewFragment;
import ly.warp.sdk.io.callbacks.CallbackReceiver;
import ly.warp.sdk.io.callbacks.SimpleCallbackReceiver;
import ly.warp.sdk.io.models.CampaignList;
import ly.warp.sdk.io.request.WarplyInboxRequest;

public class CampaignViewLoader implements CallbackReceiver<CampaignList> {

    // ===========================================================
    // Constants
    // ===========================================================

    // ===========================================================
    // Fields
    // ===========================================================

    private DisplayBuilder mLastDisplayBuilder;
    private SimpleCallbackReceiver<CampaignList> mExternalListener;

    // ===========================================================
    // Constructors
    // ===========================================================

    public CampaignViewLoader(Context context) {
        Warply.getInitializer(context).init();
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public void onSuccess(CampaignList result) {

        if (result != null && result.size() > 0) {
            showWarplyFragment(mLastDisplayBuilder, result.get(0).getSessionUUID());
        }
        notifySuccessExternalListener(result);
    }

    @Override
    public void onFailure(int errorCode) {

        notifyFailureExternalListener(errorCode);
    }

    // ===========================================================
    // Methods
    // ===========================================================

    private void requestCampaignsIfNeed(DisplayBuilder builder, WarplyInboxRequest request, String sessionUuid) {

        if (request != null && TextUtils.isEmpty(sessionUuid)) {
            mLastDisplayBuilder = builder;
            Warply.getInbox(request, this);
        } else if (!TextUtils.isEmpty(sessionUuid)) {
            showWarplyFragment(builder, sessionUuid);
            notifySuccessExternalListener(null);
        }
    }

    private void showWarplyFragment(DisplayBuilder builder, String sessionUuid) {

        if (builder != null && builder.fragmentManager != null) {

            WarpViewFragment warpFragment = WarpViewFragment.newInstance(sessionUuid, builder.isUseProgress, builder.isShowCloseBtn);
            FragmentTransaction transaction = builder.fragmentManager.beginTransaction()
                    .replace(builder.containerId, warpFragment,
                            WarpViewFragment.class.getSimpleName())
                    .setTransition(FragmentTransaction.TRANSIT_FRAGMENT_OPEN);

            if (builder.isAddToBackStack) {
                transaction.addToBackStack(null);
            }

            try {
                transaction.commit();
            } catch (IllegalStateException e) {
                if (BuildConfig.DEBUG) {
                    e.printStackTrace();
                }
            }
        }
    }

    private void notifySuccessExternalListener(CampaignList campaigns) {

        if (mExternalListener != null) {
            mExternalListener.onSuccess(campaigns);
        }
    }

    private void notifyFailureExternalListener(int errorCode) {

        if (mExternalListener != null) {
            mExternalListener.onFailure(errorCode);
        }
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public void setCampaignsListener(SimpleCallbackReceiver<CampaignList> listener) {
        this.mExternalListener = listener;
    }

    public DisplayBuilder DisplayBuilder() {
        return new DisplayBuilder();
    }

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

    public class DisplayBuilder {

        private boolean isUseProgress = false;
        private boolean isShowCloseBtn = true;
        private boolean isAddToBackStack = true;

        private FragmentManager fragmentManager;
        private int containerId;

        public void display(FragmentActivity activity, WarplyInboxRequest request) {

            if (activity != null && !activity.isFinishing()) {
                display(activity, android.R.id.content, request);
            }
        }

        public void display(FragmentActivity activity, int containerId, WarplyInboxRequest request) {

            if (activity != null && !activity.isFinishing()) {
                this.fragmentManager = activity.getSupportFragmentManager();
                this.containerId = containerId;
                requestCampaignsIfNeed(this, request, null);
            }
        }

        public void display(Fragment parentFragment, int containerId, WarplyInboxRequest request) {

            if (parentFragment != null && parentFragment.getActivity() != null && !parentFragment.getActivity().isFinishing()) {
                this.fragmentManager = parentFragment.getChildFragmentManager();
                this.containerId = containerId;
                requestCampaignsIfNeed(this, request, null);
            }
        }

        public void display(FragmentActivity activity, String sessionUuid) {

            if (activity != null && !activity.isFinishing()) {
                display(activity, android.R.id.content, sessionUuid);
            }
        }

        public void display(FragmentActivity activity, int containerId, String sessionUuid) {

            if (activity != null && !activity.isFinishing()) {
                this.fragmentManager = activity.getSupportFragmentManager();
                this.containerId = containerId;
                requestCampaignsIfNeed(this, null, sessionUuid);
            }
        }

        public void display(Fragment parentFragment, int containerId, String sessionUuid) {

            if (parentFragment != null && parentFragment.getActivity() != null && !parentFragment.getActivity().isFinishing()) {
                this.fragmentManager = parentFragment.getChildFragmentManager();
                this.containerId = containerId;
                requestCampaignsIfNeed(this, null, sessionUuid);
            }
        }

        public DisplayBuilder setUseProgress(boolean use) {
            this.isUseProgress = use;
            return this;
        }

        public DisplayBuilder setShowCloseBtn(boolean show) {
            this.isShowCloseBtn = show;
            return this;
        }

        public DisplayBuilder setAddToBackStack(boolean add) {
            this.isAddToBackStack = add;
            return this;
        }

    }
}
