/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.views;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.ActivityManager;
import android.app.AlertDialog;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.IntentSender;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.location.LocationManager;
import android.net.Uri;
import android.net.http.SslError;
import android.os.Build;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.Log;
import android.webkit.GeolocationPermissions.Callback;
import android.webkit.JavascriptInterface;
import android.webkit.SslErrorHandler;
import android.webkit.WebChromeClient;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;
import androidx.lifecycle.DefaultLifecycleObserver;
import androidx.lifecycle.LifecycleOwner;
import androidx.lifecycle.ProcessLifecycleOwner;

import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.common.api.PendingResult;
import com.google.android.gms.common.api.Status;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationServices;
import com.google.android.gms.location.LocationSettingsRequest;
import com.google.android.gms.location.LocationSettingsResult;
import com.google.android.gms.location.LocationSettingsStatusCodes;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import ly.warp.sdk.R;
import ly.warp.sdk.Warply;
import ly.warp.sdk.activities.WarpViewActivity;
import ly.warp.sdk.db.WarplyDBHelper;
import ly.warp.sdk.io.models.CouponEventModel;
import ly.warp.sdk.io.models.LoyaltySDKFirebaseEventModel;
import ly.warp.sdk.io.models.QuestionnaireEventModel;
import ly.warp.sdk.io.models.WarplyCCMSEnabledModel;
import ly.warp.sdk.io.models.WarplyPacingCardEventModel;
import ly.warp.sdk.io.models.WarplyPacingEventModel;
import ly.warp.sdk.io.models.WarplyUnifiedActivatedEventModel;
import ly.warp.sdk.services.WarplyHealthService;
import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.WarplyManagerHelper;
import ly.warp.sdk.utils.WarplyProperty;
import ly.warp.sdk.utils.WarplyUrlHandler;
import ly.warp.sdk.utils.constants.WarpConstants;
import ly.warp.sdk.utils.managers.WarplyAnalyticsManager;
import ly.warp.sdk.utils.managers.WarplyEventBusManager;

public class WarpView extends WebView implements DefaultLifecycleObserver {

    private static final String URL_ACTION_TEL = "tel";
    private static final String URL_ACTION_SMS = "sms";
    private static final String URL_ACTION_RSMS = "rsms";
    private static final String URL_ACTION_MAIL = "mailto";
    private static final String URL_ACTION_MARKET = "market";
    private static final String URL_ACTION_INTENT = "intent";
    private static final String URL_ACTION_MARKET_AUTHORITY = "play.google.com";
    private static final String TAG_STEPS_WORKER = "ly.warp.sdk.services.WarplyHealthService";

    // ===========================================================
    // Fields
    // ===========================================================

    private ProgressChangeListener mProgressListener;
    private WarplyUrlHandler mExternalUrlHandler;
    private WarplyUrlHandler mInternalUrlHandler;
    private String sessionUUID;
    private Activity WarpActivity;
    private String geolocationOrigin = "";
    private Callback geolocationCallback;

    // ===========================================================
    // Constructors
    // ===========================================================

    public WarpView(Context context) {
        this(context, null);
    }

    public WarpView(Context context, AttributeSet attrs) {
        super(context, attrs);
        init();
    }

    public WarpView(Context context, Activity activity, boolean isActivity) {
        super(context, null);
        ProcessLifecycleOwner.get().getLifecycle().addObserver(this);
        WarpActivity = activity;
        init();
    }

    public WarpView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    // ===========================================================
    // Methods
    // ===========================================================

    @SuppressLint("SetJavaScriptEnabled")
    private void init() {

        setOverScrollMode(WarpView.OVER_SCROLL_NEVER);
        setFadingEdgeLength(0);

        WebSettings settings = getSettings();
        settings.setAppCacheEnabled(true);
        settings.setJavaScriptEnabled(true);
        settings.setDatabaseEnabled(true);
        settings.setDomStorageEnabled(true);
        settings.setGeolocationDatabasePath(getContext().getCacheDir().getAbsolutePath());
        settings.setGeolocationEnabled(true);
        settings.setBuiltInZoomControls(false);
        settings.setAllowContentAccess(true);
        settings.setLoadWithOverviewMode(true);
        settings.setAllowFileAccess(true);
        settings.setJavaScriptCanOpenWindowsAutomatically(true);
        settings.setAllowFileAccessFromFileURLs(true);
        settings.setAllowUniversalAccessFromFileURLs(true);
        settings.setSupportMultipleWindows(true);
//        settings.setGeolocationDatabasePath(getContext().getFilesDir().getPath());
        WarpView.this.addJavascriptInterface(new JSInterface(), "Cosmote");
        setBackgroundColor(ContextCompat.getColor(getContext(), android.R.color.transparent));
        setWebViewClient(new WarplyWebViewClient());
        setWebChromeClient(new WarplyWebChromeClient());
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
            if (0 != (getContext().getApplicationInfo().flags &= ApplicationInfo.FLAG_DEBUGGABLE)) {
                WebView.setWebContentsDebuggingEnabled(true);
            }
        }
        initCustomActionHandler();
    }

    private boolean isMyServiceRunning(Class<?> serviceClass) {
        ActivityManager manager = (ActivityManager) getContext().getSystemService(Context.ACTIVITY_SERVICE);
        for (ActivityManager.RunningServiceInfo service : manager.getRunningServices(Integer.MAX_VALUE)) {
            if (serviceClass.getName().equals(service.service.getClassName())) {
                return true;
            }
        }
        return false;
    }

    private void initCustomActionHandler() {

        String actionHandlerName = WarplyProperty.getWebActionListenerClassName(getContext());
        if (TextUtils.isEmpty(actionHandlerName)) return;
        if (actionHandlerName.startsWith("."))
            actionHandlerName = getContext().getPackageName().concat(actionHandlerName);
        try {
            Class<?> cls = Class.forName(actionHandlerName);
            mExternalUrlHandler = (WarplyUrlHandler) cls.newInstance();
        } catch (ClassNotFoundException e) {
            WarpUtils.warn("Warply action handler class was not found", e);
        } catch (ClassCastException e) {
            WarpUtils.warn("Warply action handler does not implement the correct interface", e);
        } catch (InstantiationException e) {
            WarpUtils.warn("Warply action handler could not be instantiated, maybe missing default constructor", e);
        } catch (IllegalAccessException e) {
            WarpUtils.warn("Warply action handler could not be accessed", e);
        }
    }

    private void actionTel(Uri uri) {

        try {
            getContext().startActivity(new Intent(Intent.ACTION_DIAL).setData(uri));
            try {
                JSONObject metadata = new JSONObject();
                metadata.putOpt("session_uuid", sessionUUID);
                JSONArray results = new JSONArray("[\"call_clicked\"]");
                metadata.putOpt("result", results);
                WarplyAnalyticsManager.logUrgentEvent("TEL", "NB_CUSTOM_ACTION", metadata);
            } catch (JSONException e) {
                if (WarpConstants.DEBUG) {
                    e.printStackTrace();
                }
            }
        } catch (ActivityNotFoundException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
    }

    private void actionRSMS(Uri uri) {

        // TODO: Needs to be implemented
        WarpUtils.log(uri.toString());
    }

    private void actionSMS(Uri uri) {

        String host = uri.getHost();
        if (host == null) {
            host = "";
        }

        String body = splitQuery(uri).get("body");
        if (body == null) body = "";

        try {
            Intent i = new Intent(Intent.ACTION_VIEW);
            i.setData(Uri.parse("sms:" + host));
            i.putExtra("sms_body", body);
            getContext().startActivity(i);

            try {
                JSONObject metadata = new JSONObject();
                metadata.putOpt("session_uuid", sessionUUID);
                JSONArray results = new JSONArray("[\"sms_prompt_clicked\"]");
                metadata.putOpt("result", results);
                WarpUtils.log(metadata.toString());
                WarplyAnalyticsManager.logUrgentEvent("SMS", "NB_CUSTOM_ACTION", metadata);
            } catch (JSONException e) {
                WarpUtils.log(e.toString());
            }
        } catch (ActivityNotFoundException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }

    }

    private void actionMail(Uri uri) {

        Map<String, String> params = splitQuery(uri);
        String subject = params.get("subject");
        if (subject == null) {
            subject = "";
        }

        String body = params.get("body");
        if (body == null) {
            body = "";
        }

        try {
            Intent i = new Intent(android.content.Intent.ACTION_SEND);
            i.putExtra(Intent.EXTRA_SUBJECT, subject);
            i.putExtra(Intent.EXTRA_TEXT, body);
            i.setData(Uri.parse("mailto:"));
            i.setType("text/plain");
            getContext().startActivity(i);

            try {
                JSONObject metadata = new JSONObject();
                metadata.putOpt("session_uuid", sessionUUID);
                JSONArray results = new JSONArray("[\"mailto_prompt_clicked\"]");
                metadata.putOpt("result", results);
                WarpUtils.log(metadata.toString());
                WarplyAnalyticsManager.logUrgentEvent("MAIL_TO", "NB_CUSTOM_ACTION", metadata);
            } catch (JSONException e) {
                WarpUtils.log(e.toString());
            }

        } catch (ActivityNotFoundException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
    }

    private void actionMarket(Uri uri) {

        try {
            Intent i = new Intent(Intent.ACTION_VIEW);
            i.setData(uri);
            getContext().startActivity(i);

            try {
                JSONObject metadata = new JSONObject();
                JSONArray results = new JSONArray("[\"PLAYSTORE\"]");
                metadata.putOpt("result", results);
                metadata.putOpt("session_uuid", sessionUUID);
                metadata.putOpt("url", uri.toString());
                WarplyAnalyticsManager.logUrgentEvent("APP_DOWNLOAD", "NB_CUSTOM_ACTION", metadata);
            } catch (JSONException e) {
                if (WarpConstants.DEBUG) {
                    e.printStackTrace();
                }
            }
        } catch (ActivityNotFoundException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
    }

    private Map<String, String> splitQuery(Uri uri) {

        Map<String, String> queryPairs = new LinkedHashMap<>();
        String query = uri.getQuery();
        String[] pairs = query.split("&");
        for (String pair : pairs) {

            if (TextUtils.isEmpty(pair)) continue;

            int idx = pair.indexOf("=");
            if (idx != -1) {
                try {
                    queryPairs.put(URLDecoder.decode(pair.substring(0, idx), "UTF-8"), URLDecoder.decode(pair.substring(idx + 1), "UTF-8"));
                } catch (UnsupportedEncodingException e) {
                    if (WarpConstants.DEBUG) {
                        e.printStackTrace();
                    }
                }
            }
        }
        return queryPairs;
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public void setProgressChangeListener(ProgressChangeListener listener) {
        this.mProgressListener = listener;
    }

    public void setWarplyUrlHandler(WarplyUrlHandler handler) {
        this.mInternalUrlHandler = handler;
    }

    public void loadWarpSessionUUID(String sessionUUID) {
        if (!TextUtils.isEmpty(sessionUUID)) {
            this.sessionUUID = sessionUUID;
            loadWarpUrl(WarplyProperty.getBaseUrl(getContext()) + WarpConstants.BASE_REMOTE_PAGE_URL + sessionUUID);
        }
    }

    public void loadWarpUrl(String url) {

        if (!TextUtils.isEmpty(url)) {
            WarpUtils.log("URL Loading: " + url);
            WarpUtils.log("Header: " + "loyalty-web-id" + ":" + WarpUtils.getWebId(getContext()));

            Map<String, String> headers = new HashMap<>();
            headers.put("loyalty-web-id", WarpUtils.getWebId(getContext()));
            loadUrl(url, headers);
        }
    }

    public String getLastSessionUUID() {
        return sessionUUID;
    }

    @Override
    public void onStart(@NonNull LifecycleOwner owner) {
        DefaultLifecycleObserver.super.onStart(owner);
        if (!EventBus.getDefault().isRegistered(this))
            EventBus.getDefault().register(this);
    }

    @Override
    public void onStop(@NonNull LifecycleOwner owner) {
        DefaultLifecycleObserver.super.onStop(owner);
    }

    @Subscribe()
    public void onMessageEvent(WarplyEventBusManager event) {
        if (event.getWarplyWebviewCallbackEventModel() != null) {
            if (event.getWarplyWebviewCallbackEventModel().getRequestId() == 3001 && event.getWarplyWebviewCallbackEventModel().getResponseCode().equals("allow")) {
//                geolocationCallback.invoke(geolocationOrigin, true, false);
                enableLocationSettings();
            } else {
                geolocationCallback.invoke(geolocationOrigin, false, false);
            }
            return;
        }
        if (event.getWarplyWebviewActivityCallbackEventModel() != null) {
            if (event.getWarplyWebviewActivityCallbackEventModel().getRequestId() == 3002 && event.getWarplyWebviewActivityCallbackEventModel().getResponseCode().equals("enabled")) {
                geolocationCallback.invoke(geolocationOrigin, true, false);
            } else {
                geolocationCallback.invoke(geolocationOrigin, false, false);
            }
        }
    }

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

    public class JSInterface {
        @JavascriptInterface
        public void sendMessage(String message) {
            if (message.contains("event")) {
                String[] parts = message.split(":");
                if (parts[1].equals("closeArtwork")) {
                    QuestionnaireEventModel questionnaireEvent = new QuestionnaireEventModel();
                    questionnaireEvent.setName(parts[1]);
                    EventBus.getDefault().post(new WarplyEventBusManager(questionnaireEvent));
                } else if (parts[1].equals("userAnswered")) {
//                    QuestionnaireEventModel questionnaireEvent = new QuestionnaireEventModel();
                    try {
//                        questionnaireEvent.setName(parts[1]);
//                        questionnaireEvent.setParameter(parts[2]);
                        WarpUtils.setUserTag(Warply.getWarplyContext(), parts[2]);
//                        EventBus.getDefault().post(new WarplyEventBusManager(questionnaireEvent));
                    } catch (IndexOutOfBoundsException e) {
//                        questionnaireEvent.setName(parts[1]);
//                        questionnaireEvent.setParameter("");
                        WarpUtils.setUserTag(Warply.getWarplyContext(), "");
//                        EventBus.getDefault().post(new WarplyEventBusManager(questionnaireEvent));
                    }
                } else if (parts[1].equals("addUserTag")) {
                    QuestionnaireEventModel questionnaireEvent = new QuestionnaireEventModel();
                    try {
                        questionnaireEvent.setName(parts[1]);
                        questionnaireEvent.setParameter(parts[2]);
                        WarpUtils.setUserTag(Warply.getWarplyContext(), parts[2]);
                        EventBus.getDefault().post(new WarplyEventBusManager(questionnaireEvent));
                    } catch (IndexOutOfBoundsException e) {
                        questionnaireEvent.setName(parts[1]);
                        questionnaireEvent.setParameter("");
                        WarpUtils.setUserTag(Warply.getWarplyContext(), "");
                        EventBus.getDefault().post(new WarplyEventBusManager(questionnaireEvent));
                    }
                } else if (parts[1].equals("couponRetrieved")) {
                    EventBus.getDefault().post(new WarplyEventBusManager(new CouponEventModel()));
                } else if (parts[1].equals("ccmsRetrieved")) {
                    WarplyCCMSEnabledModel ccmsEnabled = new WarplyCCMSEnabledModel();
                    ccmsEnabled.setActivated(true);
                    EventBus.getDefault().post(new WarplyEventBusManager(ccmsEnabled));
                } else if (parts[1].equals("steps") && parts[2].equals("widgetDisabled")) { // This is for the pacing widget
                    WarplyPacingCardEventModel pacingWidgetVisible = new WarplyPacingCardEventModel();
                    pacingWidgetVisible.setVisible(false);
                    EventBus.getDefault().post(new WarplyEventBusManager(pacingWidgetVisible));
                } else if (parts[1].equals("steps") && parts[2].equals("widgetEnabled")) { // This is for the pacing widget
                    WarplyPacingCardEventModel pacingWidgetVisible = new WarplyPacingCardEventModel();
                    pacingWidgetVisible.setVisible(true);
                    EventBus.getDefault().post(new WarplyEventBusManager(pacingWidgetVisible));
                } else if (parts[1].equals("steps") && parts[2].equals("shortcutEnabled")) { // This is for the pacing service
                    if (!isMyServiceRunning(WarplyHealthService.class)) {
                        LoyaltySDKFirebaseEventModel analyticsEvent = new LoyaltySDKFirebaseEventModel();
                        analyticsEvent.setEventName("loyalty_steps_activation");
                        EventBus.getDefault().post(new WarplyEventBusManager(analyticsEvent));

                        Intent stepsServiceIntent = new Intent(Warply.getWarplyContext(), WarplyHealthService.class);
//                        Warply.getWarplyContext().startService(stepsServiceIntent);
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                            Warply.getWarplyContext().startForegroundService(stepsServiceIntent);
                        } else {
                            Warply.getWarplyContext().startService(stepsServiceIntent);
                        }

                        WarplyPacingEventModel pacingVisible = new WarplyPacingEventModel();
                        pacingVisible.setVisible(true);
                        EventBus.getDefault().post(new WarplyEventBusManager(pacingVisible));
                    }
                } else if (parts[1].equals("steps") && parts[2].equals("shortcutDisabled")) { // This is for the pacing service
                    Intent stepsServiceIntent = new Intent(Warply.getWarplyContext(), WarplyHealthService.class);
                    Warply.getWarplyContext().stopService(stepsServiceIntent);

                    WarplyPacingEventModel pacingVisible = new WarplyPacingEventModel();
                    pacingVisible.setVisible(false);
                    EventBus.getDefault().post(new WarplyEventBusManager(pacingVisible));
                } else if (parts[1].equals("request") || parts[1].equals("response")) {
                    WarpUtils.log("**************** WARPLY Webview Log START *****************");
                    WarpUtils.log(message);
                    WarpUtils.log("**************** WARPLY Webview Log END *****************");
                } else if (parts[1].equals("loyalty_questionnaire_selected")) {
                    LoyaltySDKFirebaseEventModel analyticsEvent = new LoyaltySDKFirebaseEventModel();
                    analyticsEvent.setEventName(parts[1]);
                    analyticsEvent.setParameter("completed", "true");
                    EventBus.getDefault().post(new WarplyEventBusManager(analyticsEvent));
                } else if (parts[1].equals("loyalty_questionnaire_answer_again") || parts[1].equals("loyalty_questionnaire_answers_deleted")) {
                    LoyaltySDKFirebaseEventModel analyticsEvent = new LoyaltySDKFirebaseEventModel();
                    analyticsEvent.setEventName(parts[1]);
                    EventBus.getDefault().post(new WarplyEventBusManager(analyticsEvent));
                } else if (parts[1].equals("loyalty_questionnaire_later")) {
                    LoyaltySDKFirebaseEventModel analyticsEvent = new LoyaltySDKFirebaseEventModel();
                    analyticsEvent.setEventName("loyalty_questionnaire_selected");
                    analyticsEvent.setParameter("completed", "false");
                    EventBus.getDefault().post(new WarplyEventBusManager(analyticsEvent));
                } else if (parts[1].equals("loyalty_sdk_offer_selected")) {
                    try {
                        LoyaltySDKFirebaseEventModel analyticsEvent = new LoyaltySDKFirebaseEventModel();
                        analyticsEvent.setEventName("loyalty_sdk_offer_selected");
                        analyticsEvent.setParameter("name", parts[2]);
                        analyticsEvent.setParameter("type", parts[3]);
                        EventBus.getDefault().post(new WarplyEventBusManager(analyticsEvent));
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                } else if (parts[1].equals("loyalty_offer_activated")) {
                    try {
                        LoyaltySDKFirebaseEventModel analyticsEvent = new LoyaltySDKFirebaseEventModel();
                        analyticsEvent.setEventName("loyalty_sdk_offer_selected");
                        analyticsEvent.setParameter("name", parts[2]);
                        analyticsEvent.setParameter("type", parts[3]);
                        analyticsEvent.setParameter("successful", parts[4]);
                        EventBus.getDefault().post(new WarplyEventBusManager(analyticsEvent));
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                } else if (parts[1].equals("unified_coupon_activated")) {
                    WarplyUnifiedActivatedEventModel unifiedActivated = new WarplyUnifiedActivatedEventModel();
                    unifiedActivated.setActivated(true);
                    EventBus.getDefault().post(new WarplyEventBusManager(unifiedActivated));

                    try {
                        LoyaltySDKFirebaseEventModel analyticsEvent = new LoyaltySDKFirebaseEventModel();
                        analyticsEvent.setEventName("loyalty_sdk_unified_activated");
                        analyticsEvent.setParameter("name", parts[2]);
                        analyticsEvent.setParameter("type", parts[3]);
                        analyticsEvent.setParameter("successful", parts[4]);
                        EventBus.getDefault().post(new WarplyEventBusManager(analyticsEvent));
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                } else if (parts[1].equals("time_spent_on_loyalty_sdk")) {
                    try {
                        LoyaltySDKFirebaseEventModel analyticsEvent = new LoyaltySDKFirebaseEventModel();
                        analyticsEvent.setEventName("time_spent_on_loyalty_sdk");
                        analyticsEvent.setParameter("name", parts[2]);
                        analyticsEvent.setParameter("seconds", parts[3]);
                        EventBus.getDefault().post(new WarplyEventBusManager(analyticsEvent));
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                } else if (parts[1].equals("refreshToken")) {
                    try {
                        if (Warply.getWarplyContext() != null) {
                            WarplyDBHelper.getInstance(Warply.getWarplyContext()).saveAuthAccess(parts[2], parts[3]);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        }
    }

    public interface ProgressChangeListener {
        void onProgressChanged(WebView view, int newProgress);
    }

    private class WarplyWebChromeClient extends WebChromeClient {
        @Override
        public void onGeolocationPermissionsShowPrompt(String origin, Callback callback) {
            AlertDialog.Builder builder = new AlertDialog.Builder(getContext());
            builder.setTitle(getContext().getString(R.string.lbl_cosmote_webview_permission_title));
            builder.setMessage(getContext().getString(R.string.lbl_cosmote_webview_permission_message))
                    .setCancelable(false)
                    .setPositiveButton(getContext().getString(R.string.lbl_take_photo_accept), (dialog, id) -> checkForPermissions(origin, callback))
                    .setNegativeButton(getContext().getString(R.string.lbl_take_photo_decline), (dialog, id) -> callback.invoke(origin, false, false));
            AlertDialog alert = builder.create();
            alert.show();
        }

        @Override
        public void onProgressChanged(WebView view, int newProgress) {
            super.onProgressChanged(view, newProgress);
            if (mProgressListener != null) {
                mProgressListener.onProgressChanged(view, newProgress);
            }
        }
    }

    private void checkForPermissions(String origin, Callback callback) {
        String perm = Manifest.permission.ACCESS_FINE_LOCATION;
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M || getContext().checkSelfPermission(perm) == PackageManager.PERMISSION_GRANTED) {
            if (isGPSEnabled()) {
                callback.invoke(origin, true, false);
            } else {
                enableLocationSettings();
            }
        } else {
            if (WarpActivity != null && !WarpActivity.isFinishing()) {
                if (!ActivityCompat.shouldShowRequestPermissionRationale(WarpActivity, perm)) {
                    ActivityCompat.requestPermissions(WarpActivity, new String[]{perm}, 3001);
                    geolocationOrigin = origin;
                    geolocationCallback = callback;
                } else {
                    //TODO: show infromative popup and go to settings
                }
            }
        }
    }

    private void enableLocationSettings() {
        GoogleApiClient googleApiClient = new GoogleApiClient.Builder(WarpActivity)
                .addApi(LocationServices.API)
                .build();
        googleApiClient.connect();

        LocationRequest locationRequest = LocationRequest.create();
        locationRequest.setPriority(LocationRequest.PRIORITY_HIGH_ACCURACY);
        locationRequest.setInterval(30 * 1000);
        locationRequest.setFastestInterval(5 * 1000);
        LocationSettingsRequest.Builder builder = new LocationSettingsRequest.Builder()
                .addLocationRequest(locationRequest);
        builder.setAlwaysShow(true); //this is the key ingredient

        PendingResult<LocationSettingsResult> result = LocationServices.SettingsApi.checkLocationSettings(googleApiClient, builder.build());
        result.setResultCallback(result1 -> {
            final Status status = result1.getStatus();
            switch (status.getStatusCode()) {
                case LocationSettingsStatusCodes.SUCCESS:
                    geolocationCallback.invoke(geolocationOrigin, true, false);
                    break;
                case LocationSettingsStatusCodes.RESOLUTION_REQUIRED:
                    try {
                        status.startResolutionForResult(WarpActivity, 3002);
                    } catch (IntentSender.SendIntentException ignored) {
                        geolocationCallback.invoke(geolocationOrigin, true, false);
                    }
                    break;
                case LocationSettingsStatusCodes.SETTINGS_CHANGE_UNAVAILABLE:
                    geolocationCallback.invoke(geolocationOrigin, true, false);
                    break;
            }
        });
    }

    private boolean isGPSEnabled() {
        LocationManager lm = (LocationManager) getContext().getSystemService(Context.LOCATION_SERVICE);
        try {
            boolean gpsEnabled = lm.isProviderEnabled(LocationManager.GPS_PROVIDER);
            return gpsEnabled;
        } catch (Exception ex) {
            ex.printStackTrace();
            return false;
        }
    }

    private class WarplyWebViewClient extends WebViewClient {

        @SuppressWarnings("deprecation")
        public void onReceivedError(WebView view, int errorCode, String description, String failingUrl) {

            WarpUtils.log("Error " + errorCode + " loading url " + failingUrl);
            WarpUtils.log("Error description " + description);

            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.JELLY_BEAN_MR2) {
                WarpView.this.loadUrl("about:blank");
            } else {
                WarpView.this.clearView();
            }

            Toast.makeText(getContext(), description, Toast.LENGTH_SHORT).show();
        }

        @Override
        public void onReceivedSslError(WebView view, final SslErrorHandler handler, SslError error) {
            final AlertDialog.Builder builder = new AlertDialog.Builder(getContext());
            builder.setTitle("Ειδοποίηση");
            builder.setMessage("Θα ανακατευθυνθείς σε σελίδα χωρίς έγκυρο πιστοποιητικό");
            builder.setPositiveButton("Συμφωνώ", new DialogInterface.OnClickListener() {
                @Override
                public void onClick(DialogInterface dialog, int which) {
                    handler.proceed();
                }
            });
            builder.setNegativeButton("Άκυρο", new DialogInterface.OnClickListener() {
                @Override
                public void onClick(DialogInterface dialog, int which) {
                    handler.cancel();
                }
            });
            final AlertDialog dialog = builder.create();
            dialog.show();
        }

        @Override
        public void onPageFinished(WebView view, String url) {
            String scriptSource = "passParams(" + WarpUtils.getWebviewParams(getContext()) + ");";
            view.evaluateJavascript(scriptSource, s -> {

            });

            WarpViewActivity.getMetersHandler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    String tempMeters = String.valueOf(WarplyManagerHelper.mMetersWebview);
                    String scriptSourceMeters = "passMeters(" + tempMeters + ");";
                    WarpView.this.evaluateJavascript(scriptSourceMeters, s -> {

                    });
                    WarplyManagerHelper.mStepsWebview = 0;
                    WarplyManagerHelper.mMetersWebview = 0.0d;
                    WarpViewActivity.getMetersHandler().postDelayed(this, 300);
                }
            }, 3000); //TODO: change it back to 1000

            if (url.contains("about:blank")) {
                WarpView.this.clearHistory();
            }
        }

        @Override
        public boolean shouldOverrideUrlLoading(WebView view, String strUrl) {

            if (mExternalUrlHandler != null && mExternalUrlHandler.canHandleUrl(strUrl)) {
                mExternalUrlHandler.handleUrl(getContext(), strUrl);
                return true;
            }

            if (mInternalUrlHandler != null && mInternalUrlHandler.canHandleUrl(strUrl)) {
                mInternalUrlHandler.handleUrl(getContext(), strUrl);
                return true;
            }

            Uri uri = Uri.parse(strUrl);
            String scheme = uri.getScheme();
            String authority = uri.getAuthority();

            if (scheme == null) return false;
            if (scheme.equalsIgnoreCase(URL_ACTION_TEL)) {
                actionTel(uri);
                return true;
            } else if (scheme.equalsIgnoreCase(URL_ACTION_RSMS)) {
                actionRSMS(uri);
                return true;
            } else if (scheme.equalsIgnoreCase(URL_ACTION_SMS)) {
                actionSMS(uri);
                return true;
            } else if (scheme.equalsIgnoreCase(URL_ACTION_MAIL)) {
                actionMail(uri);
                return true;
            } else if (scheme.equals(URL_ACTION_MARKET)) {
                actionMarket(uri);
                return true;
            } else if (authority != null && authority.equalsIgnoreCase(URL_ACTION_MARKET_AUTHORITY)) {
                actionMarket(uri);
                return true;
            } else if (scheme.equals(WarplyProperty.getDlUrlScheme(Warply.getWarplyContext()))) {
                Uri webpage = Uri.parse(strUrl);
                Intent intent = new Intent(Intent.ACTION_VIEW, webpage);
                getContext().startActivity(intent);
                return true;
            } else if (scheme.equalsIgnoreCase(URL_ACTION_INTENT)) {
                try {
                    Intent intent = Intent.parseUri(strUrl, Intent.URI_INTENT_SCHEME);
//                    String fallbackUrl = intent.getStringExtra("browser_fallback_url");
                    String data = intent.getData().toString();
                    if (!TextUtils.isEmpty(data)) {
//                        view.loadUrl(fallbackUrl);
                        Uri webpage = Uri.parse(data);
                        Intent intentWebpage = new Intent(Intent.ACTION_VIEW, webpage);
                        getContext().startActivity(intentWebpage);
                        return true;
                    }
                } catch (URISyntaxException e) {
                    return true;
                }
            }
            return false;
        }
    }
}