package ly.warp.sdk.views.adapters;

import android.content.Context;
import android.os.Handler;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;

import java.util.Collections;
import java.util.Comparator;
import java.util.concurrent.atomic.AtomicBoolean;

import ly.warp.sdk.BuildConfig;
import ly.warp.sdk.Warply;
import ly.warp.sdk.io.callbacks.CallbackReceiver;
import ly.warp.sdk.io.callbacks.SimpleCallbackReceiver;
import ly.warp.sdk.io.models.Campaign;
import ly.warp.sdk.io.models.CampaignList;
import ly.warp.sdk.io.request.WarplyInboxRequest;
import ly.warp.sdk.views.CampaignItemViewHolder;
import ly.warp.sdk.views.CampaignItemWebHolder;

/**
 * The BaseAdapter to be used
 * to display Warply's Campaign Offers on a ListView
 *
 * @param <VH> The ViewHolder Type of offers items to be shown on the ListView
 */
public class CampaignsArrayAdapter<VH extends CampaignItemViewHolder> extends BaseAdapter
        implements CallbackReceiver<CampaignList> {

    // ===========================================================
    // Constants
    // ===========================================================

    // ===========================================================
    // Fields
    // ===========================================================

    private Class<VH> mViewHolderClass;
    private CampaignList mCampaigns;
    private WarplyInboxRequest mCampaignsRequest;
    private AtomicBoolean isRequestComplete = new AtomicBoolean(true);

    private static Handler mHandler = new Handler();
    private SimpleCallbackReceiver<CampaignList> mExternalListener;

    private int mWebItemWidth = 0;
    private int mWebItemHeight = 0;

    // ===========================================================
    // Constructors
    // ===========================================================

    /**
     * CampaignsArrayAdapter Constructor
     *
     * @param context          Activity's {@link Context}
     * @param campaignsRequest {@link WarplyInboxRequest} used to fetch campaign offers.
     */
    public CampaignsArrayAdapter(Context context, WarplyInboxRequest campaignsRequest) {
        this(context, campaignsRequest, (Class<VH>) CampaignItemWebHolder.class);
    }

    /**
     * CampaignsArrayAdapter Constructor
     * not used often
     *
     * @param context          Activity's {@link Context}
     * @param campaignsRequest {@link WarplyInboxRequest} used to fetch campaign offers.
     * @param webItemWidth     Offer's item width
     * @param webItemHeight    Offer's item height
     */
    public CampaignsArrayAdapter(Context context, WarplyInboxRequest campaignsRequest, int webItemWidth, int webItemHeight) {
        this(context, campaignsRequest);
        this.mWebItemWidth = webItemWidth;
        this.mWebItemHeight = webItemHeight;
    }

    /**
     * CampaignsArrayAdapter Basic Constructor
     *
     * @param context          Activity's {@link Context}
     * @param campaignsRequest {@link WarplyInboxRequest} used to fetch campaign offers.
     * @param viewHolderClass  Class of the offers' ViewHolder item to be inflated inside {@link android.widget.ListView}
     */
    public CampaignsArrayAdapter(Context context, WarplyInboxRequest campaignsRequest, Class<VH> viewHolderClass) {

        Warply.getInitializer(context).init();
        mCampaignsRequest = campaignsRequest;
        mViewHolderClass = viewHolderClass;
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public int getCount() {
        if (mCampaigns == null)
            return 0;
        else
            return mCampaigns.size();
    }

    @Override
    public Campaign getItem(int position) {
        if (mCampaigns == null || position < 0 || position >= mCampaigns.size())
            return null;
        else
            return mCampaigns.get(position);
    }

    @Override
    public long getItemId(int position) {
        return position;
    }

    /**
     * This method will only be called when when the adapter is set on the the AdapterView.
     */
    @Override
    public int getViewTypeCount() {
        requestCampaigns();
        return super.getViewTypeCount();
    }

    @Override
    @SuppressWarnings("unchecked")
    public View getView(int position, View convertView, ViewGroup parent) {

        CampaignItemViewHolder viewHolder = null;
        View itemView = convertView;
        if (itemView == null) {
            try {
                if (mViewHolderClass == CampaignItemWebHolder.class) {
                    viewHolder = mViewHolderClass.getDeclaredConstructor(ViewGroup.class, int.class, int.class).
                            newInstance(parent, mWebItemWidth, mWebItemHeight);
                } else {
                    viewHolder = mViewHolderClass.getDeclaredConstructor(ViewGroup.class).newInstance(parent);
                }
                itemView = viewHolder.itemView;
                itemView.setTag(viewHolder);
            } catch (Exception e) {
                if (BuildConfig.DEBUG) {
                    e.printStackTrace();
                }
            }
        } else {
            viewHolder = (VH) itemView.getTag();
        }

        Campaign item = getItem(position);
        if (item != null && viewHolder != null) {
            viewHolder.bindData(item, position);
            itemView = viewHolder.itemView;
        }
        return itemView;
    }

    @Override
    public View getDropDownView(int position, View convertView, ViewGroup parent) {
        return getView(position, convertView, parent);
    }

    @Override
    public void onSuccess(CampaignList result) {

        mCampaigns = result;
        updateAdapter();
        notifySuccessExternalListener(result);
    }

    @Override
    public void onFailure(int errorCode) {

        clear();
        updateAdapter();
        notifyFailureExternalListener(errorCode);
    }

    // ===========================================================
    // Methods
    // ===========================================================

    private void requestCampaigns() {

        if (isRequestComplete.get()) {
            if (isEmpty()) {
                isRequestComplete.set(false);
                Warply.getInbox(mCampaignsRequest, this);
            }
        }
    }

    private void updateAdapter() {
        mHandler.removeCallbacks(mUpdateItemsRunnable);
        mHandler.post(mUpdateItemsRunnable);
    }

    /**
     * clears adapter's {@link CampaignList}
     */
    public void clear() {

        if (this.mCampaigns != null) {
            this.mCampaigns.clear();
        }
    }

    private void notifySuccessExternalListener(CampaignList campaigns) {

        if (mExternalListener != null) {
            mExternalListener.onSuccess(campaigns);
        }
    }

    private void notifyFailureExternalListener(int errorCode) {

        if (mExternalListener != null) {
            mExternalListener.onFailure(errorCode);
        }
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    /**
     * @return adapter's {@link CampaignList}
     */
    public CampaignList getItems() {
        return mCampaigns;
    }

    /**
     * @return number of unread {@link Campaign} messages
     */
    public int getCurrentUnreadMessages() {
        return getCount() > 0 ? mCampaigns.getUnreadCampaignsCount() : 0;
    }

    /**
     * sorts the adapter by the given {@link Comparator}
     *
     * @param comparator the {@link Comparator} to sort the adapter
     */
    public void sort(Comparator<Campaign> comparator) {

        if (mCampaigns != null && mCampaigns.size() > 0) {
            Collections.sort(mCampaigns, comparator);
            notifyDataSetChanged();
        }
    }

    /**
     * updates the offers inside the {@link android.widget.ListView}
     * by a specific {@link WarplyInboxRequest}
     *
     * @param request the {@link WarplyInboxRequest} used to update the
     *                offers inside the {@link android.widget.ListView}
     */
    public void updateCampaigns(WarplyInboxRequest request) {
        mCampaignsRequest = request;
        clear();
        isRequestComplete.set(true);
        requestCampaigns();
    }


    /**
     * updates the offers inside the {@link android.widget.ListView}
     * with certain campaigns.
     *
     * @param result the {@link CampaignList} used to update the
     *                offers inside the {@link android.widget.ListView}
     */
    public void updateCampaigns(CampaignList result) {
        this.mCampaigns = result;
        this.clear();
        this.updateAdapter();
        this.notifySuccessExternalListener(result);
    }

    /**
     * Used to set a listener to be notified when the response of the {@link WarplyInboxRequest}
     * is received
     *
     * @param listener the {@link SimpleCallbackReceiver} to be notified
     *                 of the {@link WarplyInboxRequest}'s response
     */
    public void setCampaignsListener(SimpleCallbackReceiver<CampaignList> listener) {
        this.mExternalListener = listener;
    }

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

    private Runnable mUpdateItemsRunnable = new Runnable() {
        public void run() {
            CampaignsArrayAdapter.this.notifyDataSetChanged();
            isRequestComplete.set(true);
        }
    };


}
