package ly.warp.sdk.views.adapters;

import android.content.Context;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.core.text.HtmlCompat;
import androidx.recyclerview.widget.RecyclerView;

import com.bumptech.glide.Glide;
import com.bumptech.glide.load.engine.DiskCacheStrategy;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.concurrent.TimeUnit;

import io.reactivex.Observable;
import io.reactivex.subjects.PublishSubject;
import ly.warp.sdk.R;
import ly.warp.sdk.io.models.Couponset;
import ly.warp.sdk.io.models.CouponsetsList;

public class CouponsetsAdapter extends RecyclerView.Adapter<CouponsetsAdapter.CouponsetViewHolder> {

    private Context mContext;
    private CouponsetsList mCouponsets;
    private final PublishSubject<Couponset> onClickSubject = PublishSubject.create();

    public CouponsetsAdapter(Context mContext, CouponsetsList couponsets) {
        this.mContext = mContext;
        this.mCouponsets = couponsets;
    }

    public class CouponsetViewHolder extends RecyclerView.ViewHolder {
        private ImageView ivCouponLogo;
        private TextView tvCouponTitle, tvCouponValue, tvCouponDate, tvCouponDescription;

        public CouponsetViewHolder(View view) {
            super(view);
            ivCouponLogo = view.findViewById(R.id.iv_active_coupon);
            tvCouponTitle = view.findViewById(R.id.tv_active_coupons_title);
            tvCouponValue = view.findViewById(R.id.tv_active_coupons_value);
            tvCouponDate = view.findViewById(R.id.tv_active_coupons_date);
            tvCouponDescription = view.findViewById(R.id.tv_active_coupons_description);
        }
    }

    @Override
    public int getItemCount() {
        if (mCouponsets == null)
            return 0;
        else
            return mCouponsets.size();
    }


    public Couponset getItem(int id) {
        return mCouponsets.get(id);
    }

    public void updateData(CouponsetsList couponsets) {
        mCouponsets.clear();
        mCouponsets.addAll(couponsets);
        notifyDataSetChanged();
    }


    @Override
    public CouponsetViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
        View itemView;
        itemView = LayoutInflater.from(parent.getContext()).inflate(R.layout.active_coupon_layout, parent, false);
        return new CouponsetViewHolder(itemView);
    }

    @Override
    public void onBindViewHolder(final CouponsetViewHolder holder, int position) {
        Couponset couponsetItem = mCouponsets.get(position);

        if (couponsetItem != null) {
            if (!TextUtils.isEmpty(couponsetItem.getImgPreview())) {
                Glide.with(mContext)
//                        .setDefaultRequestOptions(
//                                RequestOptions
//                                        .placeholderOf(R.drawable.ic_default_contact_photo)
//                                        .error(R.drawable.ic_default_contact_photo))
                        .load(couponsetItem.getImgPreview())
                        .diskCacheStrategy(DiskCacheStrategy.DATA)
                        .into(holder.ivCouponLogo);
            } else {
                Glide.with(mContext)
                        .load(R.drawable.ic_cosmote_logo_horizontal_grey)
                        .into(holder.ivCouponLogo);
            }

            holder.tvCouponTitle.setText(couponsetItem.getName());
            holder.tvCouponDescription.setText(HtmlCompat.fromHtml(couponsetItem.getDescription(), HtmlCompat.FROM_HTML_MODE_COMPACT));

            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd hh:mm");
            Date newDate = new Date();
            try {
                newDate = simpleDateFormat.parse(couponsetItem.getExpiration());
            } catch (ParseException e) {
                e.printStackTrace();
            }
            simpleDateFormat = new SimpleDateFormat("dd/MM/yyyy");
            holder.tvCouponDate.setText(String.format(mContext.getString(R.string.cos_coupon_date), simpleDateFormat.format(newDate != null ? newDate : "")));

            holder.tvCouponValue.setText(couponsetItem.getDiscount());
            holder.itemView.setOnClickListener(v -> onClickSubject.onNext(couponsetItem));
        }
    }

    public Observable<Couponset> getPositionClicks() {
        return onClickSubject.cache();
    }

    private long getDifferenceDays(Date d1, Date d2) {
        long diff = d2.getTime() - d1.getTime();
        return TimeUnit.DAYS.convert(diff, TimeUnit.MILLISECONDS);
    }
}
