package ly.warp.sdk.views.adapters.mix;

import android.content.Context;
import android.os.Handler;
import android.view.View;

import java.util.LinkedList;
import java.util.Queue;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicBoolean;

import ly.warp.sdk.Warply;
import ly.warp.sdk.io.callbacks.CallbackReceiver;
import ly.warp.sdk.io.callbacks.SimpleCallbackReceiver;
import ly.warp.sdk.io.models.Campaign;
import ly.warp.sdk.io.models.CampaignList;
import ly.warp.sdk.io.request.WarplyInboxRequest;
import ly.warp.sdk.views.CampaignItemViewHolder;
import ly.warp.sdk.views.CampaignItemWebHolder;

public class CampaignsMixController implements CallbackReceiver<CampaignList> {

    // ===========================================================
    // Constants
    // ===========================================================

    public static final int CAMPAIGN_START_POSITION = 1;
    public static final int CAMPAIGN_FREQUENCY = 5;

    // ===========================================================
    // Fields
    // ===========================================================

    private CampaignList mCampaigns;
    private WarplyInboxRequest mCampaignsRequest;
    private TreeMap<Integer, Campaign> mPositionsToCampaigns;
    private Queue<Campaign> mCampaignsPool;
    private AtomicBoolean isRequestComplete = new AtomicBoolean(true);
    private int mCampaignsStartPosition;
    private int mCampaignsFrequency;
    private boolean isRepeatableSet;

    private static Handler mHandler = new Handler();
    private MixControllerListener mControllerListener;

    private SimpleCallbackReceiver<CampaignList> mExternalListener;
    private CampaignsMixTracker mAnalyticsTracker;

    // ===========================================================
    // Constructors
    // ===========================================================

    public CampaignsMixController(Context context, WarplyInboxRequest campaignsRequest, MixControllerListener listener) {

        Warply.getInitializer(context).init();
        mPositionsToCampaigns = new TreeMap<>();
        mCampaignsPool = new LinkedList<>();
        mCampaignsStartPosition = CAMPAIGN_START_POSITION;
        mCampaignsFrequency = CAMPAIGN_FREQUENCY;
        mCampaignsRequest = campaignsRequest;
        mControllerListener = listener;
        mAnalyticsTracker = new CampaignsMixTracker(this);
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public void onSuccess(CampaignList result) {

        mCampaigns = result;
        mCampaignsPool = new LinkedList<>(mCampaigns);

        updateAdapter();
        notifySuccessExternalListener(result);
    }

    @Override
    public void onFailure(int errorCode) {

        clear();
        updateAdapter();
        notifyFailureExternalListener(errorCode);
    }

    // ===========================================================
    // Methods
    // ===========================================================

    private boolean isCampaignForPositionAdded(int position) {
        return this.mPositionsToCampaigns.containsKey(Integer.valueOf(position));
    }

    private boolean checkIfPositionForCampaignAndAdd(int position) {

        boolean isCampaignPosition = ((position - mCampaignsStartPosition) % (mCampaignsFrequency + 1)) == 0;
        int maxCampaignsToLoad = ((mControllerListener.getUserAdapterCount() - mCampaignsStartPosition) / mCampaignsFrequency) + 1;
        if (position == mControllerListener.getUserAdapterCount() + maxCampaignsToLoad) {
            isCampaignPosition = false;
        }

        if (isCampaignPosition && !isCampaignForPositionAdded(position)) {
            getNextCampaignToShow(position);
        }

        return isCampaignPosition && isCampaignForPositionAdded(position);
    }

    private void getNextCampaignToShow(int position) {

        if (!mCampaignsPool.isEmpty()) {
            Campaign campaign = mCampaignsPool.poll();
            mPositionsToCampaigns.put(position, campaign);
            updateAdapter();
        }

        if (mCampaignsPool.size() <= 0) {
            requestCampaigns();
        }
    }

    private void requestCampaigns(/* pass page number of campaigns in future if need */) {

        if (isRequestComplete.get()) {
            // if first request
            if (mCampaigns == null) {
                isRequestComplete.set(false);
                Warply.getInbox(mCampaignsRequest, this);
            } else if (isRepeatableSet) {
                isRequestComplete.set(false);
                onSuccess(mCampaigns);
            } /*else {
            // pass next page to campaigns request
            Warply.getInbox(this, mCampaignsRequest);
        }*/
        }

    }

    private void updateAdapter() {
        mHandler.removeCallbacks(mUpdateItemsRunnable);
        mHandler.postDelayed(mUpdateItemsRunnable, 250L);
    }

    private void clear() {

        if (this.mCampaigns != null) {
            this.mCampaigns.clear();
        }

        if (this.mCampaignsPool != null) {
            this.mCampaignsPool.clear();
        }

        if (mPositionsToCampaigns != null) {
            mPositionsToCampaigns.clear();
        }
    }

    private void notifySuccessExternalListener(CampaignList campaigns) {

        if (mExternalListener != null) {
            mExternalListener.onSuccess(campaigns);
        }
    }

    private void notifyFailureExternalListener(int errorCode) {

        if (mExternalListener != null) {
            mExternalListener.onFailure(errorCode);
        }
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public int getCount() {
        return mControllerListener.getUserAdapterCount() + mPositionsToCampaigns.size();
    }

    public long getItemId(int position) {
        return isCampaignForPositionAdded(position) ? position :
                mControllerListener.getUserAdapterItemId(getUserAdapterPosition(position));
    }

    public int getViewTypeCount() {
        return mControllerListener.getUserAdapterViewTypeCount() + 1;
    }

    public int getItemViewType(int position) {
        return checkIfPositionForCampaignAndAdd(position) ? getCampaignItemViewType() :
                mControllerListener.getUserAdapterItemViewType(getUserAdapterPosition(position));
    }

    public Campaign getCampaignItem(int position) {
        return mPositionsToCampaigns.get(position);
    }

    public int getCampaignItemViewType() {
        return getViewTypeCount() - 1;
    }

    public int getUserAdapterPosition(int globalPosition) {

        return mPositionsToCampaigns.size() != 0 && globalPosition >= mCampaignsStartPosition ?
                globalPosition - mPositionsToCampaigns.subMap(0, globalPosition).size() : globalPosition;
    }

    public int getGlobalPosition(int userAdapterPosition) {

        int campaignsCountBeforePosition = (userAdapterPosition - this.mCampaignsStartPosition) / this.mCampaignsFrequency + 1;
        int globalPosition = userAdapterPosition + campaignsCountBeforePosition;
        if (mPositionsToCampaigns.size() != 0 && campaignsCountBeforePosition > 0) {
            // fix position for not loaded campaigns (in mPositionsToCampaigns)
            globalPosition += mPositionsToCampaigns.subMap(0, globalPosition).size() - campaignsCountBeforePosition;
            return globalPosition;
        } else {
            return userAdapterPosition;
        }
    }

    public void resetCampaignsPositions() {
        try {
            this.mPositionsToCampaigns = new TreeMap<>(mPositionsToCampaigns
                    .subMap(0, mControllerListener.getUserAdapterCount() + 1));
        } catch (Exception var2) {
            this.mPositionsToCampaigns.clear();
        }
    }

    public CampaignList getCampaigns() {
        return mCampaigns;
    }

    public int getCurrentUnreadMessages() {
        return mCampaigns != null && mCampaigns.size() > 0 ?
                mCampaigns.getUnreadCampaignsCount() : 0;
    }

    public void setCampaignsListener(SimpleCallbackReceiver<CampaignList> listener) {
        this.mExternalListener = listener;
    }

    public void updateCampaigns(WarplyInboxRequest campaignsRequest) {
        mCampaignsRequest = campaignsRequest;
        clear();
        updateAdapter();
    }

    /**
     * set start position and frequency before
     * set adapter in view (before load campaigns)
     */
    public void setMixBounds(int startPosition, int frequency) {

        if (mCampaigns == null || mCampaigns.size() == 0) {
            if (startPosition >= 0) {
                mCampaignsStartPosition = startPosition;
            }
            if (frequency > 0) {
                mCampaignsFrequency = frequency;
            }
        }
    }

    public void setRepeatableSet(boolean isRepeatable) {
        isRepeatableSet = isRepeatable;
    }

    /**
     * call in onScroll for correct results
     *
     * @param firstVisiblePosition - first visible position in global adapter
     * @param lastVisiblePosition  - last visible position in global adapter
     */
    public void trackCampaignsView(int firstVisiblePosition, int lastVisiblePosition) {
        mAnalyticsTracker.trackCampaignsView(firstVisiblePosition, lastVisiblePosition);
    }

    /**
     * call in bindView and getView in adapter after data binding for tracking campaigns clicks
     *
     * @param campaignViewHolder - view holder where was campaign bound
     */
    public CampaignTrackClickListener getTrackCampaignClickListener(CampaignItemViewHolder campaignViewHolder) {
        return new CampaignTrackClickListener(campaignViewHolder, mAnalyticsTracker);
    }

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

    private Runnable mUpdateItemsRunnable = new Runnable() {
        public void run() {
            mControllerListener.onMixControllerUpdate();
            isRequestComplete.set(true);
        }
    };


    public interface MixControllerListener {

        void onMixControllerUpdate();

        int getUserAdapterCount();

        long getUserAdapterItemId(int position);

        int getUserAdapterViewTypeCount();

        int getUserAdapterItemViewType(int position);
    }

    private class CampaignTrackClickListener implements View.OnClickListener {

        private CampaignItemViewHolder mmCampaignViewHolder;
        private CampaignsMixTracker mmAnalyticsTracker;

        public CampaignTrackClickListener(CampaignItemViewHolder campaignViewHolder, CampaignsMixTracker tracker) {
            this.mmCampaignViewHolder = campaignViewHolder;
            this.mmAnalyticsTracker = tracker;
        }

        @Override
        public void onClick(View v) {

            if (mmCampaignViewHolder != null) {

                if (mmCampaignViewHolder.getCampaign() != null
                        && !(mmCampaignViewHolder instanceof CampaignItemWebHolder)
                        && mmAnalyticsTracker != null) {
                    mmAnalyticsTracker.trackCampaignClick(mmCampaignViewHolder.getCampaign());
                }
                // process click inside view holder for open campaign
                mmCampaignViewHolder.onClick(v);
            }
        }
    }

}
