package ly.warp.sdk.views.adapters.mix;

import android.database.DataSetObserver;
import androidx.annotation.NonNull;
import androidx.recyclerview.widget.RecyclerView;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AbsListView;
import android.widget.BaseAdapter;
import android.widget.ListView;

import ly.warp.sdk.BuildConfig;
import ly.warp.sdk.io.callbacks.SimpleCallbackReceiver;
import ly.warp.sdk.io.models.Campaign;
import ly.warp.sdk.io.models.CampaignList;
import ly.warp.sdk.io.request.WarplyInboxRequest;
import ly.warp.sdk.views.CampaignItemViewHolder;
import ly.warp.sdk.views.CampaignItemWebHolder;

/**
 * The BaseAdapter to be used
 * to display Warply's native ads on a ListView
 *
 * @param <VH> The ViewHolder Type of Native ad item to be mixed on the ListView
 */
public class CampaignsMixListAdapter<VH extends CampaignItemViewHolder> extends BaseAdapter
        implements CampaignsMixController.MixControllerListener {

    // ===========================================================
    // Constants
    // ===========================================================

    // ===========================================================
    // Fields
    // ===========================================================

    private ListView mListView;
    private Class<VH> mViewHolderClass;
    private BaseAdapter mUserAdapter;
    protected CampaignsMixController mMixController;

    private int mWebItemWidth = 0;
    private int mWebItemHeight = 0;

    // ===========================================================
    // Constructors
    // ===========================================================

    protected CampaignsMixListAdapter(Builder<VH> builder) {

        mListView = builder.listView;
        mMixController = new CampaignsMixController(mListView.getContext(), builder.campaignsRequest, this);
        mMixController.setRepeatableSet(builder.isRepeatableSet);
        mMixController.setMixBounds(builder.startPosition, builder.frequency);

        mWebItemHeight = builder.webItemHeight;
        mWebItemWidth = builder.webItemWidth;

        mViewHolderClass = builder.viewHolderClass;
        mUserAdapter = builder.userAdapter;
        mUserAdapter.registerDataSetObserver(new DataSetObserver() {

            @Override
            public void onChanged() {
                super.onChanged();

                mMixController.resetCampaignsPositions();
                notifyDataSetChanged();
            }

            @Override
            public void onInvalidated() {
                super.onInvalidated();
                notifyDataSetInvalidated();
            }
        });
        mListView.setOnScrollListener(new AbsListView.OnScrollListener() {

            @Override
            public void onScrollStateChanged(AbsListView view, int scrollState) {

            }

            @Override
            public void onScroll(AbsListView view, int firstVisibleItem, int visibleItemCount, int totalItemCount) {
                if (visibleItemCount > 0) {
                    mMixController.trackCampaignsView(firstVisibleItem, firstVisibleItem + visibleItemCount - 1);
                }
            }
        });
        start();
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public int getCount() {
        return mMixController.getCount();
    }

    @Override
    public Object getItem(int position) {
        return mUserAdapter.getItem(mMixController.getUserAdapterPosition(position));
    }

    @Override
    public long getItemId(int position) {
        return mMixController.getItemId(position);
    }

    @Override
    public int getViewTypeCount() {
        return mMixController.getViewTypeCount();
    }

    @Override
    public int getItemViewType(int position) {
        return mMixController.getItemViewType(position);
    }

    @Override
    public void onMixControllerUpdate() {
        notifyDataSetChanged();
    }

    @Override
    public int getUserAdapterCount() {
        return mUserAdapter.getCount();
    }

    @Override
    public long getUserAdapterItemId(int position) {
        return mUserAdapter.getItemId(position);
    }

    @Override
    public int getUserAdapterViewTypeCount() {
        return mUserAdapter.getViewTypeCount();
    }

    @Override
    public int getUserAdapterItemViewType(int position) {
        return mUserAdapter.getItemViewType(position);
    }

    @Override
    public View getView(int position, View convertView, ViewGroup parent) {

        if (getItemViewType(position) == mMixController.getCampaignItemViewType()) {
            return getCampaignView(position, convertView, parent);
        } else {
            return getUserAdapterView(mMixController.getUserAdapterPosition(position), convertView, parent);
        }
    }

    @Override
    public View getDropDownView(int position, View convertView, ViewGroup parent) {
        return getView(position, convertView, parent);
    }

    // ===========================================================
    // Methods
    // ===========================================================

    protected void start() {
        mListView.setAdapter(this);
    }

    @SuppressWarnings("unchecked")
    private View getCampaignView(int position, View convertView, ViewGroup parent) {

        CampaignItemViewHolder viewHolder = null;
        View itemView = convertView;
        if (itemView == null || !(convertView.getTag() instanceof CampaignItemViewHolder)) {
            try {
                if (mViewHolderClass == CampaignItemWebHolder.class) {
                    viewHolder = mViewHolderClass.getDeclaredConstructor(ViewGroup.class, int.class, int.class).
                            newInstance(parent, mWebItemWidth, mWebItemHeight);
                } else {
                    viewHolder = mViewHolderClass.getDeclaredConstructor(ViewGroup.class).newInstance(parent);
                }

                itemView = viewHolder.itemView;
                itemView.setTag(viewHolder);
            } catch (Exception e) {
                if (BuildConfig.DEBUG) {
                    e.printStackTrace();
                }
            }
        } else {
            viewHolder = (VH) itemView.getTag();
        }

        Campaign item = mMixController.getCampaignItem(position);
        if (item != null && viewHolder != null) {
            viewHolder.bindData(item, position);
            viewHolder.itemView.setOnClickListener(mMixController.getTrackCampaignClickListener(viewHolder));
            itemView = viewHolder.itemView;
        }
        return itemView;
    }

    private View getUserAdapterView(int position, View convertView, ViewGroup parent) {

        // if convertView is not the user adapter view - let create new user view
        if (convertView != null && convertView.getTag() instanceof CampaignItemViewHolder) {
            convertView = null;
        }
        return mUserAdapter.getView(position, convertView, parent);
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    /**
     * @return native ads' {@link CampaignList}
     */
    public CampaignList getCampaigns() {
        return mMixController.getCampaigns();
    }

    /**
     * @return number of unread {@link Campaign} messages
     */
    public int getCurrentUnreadMessages() {
        return mMixController.getCurrentUnreadMessages();
    }

    /**
     * Used to set a listener to be notified when the response of the {@link WarplyInboxRequest}
     * is received
     *
     * @param listener the {@link SimpleCallbackReceiver} to be notified
     *                 of the {@link WarplyInboxRequest}'s response
     */
    public void setCampaignsListener(SimpleCallbackReceiver<CampaignList> listener) {
        mMixController.setCampaignsListener(listener);
    }

    /**
     * updates the native ads inside the {@link android.widget.ListView}
     * by a specific {@link WarplyInboxRequest}
     *
     * @param campaignsRequest the {@link WarplyInboxRequest} used to update the
     *                         native ads inside the {@link android.widget.ListView}
     */
    public void updateCampaigns(WarplyInboxRequest campaignsRequest) {
        mMixController.updateCampaigns(campaignsRequest);
    }

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

    public static class Builder<VH extends CampaignItemViewHolder> {

        private ListView listView;
        private WarplyInboxRequest campaignsRequest;
        private Class<VH> viewHolderClass;
        private BaseAdapter userAdapter;

        private int webItemWidth = 0;
        private int webItemHeight = 0;

        private boolean isRepeatableSet;
        private int startPosition;
        private int frequency;

        public CampaignsMixListAdapter adapt() {
            return new CampaignsMixListAdapter<>(this);
        }

        /**
         * CampaignsMixListAdapter constructor
         *
         * @param listView         the ListView {@link ListView} where will adapted data sets
         * @param campaignsRequest {@link WarplyInboxRequest} used to fetch campaigns for native ads.
         *                         In order to show native ads within the {@link android.widget.ListView} a
         *                         {@link ly.warp.sdk.io.request.WarplyInboxRequest.DisplayType} must be added
         *                         on the {@link WarplyInboxRequest} through {@code new WarplyInboxRequest().addDisplayTypeFilter(DisplayType displayType)}
         * @param userAdapter      original Adapter that will be mixed with native ads
         */
        public Builder(@NonNull ListView listView, WarplyInboxRequest campaignsRequest, BaseAdapter userAdapter) {
            this(listView, campaignsRequest, (Class<VH>) CampaignItemWebHolder.class, userAdapter);
        }

        /**
         * CampaignsMixListAdapter constructor
         * this constructor is not used often
         *
         * @param listView         the ListView {@link ListView} where will adapted data sets
         * @param campaignsRequest {@link WarplyInboxRequest} used to fetch campaigns for native ads.
         *                         In order to show native ads within the {@link android.widget.ListView} a
         *                         {@link ly.warp.sdk.io.request.WarplyInboxRequest.DisplayType} must be added
         *                         on the {@link WarplyInboxRequest} through {@code new WarplyInboxRequest().addDisplayTypeFilter(DisplayType displayType)}
         * @param userAdapter      original Adapter that will be mixed with native ads
         * @param webItemWidth     width of native ad item
         * @param webItemHeight    height of native ad item
         */
        public Builder(@NonNull ListView listView, WarplyInboxRequest campaignsRequest,
                       BaseAdapter userAdapter, int webItemWidth, int webItemHeight) {
            this(listView, campaignsRequest, userAdapter);
            this.webItemWidth = webItemWidth;
            this.webItemHeight = webItemHeight;
        }

        /**
         * CampaignsMixRecyclerAdapter Basic constructor
         *
         * @param listView         the ListView {@link ListView} where will adapted data sets
         * @param campaignsRequest {@link WarplyInboxRequest} used to fetch campaigns for native ads.
         *                         In order to show native ads within the {@link RecyclerView} a
         *                         {@link ly.warp.sdk.io.request.WarplyInboxRequest.DisplayType} must be added
         *                         on the {@link WarplyInboxRequest} through {@code new WarplyInboxRequest().addDisplayTypeFilter(DisplayType displayType)}
         * @param viewHolderClass  Class of the native ad ViewHolder item to be inflated inside {@link android.widget.ListView}
         * @param userAdapter      original Adapter that will be mixed with native ads
         */
        public Builder(@NonNull ListView listView, WarplyInboxRequest campaignsRequest,
                       Class<VH> viewHolderClass, @NonNull BaseAdapter userAdapter) {

            this.isRepeatableSet = false;
            this.startPosition = CampaignsMixController.CAMPAIGN_START_POSITION;
            this.frequency = CampaignsMixController.CAMPAIGN_FREQUENCY;

            this.listView = listView;
            this.campaignsRequest = campaignsRequest;
            this.viewHolderClass = viewHolderClass;
            this.userAdapter = userAdapter;
        }

        /**
         * set start position and frequency
         */
        public Builder setMixBounds(int startPosition, int frequency) {
            this.startPosition = startPosition;
            this.frequency = frequency;
            return this;
        }

        /**
         * @param isRepeatable true to repeat ads after the last ad is shown
         *                     false otherwise
         */
        public Builder setRepeatableSet(boolean isRepeatable) {
            this.isRepeatableSet = isRepeatable;
            return this;
        }
    }
}
