package ly.warp.sdk.views.adapters.mix;

import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import android.view.ViewGroup;

import ly.warp.sdk.io.callbacks.SimpleCallbackReceiver;
import ly.warp.sdk.io.models.Campaign;
import ly.warp.sdk.io.models.CampaignList;
import ly.warp.sdk.io.request.WarplyInboxRequest;
import ly.warp.sdk.utils.constants.WarpConstants;
import ly.warp.sdk.views.CampaignItemViewHolder;
import ly.warp.sdk.views.CampaignItemWebHolder;

/**
 * The RecyclerView.Adapter to be used
 * to display Warply's native ads on a RecyclerView
 *
 * @param <VH> The ViewHolder Type of Native ad item to be mixed on the RecyclerView
 */
public class CampaignsMixRecyclerAdapter<VH extends CampaignItemViewHolder> extends
        RecyclerView.Adapter implements CampaignsMixController.MixControllerListener {

    // ===========================================================
    // Constants
    // ===========================================================

    // ===========================================================
    // Fields
    // ===========================================================

    private RecyclerView mRecycler;
    private Class<VH> mViewHolderClass;
    private RecyclerView.Adapter mUserAdapter;
    private CampaignsMixController mMixController;

    private int mWebItemWidth = 0;
    private int mWebItemHeight = 0;

    // ===========================================================
    // Constructors
    // ===========================================================

    protected CampaignsMixRecyclerAdapter(Builder<VH> builder) {

        mRecycler = builder.recycler;
        mMixController = new CampaignsMixController(mRecycler.getContext(), builder.campaignsRequest, this);
        mMixController.setRepeatableSet(builder.isRepeatableSet);
        mMixController.setMixBounds(builder.startPosition, builder.frequency);

        mWebItemHeight = builder.webItemHeight;
        mWebItemWidth = builder.webItemWidth;

        mViewHolderClass = builder.viewHolderClass;
        mUserAdapter = builder.userAdapter;
        mUserAdapter.registerAdapterDataObserver(new RecyclerView.AdapterDataObserver() {
            @Override
            public void onChanged() {
                super.onChanged();

                mMixController.resetCampaignsPositions();
                notifyDataSetChanged();
            }
        });
        mRecycler.addOnScrollListener(new RecyclerView.OnScrollListener() {
            @Override
            public void onScrolled(RecyclerView recyclerView, int dx, int dy) {
                super.onScrolled(recyclerView, dx, dy);

                RecyclerView.LayoutManager lm = recyclerView.getLayoutManager();
                if (lm instanceof LinearLayoutManager) {
                    int firstPosition = ((LinearLayoutManager) lm).findFirstVisibleItemPosition();
                    int lastPosition = ((LinearLayoutManager) lm).findLastVisibleItemPosition();
                    mMixController.trackCampaignsView(firstPosition, lastPosition);
                }
            }
        });
        start();
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public int getItemCount() {
        return mMixController.getCount();
    }

    @Override
    public int getItemViewType(int position) {
        return mMixController.getItemViewType(position);
    }

    @Override
    public void onMixControllerUpdate() {
        notifyDataSetChanged();
    }

    @Override
    public int getUserAdapterCount() {
        return mUserAdapter.getItemCount();
    }

    @Override
    public long getUserAdapterItemId(int position) {
        return mUserAdapter.getItemId(position);
    }

    @Override
    public int getUserAdapterViewTypeCount() {
        return Integer.MAX_VALUE - 1;
    }

    @Override
    public int getUserAdapterItemViewType(int position) {
        return mUserAdapter.getItemViewType(position);
    }

    @Override
    public RecyclerView.ViewHolder onCreateViewHolder(ViewGroup viewGroup, int type) {

        if (type == mMixController.getCampaignItemViewType()) {
            return createCampaignViewHolder(viewGroup);
        } else {
            return mUserAdapter.onCreateViewHolder(viewGroup, type);
        }
    }

    @Override
    public void onBindViewHolder(RecyclerView.ViewHolder viewHolder, int position) {

        if (viewHolder instanceof CampaignItemViewHolder) {
            bindCampaignViewHolder((CampaignItemViewHolder) viewHolder, position);
        } else {
            mUserAdapter.onBindViewHolder(viewHolder, mMixController.getUserAdapterPosition(position));
        }
    }

    @Override
    public void onAttachedToRecyclerView(RecyclerView recyclerView) {
        super.onAttachedToRecyclerView(recyclerView);
    }

    // ===========================================================
    // Methods
    // ===========================================================

    protected void start() {
        mRecycler.setAdapter(this);
    }

    private VH createCampaignViewHolder(ViewGroup viewGroup) {

        try {
            if (mViewHolderClass == CampaignItemWebHolder.class) {
                return mViewHolderClass.getDeclaredConstructor(ViewGroup.class, int.class, int.class).
                        newInstance(viewGroup, mWebItemWidth, mWebItemHeight);
            } else {
                return mViewHolderClass.getDeclaredConstructor(ViewGroup.class).
                        newInstance(viewGroup);
            }
        } catch (Exception e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        return null;
    }

    private void bindCampaignViewHolder(CampaignItemViewHolder viewHolder, int position) {

        Campaign item = mMixController.getCampaignItem(position);
        if (viewHolder != null && item != null) {
            viewHolder.bindData(item, position);
            viewHolder.itemView.setOnClickListener(mMixController.getTrackCampaignClickListener(viewHolder));
        }
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    /**
     * @return return adapter's {@link CampaignList}
     */
    public CampaignList getCampaigns() {
        return mMixController.getCampaigns();
    }

    /**
     * @return returns number of unread {@link Campaign} messages
     */
    public int getCurrentUnreadMessages() {
        return mMixController.getCurrentUnreadMessages();
    }

    /**
     * Used to set a listener to be notified when the response of the {@link WarplyInboxRequest}
     * is received
     *
     * @param listener the {@link SimpleCallbackReceiver} to be notified
     *                 of the {@link WarplyInboxRequest}'s response
     */
    public void setCampaignsListener(SimpleCallbackReceiver<CampaignList> listener) {
        mMixController.setCampaignsListener(listener);
    }

    /**
     * updates the native ads inside the {@link RecyclerView}
     * by a specific {@link WarplyInboxRequest}
     *
     * @param campaignsRequest the {@link WarplyInboxRequest} used to update the
     *                         native ads inside the {@link RecyclerView}
     */
    public void updateCampaigns(WarplyInboxRequest campaignsRequest) {
        mMixController.updateCampaigns(campaignsRequest);
    }

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

    public static class Builder<VH extends CampaignItemViewHolder> {

        private RecyclerView recycler;
        private WarplyInboxRequest campaignsRequest;
        private Class<VH> viewHolderClass;
        private RecyclerView.Adapter userAdapter;

        private int webItemWidth = 0;
        private int webItemHeight = 0;

        private boolean isRepeatableSet;
        private int startPosition;
        private int frequency;

        public CampaignsMixRecyclerAdapter adapt() {
            return new CampaignsMixRecyclerAdapter<>(this);
        }

        /**
         * CampaignsMixRecyclerAdapter Constructor
         *
         * @param recycler         the RecyclerView {@link RecyclerView} where will adapted data sets
         * @param campaignsRequest {@link WarplyInboxRequest} used to fetch campaigns for native ads.
         *                         In order to show native ads within the {@link RecyclerView} a
         *                         {@link ly.warp.sdk.io.request.WarplyInboxRequest.DisplayType} must be added
         *                         on the {@link WarplyInboxRequest} through {@code new WarplyInboxRequest().addDisplayTypeFilter(WarplyInboxRequest.DisplayType displayType)}
         * @param userAdapter      original {@link RecyclerView.Adapter} that will be mixed with native ads
         */
        public Builder(RecyclerView recycler, WarplyInboxRequest campaignsRequest,
                       RecyclerView.Adapter userAdapter) {
            this(recycler, campaignsRequest, (Class<VH>) CampaignItemWebHolder.class, userAdapter);
        }

        /**
         * CampaignsMixRecyclerAdapter Constructor
         * this constructor is not used often
         *
         * @param recycler         the RecyclerView {@link RecyclerView} where will adapted data sets
         * @param campaignsRequest {@link WarplyInboxRequest} used to fetch campaigns for native ads.
         *                         In order to show native ads within the {@link RecyclerView} a
         *                         {@link ly.warp.sdk.io.request.WarplyInboxRequest.DisplayType} must be added
         *                         on the {@link WarplyInboxRequest} through {@code new WarplyInboxRequest().addDisplayTypeFilter(DisplayType displayType)}
         * @param userAdapter      original {@link RecyclerView.Adapter} that will be mixed with native ads
         * @param webItemWidth     width of native ad item
         * @param webItemHeight    height of native ad item
         */
        public Builder(RecyclerView recycler, WarplyInboxRequest campaignsRequest,
                       RecyclerView.Adapter userAdapter, int webItemWidth, int webItemHeight) {
            this(recycler, campaignsRequest, userAdapter);
            this.webItemWidth = webItemWidth;
            this.webItemHeight = webItemHeight;
        }

        /**
         * CampaignsMixRecyclerAdapter Basic Constructor
         *
         * @param recycler         the RecyclerView {@link RecyclerView} where will adapted data sets
         * @param campaignsRequest {@link WarplyInboxRequest} used to fetch campaigns for native ads.
         *                         In order to show native ads within the {@link RecyclerView} a
         *                         {@link ly.warp.sdk.io.request.WarplyInboxRequest.DisplayType} must be added
         *                         on the {@link WarplyInboxRequest} through {@code new WarplyInboxRequest().addDisplayTypeFilter(DisplayType displayType)}
         * @param viewHolderClass  Class of the {@link RecyclerView.ViewHolder}
         *                         native ad item to be inflated inside {@link RecyclerView}
         * @param userAdapter      original {@link RecyclerView.Adapter} that will be mixed with native ads
         */
        public Builder(RecyclerView recycler, WarplyInboxRequest campaignsRequest,
                       Class<VH> viewHolderClass,
                       RecyclerView.Adapter userAdapter) {

            this.isRepeatableSet = false;
            this.startPosition = CampaignsMixController.CAMPAIGN_START_POSITION;
            this.frequency = CampaignsMixController.CAMPAIGN_FREQUENCY;

            this.recycler = recycler;
            this.campaignsRequest = campaignsRequest;
            this.viewHolderClass = viewHolderClass;
            this.userAdapter = userAdapter;
        }

        /**
         * set start position and frequency
         */
        public Builder setMixBounds(int startPosition, int frequency) {
            this.startPosition = startPosition;
            this.frequency = frequency;
            return this;
        }

        /**
         * @param isRepeatable true to repeat ads after the last ad is shown
         *                     false otherwise
         */
        public Builder setRepeatableSet(boolean isRepeatable) {
            this.isRepeatableSet = isRepeatable;
            return this;
        }
    }

}
