package ly.warp.sdk.views.adapters.mix;

import android.os.AsyncTask;
import androidx.annotation.NonNull;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import ly.warp.sdk.io.models.Campaign;
import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.managers.WarplyAnalyticsManager;

/**
 *  Helper class that used within the {@Link CampaignsMixController}
 *  and allow to track native campaigns events.
 *  Mix adapters monitor 2 types of events: view campaign in list and click on campaign item.
 */
public class CampaignsMixTracker {

    // ===========================================================
    // Constants
    // ===========================================================

    private enum TrackEvent {

        CAMPAIGN_VIEW("native_campaign_view"), CAMPAIGN_CLICK("native_campaign_click");

        private String type;

        TrackEvent(String type) {
            this.type = type;
        }

        public String getTrackMessage(@NonNull Campaign campaign) {
            return type + ":" + campaign.getSessionUUID();
        }
    }

    // ===========================================================
    // Fields
    // ===========================================================

    private CampaignsMixController mMixController;
    private int mOldFirstVisibleItem = -1;
    private HashSet<Integer> mVisibleCampaignIndexes;
    private Set<Integer> mNewCampaignIndexes;
    private Set<Integer> mUpdatedCampaignIndexes;

    // ===========================================================
    // Constructors
    // ===========================================================

    public CampaignsMixTracker(@NonNull CampaignsMixController mixController) {

        mMixController = mixController;
        mVisibleCampaignIndexes = new HashSet<>();
        mNewCampaignIndexes = new HashSet<>();
        mUpdatedCampaignIndexes = new HashSet<>();
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    // ===========================================================
    // Methods
    // ===========================================================

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    /**
     * call in onScroll for correct results
     *
     * @param firstVisiblePosition    - first visible position in global adapter
     * @param lastVisibleItemPosition - last visible position in global adapter
     */
    public void trackCampaignsView(int firstVisiblePosition, int lastVisibleItemPosition) {

        if (lastVisibleItemPosition >= firstVisiblePosition && mOldFirstVisibleItem != firstVisiblePosition) {
            mOldFirstVisibleItem = firstVisiblePosition;

                    /*Log.i("list_view_scroll", "first: " + firstVisibleItem + " visibleCount: " + visibleItemCount +
                            " last: " +lastVisibleItem);*/

            mNewCampaignIndexes.clear();
            mUpdatedCampaignIndexes.clear();
            for (int i = firstVisiblePosition; i < lastVisibleItemPosition + 1; i++) {
                if (mMixController.getItemViewType(i) == mMixController.getCampaignItemViewType()) {
                    // if new campaign index
                    if (mVisibleCampaignIndexes.add(i)) {
                        mNewCampaignIndexes.add(i);
                    } else {
                        mUpdatedCampaignIndexes.add(i);
                    }
                }
            }

            // retain only visible indexes
            mVisibleCampaignIndexes.clear();
            mVisibleCampaignIndexes.addAll(mUpdatedCampaignIndexes);
            mVisibleCampaignIndexes.addAll(mNewCampaignIndexes);

            // update new campaigns
            if (mNewCampaignIndexes.size() > 0) {
                new TrackCampaignsTask(mNewCampaignIndexes, TrackEvent.CAMPAIGN_VIEW).execute();
            }
        }
    }

    public void trackCampaignClick(Campaign campaign) {

        if (campaign != null) {
            new TrackCampaignsTask(campaign, TrackEvent.CAMPAIGN_CLICK).execute();
        }

    }

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

    private class TrackCampaignsTask extends AsyncTask<Void, Void, Void> {

        private List<Campaign> mmTrackCampaigns;
        private TrackEvent mmTrackEvent;

        private TrackCampaignsTask(Set<Integer> indexes, TrackEvent event) {

            mmTrackEvent = event;
            if (indexes != null && indexes.size() > 0) {
                mmTrackCampaigns = new ArrayList<>(indexes.size());
                for (int index : indexes) {
                    Campaign camp = mMixController.getCampaignItem(index);
                    if (camp != null) {
                        mmTrackCampaigns.add(camp);
                    }
                }
            }
        }

        private TrackCampaignsTask(Campaign campaign, TrackEvent event) {

            mmTrackEvent = event;
            if (campaign != null) {
                mmTrackCampaigns = new ArrayList<>(1);
                mmTrackCampaigns.add(campaign);
            }
        }

        @Override
        protected Void doInBackground(Void... params) {

            if (mmTrackCampaigns != null) {
                for (Campaign campaign : mmTrackCampaigns) {

                    WarplyAnalyticsManager.logEvent(null, mmTrackEvent.getTrackMessage(campaign), null);
                    WarpUtils.log("track_campaigns - " + mmTrackEvent.getTrackMessage(campaign) + " - " + campaign.getTitle());
                }
            }
            return null;
        }
    }
}
