package name.remal.building.gradle_plugins

import com.jfrog.bintray.gradle.BintrayExtension
import com.jfrog.bintray.gradle.BintrayUploadTask
import name.remal.building.gradle_plugins.dsl.default
import name.remal.building.gradle_plugins.utils.*
import name.remal.building.gradle_plugins.utils.PluginIds.BINTRAY_PLUGIN_IDS
import name.remal.building.gradle_plugins.utils.PluginIds.MAVEN_PUBLISH_PLUGIN_ID
import org.gradle.api.Project
import org.gradle.api.ProjectConfigurationException

@API
class BintraySettingsPlugin : ProjectPlugin() {

    override fun apply(project: Project) {
        project.withPlugin(BINTRAY_PLUGIN_IDS) {
            project.applyPlugin(MAVEN_PUBLISH_PLUGIN_ID)
            project.applyPlugin(PublishSettingsPlugin::class.java)

            setupCredentials(project)
            setupPublications(project)
            setupBintrayUpload(project)
            setupPackage(project)
            setupVersion(project)
        }
    }

    private fun setupCredentials(project: Project) {
        project[BintrayExtension::class.java].run {
            user = user ?: System.getenv("BINTRAY_USER")
            key = key ?: System.getenv("BINTRAY_API_KEY")
        }
    }

    private fun setupPublications(project: Project) {
        project[BintrayExtension::class.java].let { bintray ->
            bintray.setPublications(PublishSettingsPlugin.MAVEN_PUBLICATION_NAME)
        }
    }

    private fun setupBintrayUpload(project: Project) {
        project.tasks.configure(BintrayUploadTask::class.java) { task ->
            task.doFirst {
                if (task.user.isNullOrEmpty()) task.logger.warn("Bintray user not set or empty")
                if (task.apiKey.isNullOrEmpty()) task.logger.warn("Bintray API key not set or empty")
                if (task.repoName.isNullOrEmpty()) task.logger.warn("Bintray repository name not set or empty")
                if (task.packageName.isNullOrEmpty()) task.logger.warn("Bintray package name not set or empty")

                if (task.apiUrl?.contains("api.bintray.com") == true) {
                    if (task.versionName?.toString()?.toLowerCase()?.contains("snapshot") == true) {
                        throw IllegalStateException("Snapshots can't be uploaded to ${task.apiUrl}").let {
                            ProjectConfigurationException(it.message.default(), it)
                        }
                    }
                }
            }
        }
    }

    private fun setupPackage(project: Project) {
        project.afterEvaluate {
            project[BintrayExtension::class.java].let { bintray ->
                bintray.pkg.run {
                    name = name ?: project.group.toString().default()
                }
            }
        }
    }

    private fun setupVersion(project: Project) {
        project[BintrayExtension::class.java].let { bintray ->
            bintray.pkg.version.mavenCentralSync.run {
                user = user ?: System.getenv("OSS_USER")
                password = password ?: System.getenv("OSS_PASSWORD")
                close = "1"
                sync = !user.isNullOrEmpty() && !password.isNullOrEmpty()
            }
        }
    }

}
