package name.remal.building.gradle_plugins

import groovy.json.JsonSlurper
import name.remal.building.gradle_plugins.utils.ProjectPlugin
import name.remal.building.gradle_plugins.utils.getHttpBody
import org.gradle.api.DefaultTask
import org.gradle.api.Project
import org.gradle.api.plugins.HelpTasksPlugin
import org.gradle.api.tasks.TaskAction
import org.gradle.util.GradleVersion
import java.net.URL

@API
class CheckGradleUpdatesPlugin : ProjectPlugin() {

    companion object {
        const val CHECK_GRADLE_UPDATES_TASK_NAME = "checkGradleUpdates"
    }

    override fun apply(project: Project) {
        project.tasks.create(CHECK_GRADLE_UPDATES_TASK_NAME, GradleUpdatesTask::class.java)
    }

}


@API
class GradleUpdatesTask : DefaultTask() {

    init {
        group = HelpTasksPlugin.HELP_GROUP
        description = "Displays the if there is newer version of Gradle"

        onlyIf { !project.gradle.startParameter.isOffline }
    }

    @TaskAction
    fun doCheckGradleUpdates() {
        try {
            val currentVersion = GradleVersion.current()
            logger.info("Current Gradle version: {}", currentVersion.version)

            val info = getGradleCurrentVersionInfo()

            val latestVersion = GradleVersion.version(info["version"].toString())
            logger.info("Latest Gradle version: {}", latestVersion.version)

            if (currentVersion < latestVersion) {
                logger.warn("New Gradle version is available: {} (downloadUrl: {})", latestVersion.version, info["downloadUrl"])
            }

            didWork = true

        } catch (e: Exception) {
            logger.error(e.message, e)
        }
    }

    @Suppress("UNCHECKED_CAST")
    private fun getGradleCurrentVersionInfo(): Map<String, Any?> {
        val jsonContent = URL("https://services.gradle.org/versions/current").getHttpBody()
        return JsonSlurper().parseText(jsonContent) as Map<String, Any?>
    }

}
