package name.remal.building.gradle_plugins

import name.remal.building.gradle_plugins.artifact.ArtifactsCacheCleanerPlugin
import name.remal.building.gradle_plugins.dsl.*
import name.remal.building.gradle_plugins.utils.ProjectPlugin
import name.remal.building.gradle_plugins.utils.applyPlugin
import name.remal.building.gradle_plugins.utils.configureTasks
import org.eclipse.jgit.api.Git
import org.eclipse.jgit.lib.Constants.DOT_GIT
import org.eclipse.jgit.storage.file.FileRepositoryBuilder
import org.gradle.api.Project
import org.gradle.api.file.DuplicatesStrategy
import org.gradle.api.invocation.Gradle
import org.gradle.api.logging.configuration.ShowStacktrace
import org.gradle.api.tasks.AbstractCopyTask
import org.gradle.api.tasks.bundling.AbstractArchiveTask
import org.gradle.api.tasks.wrapper.Wrapper
import java.io.File
import kotlin.text.isNotEmpty

@API
class CommonSettingsPlugin : ProjectPlugin() {

    override fun apply(project: Project) {
        configureGradle(project.gradle)

        project.applyPlugin(ArtifactsCacheCleanerPlugin::class.java)

        configureWrapperTasks(project)
        configureCopyTasks(project)
        configureArchiveTasks(project)

        project.setupJavaExtensions()

        setSubprojectsGroup(project)
        setSubprojectsVersion(project)
    }

    private fun configureGradle(gradle: Gradle) {
        if (true == System.getenv("CI")?.toBoolean()) {
            gradle.startParameter.showStacktrace = ShowStacktrace.ALWAYS_FULL
        }
    }

    private fun configureWrapperTasks(project: Project) {
        project.configureTasks(Wrapper::class.java) { task ->
            task.doLast {
                task.propertiesFile.writeText(
                    task.propertiesFile.readLines().asSequence()
                        .map { it.substringBefore('#').trim() }
                        .filter(String::isNotEmpty)
                        .joinToString("\n")
                )

                var repositoryRootDir = project.rootDir.absoluteFile
                while (repositoryRootDir != repositoryRootDir.parentFile) {
                    if (repositoryRootDir.resolve(DOT_GIT).isDirectory) {
                        FileRepositoryBuilder.create(repositoryRootDir.resolve(DOT_GIT)).use useRepository@ { repository ->
                            Git(repository).use { git ->
                                val propertiesFileRelativePath = task.propertiesFile.absoluteFile.toRelativeString(repositoryRootDir).replace(File.separatorChar, '/')
                                val status = git.status().addPath(propertiesFileRelativePath).call()
                                if (status.isClean) {
                                    try {
                                        val archiveRelativePath = task.jarFile.absoluteFile.toRelativeString(repositoryRootDir).replace(File.separatorChar, '/')
                                        git.checkout().addPath(archiveRelativePath).call()
                                    } catch (ignored: Throwable) {
                                        // do nothing
                                    }
                                }
                            }
                        }
                        break
                    }
                    repositoryRootDir = repositoryRootDir.parentFile ?: break
                }
            }
        }
    }

    private fun configureCopyTasks(project: Project) {
        project.configureTasks(AbstractCopyTask::class.java) {
            it.duplicatesStrategy = DuplicatesStrategy.WARN
        }
    }

    private fun configureArchiveTasks(project: Project) {
        project.configureTasks(AbstractArchiveTask::class.java) {
            it.isReproducibleFileOrder = true
        }
    }

    private fun setSubprojectsGroup(project: Project) {
        project.setupSubprojectsConditionally(
            Project::isGroupSet,
            { subproject ->
                val groupTokens = mutableListOf<String>()
                var parent = subproject.parent
                while (null != parent) {
                    if (parent.isRootProject) {
                        if (parent.isGroupSet) {
                            groupTokens += parent.group.toString()
                        }
                        break
                    }
                    groupTokens += parent.name
                    parent = parent.parent
                }

                subproject.group = groupTokens.reversed().joinToString(".")
            }
        )
    }

    private fun setSubprojectsVersion(project: Project) {
        project.setupSubprojectsConditionally(
            Project::isVersionSet,
            { subproject ->
                subproject.version = project.version
            }
        )
    }

    private fun Project.setupSubprojectsConditionally(condition: Project.() -> Boolean, subprojectAction: (Project) -> Unit) {
        if (condition(this)) {
            subprojects {
                if (!condition(it)) {
                    subprojectAction(it)
                }
            }

        } else {
            afterEvaluate {
                if (condition(this)) {
                    subprojects {
                        if (!condition(it)) {
                            subprojectAction(it)
                        }
                    }
                }
            }
        }
    }

}
