package name.remal.building.gradle_plugins

import com.google.common.collect.BiMap
import com.google.common.collect.HashBiMap
import name.remal.building.gradle_plugins.coverage_metrics.CoverageMetricsAnnotationParser
import name.remal.building.gradle_plugins.coverage_metrics.CoverageMetricsExtension
import name.remal.building.gradle_plugins.dsl.Version
import name.remal.building.gradle_plugins.dsl.java
import name.remal.building.gradle_plugins.dsl.nullIfEmpty
import name.remal.building.gradle_plugins.utils.*
import name.remal.building.gradle_plugins.utils.BuildProperties.JACOCO_VERSION
import name.remal.building.gradle_plugins.utils.Constants.CLASS_FILE_NAME_SUFFIX
import name.remal.building.gradle_plugins.utils.PluginIds.JACOCO_PLUGIN_ID
import name.remal.building.gradle_plugins.utils.PluginIds.JAVA_PLUGIN_ID
import org.gradle.api.Project
import org.gradle.api.file.FileTreeElement
import org.gradle.api.plugins.JavaBasePlugin.CHECK_TASK_NAME
import org.gradle.api.plugins.JavaPlugin.TEST_TASK_NAME
import org.gradle.api.tasks.SourceSet
import org.gradle.api.tasks.SourceSet.TEST_SOURCE_SET_NAME
import org.gradle.testing.jacoco.plugins.JacocoPluginExtension
import org.gradle.testing.jacoco.tasks.JacocoCoverageVerification
import org.gradle.testing.jacoco.tasks.JacocoReportBase
import org.objectweb.asm.ClassReader
import org.objectweb.asm.tree.ClassNode
import org.slf4j.LoggerFactory
import java.io.File

@API
class CoverageMetricsPlugin : ProjectPlugin() {

    companion object {
        @JvmStatic private val logger = LoggerFactory.getLogger(CoverageMetricsPlugin::class.java)
        const val COVERAGE_METRICS_EXTENSION_NAME = "coverageMetrics"
    }

    override fun apply(project: Project) {
        val coverageMetricsExt = project.extensions.create(COVERAGE_METRICS_EXTENSION_NAME, CoverageMetricsExtension::class.java)

        project.withPlugin(JAVA_PLUGIN_ID) {
            project.applyPlugin(JACOCO_PLUGIN_ID) {
                setupExtension(project)
                setupTasks(project)
                setupExcludings(project, coverageMetricsExt)
                setupMinimumCoverage(project, coverageMetricsExt)
                setupTaskDependencies(project)
            }
        }
    }

    private fun setupExtension(project: Project) {
        project[JacocoPluginExtension::class.java].let { ext ->
            val toolVersion = ext.toolVersion?.let { Version.parseOrNull(it) }
            val buildVersion = Version.parseOrNull(JACOCO_VERSION)
            if (null != toolVersion && null != buildVersion) {
                if (toolVersion < buildVersion) {
                    ext.toolVersion = buildVersion.toString()
                }
            }
        }
    }

    private fun setupTasks(project: Project) {
        project.configureTasks(JacocoCoverageVerification::class.java) { task ->
            task.violationRules.isFailOnViolation = false
        }
    }

    private fun setupExcludings(project: Project, coverageMetricsExt: CoverageMetricsExtension) {
        project.configureTasks(JacocoReportBase::class.java) { task ->
            task.doFirstOrdered {
                val classesByAnnotations = coverageMetricsExt.excludings.classesByAnnotations.asSequence()
                    .map { CoverageMetricsAnnotationParser.parse(it) }
                    .toSet()
                if (classesByAnnotations.isEmpty()) return@doFirstOrdered

                val classes = mutableMapOf<String, File>()
                val outerClassNames = mutableMapOf<String, String?>()
                val excludedClasses: BiMap<File, String> = HashBiMap.create()
                val visitFiles: (FileTreeElement) -> Unit = visitFiles@ { element ->
                    if (element.isDirectory) return@visitFiles
                    val file = element.file
                    if (!file.name.endsWith(CLASS_FILE_NAME_SUFFIX)) return@visitFiles
                    val classNode = ClassNode().apply {
                        val bytecode = file.readBytes()
                        try {
                            ClassReader(bytecode).accept(OnlyClassInfoVisitor(this), 0)
                        } catch (e: Exception) {
                            logger.warn("Error processing $file: ${e.message}", e)
                            return@visitFiles
                        }
                    }

                    classes[classNode.name] = file
                    classNode.outerClass?.let { outerClassNames[classNode.name] = it }
                    classNode.innerClasses?.asSequence()
                        ?.filter { null != it.outerName && null != it.innerName }
                        ?.filter { it.outerName == classNode.name }
                        ?.map { it.outerName + '$' + it.innerName }
                        ?.forEach { outerClassNames[it] = classNode.name }

                    val hasExcludingAnnotations = classesByAnnotations.any { info ->
                        classNode.annotations.any { info.match(it) }
                    }
                    if (hasExcludingAnnotations) {
                        excludedClasses[file] = classNode.name
                    }
                }
                task.classDirectories?.apply { asFileTree.visit(visitFiles) }
                task.additionalClassDirs?.apply { asFileTree.visit(visitFiles) }

                outerClassNames.entries.iterator().let {
                    while (it.hasNext()) {
                        val (key, value) = it.next()
                        if (key == value) it.remove()
                    }
                }

                classes.keys.forEach { className ->
                    val packageName = className.substringBeforeLast('/', "").nullIfEmpty()
                    val simpleClassName = className.substringAfterLast('/')
                    if ("package-info" != simpleClassName && className !in outerClassNames) {
                        if (null != packageName) {
                            var packageClassName = packageName + "/package-info"
                            outerClassNames[className] = packageClassName

                            var curPackageName = packageName
                            while (null != curPackageName) {
                                curPackageName = curPackageName.substringBeforeLast('/', "").nullIfEmpty()
                                if (null != curPackageName) {
                                    outerClassNames[packageClassName] = curPackageName + "/package-info"
                                } else {
                                    outerClassNames[packageClassName] = "package-info"
                                }
                                packageClassName = outerClassNames[packageClassName]!!
                            }

                        } else {
                            outerClassNames[className] = "package-info"
                        }
                    }
                }

                fun isClassExcluded(className: String): Boolean {
                    if (className in excludedClasses.values) return true
                    val outerClass = outerClassNames[className] ?: return false
                    if (outerClass == className) return false
                    return isClassExcluded(outerClass)
                }
                classes.forEach { className, file ->
                    if (isClassExcluded(className)) {
                        excludedClasses[file] = className
                    }
                }

                task.filterAllClassDirectories { it.exclude { (it.file in excludedClasses) } }
            }
        }
    }

    private fun setupMinimumCoverage(project: Project, coverageMetricsExt: CoverageMetricsExtension) {
        project.afterEvaluateOrdered {
            project.configureTasks(JacocoCoverageVerification::class.java) { task ->
                coverageMetricsExt.minimumCoverage?.let { minimumCoverage ->
                    task.violationRules.rule {
                        it.limit {
                            it.minimum = minimumCoverage
                        }
                    }
                }
            }
        }
    }

    private fun setupTaskDependencies(project: Project) {
        project.afterEvaluateOrdered {
            val testTask = project.tasks.findByName(TEST_TASK_NAME) ?: return@afterEvaluateOrdered
            val reportTask = project.tasks.findByName(project.getJacocoReportTaskName()) ?: return@afterEvaluateOrdered
            val coverageVerificationTask = project.tasks.findByName(project.getJacocoCoverageVerificationTaskName()) ?: return@afterEvaluateOrdered
            val checkTask = project.tasks.findByName(CHECK_TASK_NAME) ?: return@afterEvaluateOrdered

            reportTask.dependsOn(testTask)
            coverageVerificationTask.dependsOn(reportTask)
            checkTask.dependsOn(coverageVerificationTask)
        }
    }

    private fun SourceSet.getJacocoReportTaskName() = getTaskName("jacoco", "Report")
    private fun Project.getJacocoReportTaskName() = project.java.sourceSets[TEST_SOURCE_SET_NAME].getJacocoReportTaskName()

    private fun SourceSet.getJacocoCoverageVerificationTaskName() = getTaskName("jacoco", "CoverageVerification")
    private fun Project.getJacocoCoverageVerificationTaskName() = project.java.sourceSets[TEST_SOURCE_SET_NAME].getJacocoCoverageVerificationTaskName()

}
