package name.remal.building.gradle_plugins

import name.remal.building.gradle_plugins.dsl.filterIsInstance
import name.remal.building.gradle_plugins.dynamic_dependencies.DynamicDependenciesExtension
import name.remal.building.gradle_plugins.dynamic_dependencies.DynamicDependency
import name.remal.building.gradle_plugins.utils.ProjectPlugin
import name.remal.building.gradle_plugins.utils.concurrentSetOf
import name.remal.building.gradle_plugins.utils.configure
import name.remal.building.gradle_plugins.utils.resolveCopyOf
import org.gradle.api.Project
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ExternalModuleDependency
import org.gradle.api.artifacts.component.ModuleComponentIdentifier
import org.slf4j.LoggerFactory
import java.util.stream.Collectors.toList

@API
class DynamicDependenciesPlugin : ProjectPlugin() {

    companion object {
        @JvmStatic private val logger = LoggerFactory.getLogger(DynamicDependenciesPlugin::class.java)
        const val DYNAMIC_DEPENDENCIES_EXTENSION_NAME = "dynamicDependencies"
    }

    override fun apply(project: Project) {
        val dynamicDependencies = project.extensions.create(DYNAMIC_DEPENDENCIES_EXTENSION_NAME, DynamicDependenciesExtension::class.java, project.dependencies)

        val processedConfigurations = concurrentSetOf<Configuration>()
        project.configurations.configure { conf ->
            conf.incoming.beforeResolve {
                if (!processedConfigurations.add(conf)) return@beforeResolve
                if (dynamicDependencies.dynamicDependencies.isEmpty()) return@beforeResolve
                logger.info("Processing dynamic dependencies for: {}", conf)

                val notMatchedDynamicDeps = mutableListOf<DynamicDependency>()
                val allDeps = conf.allDependencies.stream()
                    .filterIsInstance(ExternalModuleDependency::class.java)
                    .collect(toList())
                dynamicDependencies.dynamicDependencies.forEach { dynamicDep ->
                    val doHaveAllRequiredDeps = dynamicDep.checkInfos.all { check ->
                        allDeps.any { it.group == check.group && it.name == check.name }
                    }
                    if (doHaveAllRequiredDeps) {
                        logger.info("{} have all required dependencies {}, so add dynamic dependency: {}", conf, dynamicDep.checkInfos, dynamicDep.dependency)
                        conf.dependencies.add(dynamicDep.dependency)
                    } else {
                        notMatchedDynamicDeps.add(dynamicDep)
                    }
                }
                if (notMatchedDynamicDeps.isEmpty()) return@beforeResolve

                val resolvedDeps = project.configurations.resolveCopyOf(conf).stream()
                    .map { it.selected.id }
                    .filterIsInstance(ModuleComponentIdentifier::class.java)
                    .collect(toList())
                notMatchedDynamicDeps.forEach { dynamicDep ->
                    val doHaveAllRequiredDeps = dynamicDep.checkInfos.all { check ->
                        resolvedDeps.any { it.group == check.group && it.module == check.name }
                    }
                    if (doHaveAllRequiredDeps) {
                        logger.info("{} have all required dependencies {}, so add dynamic dependency: {}", conf, dynamicDep.checkInfos, dynamicDep.dependency)
                        conf.dependencies.add(dynamicDep.dependency)
                    }
                }
            }
        }
    }

}
