package name.remal.building.gradle_plugins

import name.remal.building.gradle_plugins.dsl.getOrCreateSourcesJarTask
import name.remal.building.gradle_plugins.dsl.java
import name.remal.building.gradle_plugins.utils.PluginIds.JAVA_PLUGIN_ID
import name.remal.building.gradle_plugins.utils.ProjectPlugin
import name.remal.building.gradle_plugins.utils.applyPlugin
import name.remal.building.gradle_plugins.utils.configure
import name.remal.building.gradle_plugins.utils.get
import org.gradle.api.Project
import org.gradle.api.artifacts.ProjectDependency
import org.gradle.api.file.FileTree
import org.gradle.api.plugins.JavaPlugin.JAR_TASK_NAME
import org.gradle.api.tasks.SourceSet.MAIN_SOURCE_SET_NAME
import org.gradle.api.tasks.bundling.AbstractArchiveTask
import org.slf4j.LoggerFactory
import java.util.concurrent.Callable

@API
class FlattenJavaProjectPlugin : ProjectPlugin() {

    companion object {
        @JvmStatic private val logger = LoggerFactory.getLogger(FlattenJavaProjectPlugin::class.java)
    }

    override fun apply(project: Project) {
        project.evaluationDependsOnChildren()

        project.applyPlugin(JAVA_PLUGIN_ID)
        project.applyPlugin(MergeJavaResourcesPlugin::class.java)

        this.setupConfigurations(project)
        this.setupTasks(project)
    }

    private fun setupConfigurations(project: Project) {
        val subprojectsToFlatten = project.subprojectsToFlatten

        val mainSourceSet = project.java.sourceSets[MAIN_SOURCE_SET_NAME]
        arrayOf(
            mainSourceSet.compileConfigurationName,
            mainSourceSet.apiConfigurationName,
            mainSourceSet.apiElementsConfigurationName,
            mainSourceSet.implementationConfigurationName,
            mainSourceSet.runtimeConfigurationName
        ).forEach { confName ->

            project.configurations.configure(confName) { conf ->
                subprojectsToFlatten.forEach { subproject ->
                    subproject.configurations.findByName(conf.name)?.let {
                        it.dependencies.all {
                            if (it is ProjectDependency && it.dependencyProject in subprojectsToFlatten) {
                                // do nothing
                            } else {
                                conf.dependencies.add(it)
                            }
                        }
                    }
                }
            }

        }
    }

    private fun setupTasks(project: Project) {
        val jar = project.tasks[JAR_TASK_NAME] as AbstractArchiveTask
        project.subprojectsToFlatten.forEach { subproject ->
            subproject.tasks.findByName(jar.name)?.let {
                jar.dependsOn(it)

                it as AbstractArchiveTask
                jar.from(Callable<FileTree> { project.zipTree(it.archivePath) }) {
                    it.exclude("META-INF/*.MF")
                    it.exclude("META-INF/*.SF")
                    it.exclude("META-INF/*.DSA")
                    it.exclude("META-INF/*.RSA")
                }
            }
        }


        val sourcesJar = project.getOrCreateSourcesJarTask()
        project.subprojectsToFlatten.forEach { subproject ->
            sourcesJar.from(subproject.java.sourceSets[MAIN_SOURCE_SET_NAME].allSource)
        }
    }

    private val Project.subprojectsToFlatten: Set<Project>
        get() = this.subprojects.asSequence()
            .filter { it.pluginManager.hasPlugin(JAVA_PLUGIN_ID) }
            .toSet()

}
