package name.remal.building.gradle_plugins

import name.remal.building.gradle_plugins.dsl.*
import name.remal.building.gradle_plugins.generate_sources.java.GenerateJavaSourcesTask
import name.remal.building.gradle_plugins.generate_sources.kotlin.GenerateKotlinSourcesTask
import name.remal.building.gradle_plugins.utils.*
import name.remal.building.gradle_plugins.utils.PluginIds.*
import org.gradle.api.Project
import org.gradle.api.tasks.compile.JavaCompile
import java.io.File

@API
class GenerateSourcesPlugin : ProjectPlugin() {

    override fun apply(project: Project) {
        project.withPlugin(JAVA_PLUGIN_ID) { setupJava(project) }
        project.withOneOfPlugin(KOTLIN_PLUGIN_IDS, KOTLIN2JS_PLUGIN_IDS) { setupKotlin(project) }
    }

    private fun setupJava(project: Project) {
        project.java.sourceSets.configure { sourceSet ->
            val generateSourcesTask = project.tasks.create(sourceSet.generateJavaSourcesTaskName, GenerateJavaSourcesTask::class.java) { task ->
                task.description = "Generates Java sources for ${sourceSet.name} source set"
                task.outputDir = File(project.buildDir, "generated/java/${sourceSet.name}").absoluteFile
                sourceSet.java.srcDir(task.outputDir)
            }
            project.tasks.configure(sourceSet.compileKotlinTaskName) { it.dependsOn(generateSourcesTask) }
            project.tasks.configure(sourceSet.compileKotlin2JSTaskName) { it.dependsOn(generateSourcesTask) }
            project.tasks.configure(sourceSet.compileJavaTaskName) { it.dependsOn(generateSourcesTask) }
            project.afterEvaluate {
                project.tasks[JavaCompile::class.java].findByName(sourceSet.compileJavaTaskName)?.let {
                    generateSourcesTask.textFilesEncoding = it.options.encoding
                }
            }
        }
    }

    private fun setupKotlin(project: Project) {
        project.java.sourceSets.configure { sourceSet ->
            val generateSourcesTask = project.tasks.create(sourceSet.generateKotlinSourcesTaskName, GenerateKotlinSourcesTask::class.java) { task ->
                task.description = "Generates Kotlin sources for ${sourceSet.name} source set"
                task.outputDir = File(project.buildDir, "generated/kotlin/${sourceSet.name}").absoluteFile
                sourceSet.kotlin.kotlin.srcDir(task.outputDir)
            }
            project.tasks.configure(sourceSet.compileKotlinTaskName) { it.dependsOn(generateSourcesTask) }
            project.tasks.configure(sourceSet.compileKotlin2JSTaskName) { it.dependsOn(generateSourcesTask) }
            project.afterEvaluate {
                project.tasks[JavaCompile::class.java].findByName(sourceSet.compileJavaTaskName)?.let {
                    generateSourcesTask.textFilesEncoding = it.options.encoding
                }
            }
        }
    }

}
