package name.remal.building.gradle_plugins

import name.remal.building.gradle_plugins.dsl.isRootProject
import name.remal.building.gradle_plugins.utils.*
import name.remal.building.gradle_plugins.utils.PluginIds.IDEA_PLUGIN_ID
import org.gradle.api.Project
import org.gradle.plugins.ide.idea.model.IdeaModel

@API
class IdeaExtendedSettingsPlugin : ProjectPlugin() {

    override fun apply(project: Project) {
        if (!project.isRootProject) return
        project.applyPlugin(IDEA_PLUGIN_ID)
        project[IdeaModel::class.java].setup(
            this::setupEntryPointsManager,
            this::setupNullableNotNullManager,
            this::clearEncodings
        )
    }

    private fun setupEntryPointsManager(ideaModel: IdeaModel) {
        val componentName = "EntryPointsManager"
        ideaModel.writeComponentIfNotExists(componentName, IdeaConfigType.IPR, "misc.xml", { it.appendElement("project", mapOf("version" to IDEA_CONFIG_PROJECT_VERSION)) })

        ideaModel.processConfigs { document ->
            val entryPointsManager = document.findIdeaComponentElement(componentName) ?: return@processConfigs

            entryPointsManager.addIdeaListStringValuesIfNotContains(listOf(
                "org.gradle.api.tasks.TaskAction"
                , "org.immutables.metainf.Metainf.Service"
                , "com.google.auto.service.AutoService"
                , "org.mangosdk.spi.ProviderFor"
                , "org.junit.Test"
                , "org.testng.annotations.Test"
                , "javax.inject.Inject"
                , "javax.inject.PostConstruct"
                , "org.springframework.stereotype.Component"
                , "org.springframework.stereotype.Controller"
                , "org.springframework.stereotype.Repository"
                , "org.springframework.stereotype.Service"
                , "org.springframework.web.bind.annotation.RequestMapping"
                , "org.springframework.web.bind.annotation.RestController"
            ))

            val writeAnnotations = entryPointsManager.findOrAppendElement("writeAnnotations")
            arrayOf(
                "javax.annotation.Resource",
                "javax.inject.Inject",
                "org.springframework.beans.factory.annotation.Autowired",
                "org.springframework.beans.factory.annotation.Value"
            ).forEach { value ->
                writeAnnotations.findOrAppendElement("writeAnnotation", mapOf("name" to value))
            }
        }
    }

    private fun setupNullableNotNullManager(ideaModel: IdeaModel) {
        val componentName = "NullableNotNullManager"
        ideaModel.writeComponentIfNotExists(componentName, IdeaConfigType.IPR, "misc.xml", { it.appendElement("project", mapOf("version" to IDEA_CONFIG_PROJECT_VERSION)) })

        ideaModel.processConfigs { document ->
            val nullableNotNullManager = document.findIdeaComponentElement(componentName) ?: return@processConfigs

            nullableNotNullManager.findOrAppendElement("option", mapOf("name" to "myDefaultNullable")) {
                it.setAttribute("value", "javax.annotation.Nullable")
            }

            nullableNotNullManager.findOrAppendElement("option", mapOf("name" to "myNullables")).findOrAppendElement("value")
                .addIdeaListStringValuesIfNotContains(listOf(
                    "org.jetbrains.annotations.Nullable"
                    , "javax.annotation.Nullable"
                    , "javax.annotation.CheckForNull"
                    , "org.springframework.lang.Nullable"
                    , "edu.umd.cs.findbugs.annotations.Nullable"
                    , "android.support.annotation.Nullable"
                    , "org.eclipse.jdt.annotation.Nullable"
                    , "org.gradle.api.Nullable"
                    , "javax.validation.constraints.Null"
                    , "javax.validation.constraints.Null.List"
                ))

            nullableNotNullManager.findOrAppendElement("option", mapOf("name" to "myDefaultNotNull")) {
                it.setAttribute("value", "javax.annotation.Nonnull")
            }

            nullableNotNullManager.findOrAppendElement("option", mapOf("name" to "myNotNulls")).findOrAppendElement("value")
                .addIdeaListStringValuesIfNotContains(listOf(
                    "org.jetbrains.annotations.NotNull"
                    , "javax.annotation.Nonnull"
                    , "edu.umd.cs.findbugs.annotations.NonNull"
                    , "android.support.annotation.NonNull"
                    , "org.eclipse.jdt.annotation.NonNull"
                    , "lombok.NonNull"
                    , "javax.validation.constraints.NotNull"
                    , "javax.validation.constraints.NotNull.List"
                    , "org.hibernate.validator.constraints.NotBlank"
                    , "org.hibernate.validator.constraints.NotBlank.List"
                    , "org.hibernate.validator.constraints.NotEmpty"
                    , "org.hibernate.validator.constraints.NotEmpty.List"
                ))
        }
    }

    private fun clearEncodings(ideaModel: IdeaModel) {
        ideaModel.processConfigs { document ->
            val encodingComponent = document.findIdeaComponentElement("Encoding") ?: return@processConfigs

            val projectUrl = "PROJECT"
            val projectCharset = encodingComponent.findElement("file", mapOf("url" to projectUrl))?.getAttribute("charset") ?: return@processConfigs

            encodingComponent.getChildElements("file").forEach {
                if (projectUrl == it.getAttribute("url")) return@forEach
                if (projectCharset == it.getAttribute("charset")) {
                    it.remove()
                }
            }
        }
    }

}
