package name.remal.building.gradle_plugins

import name.remal.building.gradle_plugins.artifact.ArtifactsCacheCleanerPlugin
import name.remal.building.gradle_plugins.artifact.CachedArtifactsCollection
import name.remal.building.gradle_plugins.dsl.isTestFrameworkSet
import name.remal.building.gradle_plugins.dsl.java
import name.remal.building.gradle_plugins.dsl.setupJavaExtensions
import name.remal.building.gradle_plugins.utils.*
import name.remal.building.gradle_plugins.utils.Constants.*
import name.remal.building.gradle_plugins.utils.PluginIds.JAVA_PLUGIN_ID
import org.gradle.api.Project
import org.gradle.api.artifacts.Configuration
import org.gradle.api.tasks.compile.JavaCompile
import org.gradle.api.tasks.testing.Test
import org.gradle.api.tasks.testing.junit.JUnitOptions
import org.gradle.api.tasks.testing.logging.TestExceptionFormat.FULL
import org.gradle.api.tasks.testing.logging.TestLogEvent
import org.gradle.api.tasks.testing.logging.TestLogEvent.FAILED
import org.gradle.api.tasks.testing.logging.TestLogEvent.STANDARD_ERROR
import org.gradle.api.tasks.testing.testng.TestNGOptions
import org.gradle.jvm.tasks.Jar

@API
class JavaSettingsPlugin : ProjectPlugin() {

    companion object {
        const val COMPILE_ONLY_All_CONFIGURATION_NAME = "compileOnlyAll"
    }

    override fun apply(project: Project) {
        project.applyPlugin(ArtifactsCacheCleanerPlugin::class.java)

        project.setupJavaExtensions()
        project.withPlugin(JAVA_PLUGIN_ID) {
            setupRepositories(project)
            setupConfigurations(project)
            setupCompatibility(project)
            setupJavaCompile(project)
            setupTesting(project)
            setupCoverageMetrics(project)
            setupJars(project)
        }
    }

    private fun setupRepositories(project: Project) {
        project.repositories.apply {
            if (isEmpty()) {
                mavenLocal()
                jcenter()
            }
        }
    }

    private fun setupConfigurations(project: Project) {
        val compileOnlyAll = project.configurations.create(COMPILE_ONLY_All_CONFIGURATION_NAME) {
            it.description = "All compile-only configurations extends this configuration"
        }

        project.java.sourceSets.configure { sourceSet ->
            project.configurations.configure(sourceSet.compileOnlyConfigurationName) { configuration ->
                configuration.extendsFrom(compileOnlyAll)
            }
        }

        project.java.sourceSets.configure { sourceSet ->
            val compileConf = project.configurations[sourceSet.compileConfigurationName]
            project.configurations.getOrCreate(sourceSet.apiConfigurationName).extendsFrom(compileConf)
            project.configurations.getOrCreate(sourceSet.apiElementsConfigurationName).extendsFrom(compileConf)
            project.configurations.getOrCreate(sourceSet.implementationConfigurationName).extendsFrom(compileConf)
        }
    }

    private fun setupCompatibility(project: Project) {
        project.java.sourceCompatibility = DEFAULT_SOURCE_COMPATIBILITY
        project.java.targetCompatibility = DEFAULT_TARGET_COMPATIBILITY
    }

    private fun setupJavaCompile(project: Project) {
        project.configureTasks(JavaCompile::class.java) {
            it.options.apply {
                isDebug = true
                debugOptions.debugLevel = "source,lines,vars"
                isDeprecation = true
                encoding = encoding ?: ENCODING
            }
        }
    }

    private fun setupTesting(project: Project) {
        project.configureTasks(Test::class.java) { task ->
            task.enableAssertions = true

            task.testLogging.apply {
                showStackTraces = true
                setExceptionFormat(FULL as Any) // Gradle 3.5 compatibility
                events(FAILED, STANDARD_ERROR)
                info.events(*TestLogEvent.values())
            }
        }

        project.afterEvaluateOrdered {
            project.configureTasks(Test::class.java) { task ->
                if (task.isTestFrameworkSet) return@configureTasks

                val testFramework: TestFramework? = run {
                    val classpath = task.classpath
                    if (classpath is Configuration) {
                        val allDependencies = classpath.allDependencies
                        if (allDependencies.any { "org.testng" == it.group && "testng" == it.name }) {
                            return@run TestFramework.TESTNG
                        } else if (allDependencies.any { "junit" == it.group && "junit" == it.name }) {
                            return@run TestFramework.JUNIT
                        }
                    }

                    val classpathArtifacts = CachedArtifactsCollection(classpath)
                    if (classpathArtifacts.containsClass("org.testng.annotations.Test")) {
                        return@run TestFramework.TESTNG
                    } else if (classpathArtifacts.containsClass("org.junit.Test")) {
                        return@run TestFramework.JUNIT
                    }

                    return@run null
                }

                if (TestFramework.TESTNG == testFramework) {
                    task.useTestNG()
                    task.maxParallelForks = 1
                    (task.options as TestNGOptions).apply {
                        parallel = "methods"
                        threadCount = Runtime.getRuntime().availableProcessors()
                    }

                } else if (TestFramework.JUNIT == testFramework) {
                    task.useJUnit()
                    task.maxParallelForks = Runtime.getRuntime().availableProcessors()
                    (task.options as JUnitOptions).apply {
                    }
                }
            }
        }
    }

    private fun setupCoverageMetrics(project: Project) {
        project.applyPlugin(CoverageMetricsPlugin::class.java)
    }

    private fun setupJars(project: Project) {
        project.afterEvaluate {
            project.tasks.configure(Jar::class.java) {
                val attributes = it.manifest.attributes
                val moduleNameAttribute = "Automatic-Module-Name"
                if (moduleNameAttribute !in attributes) {
                    val moduleName = project.id
                    it.inputs.property(JavaSettingsPlugin::class.java.name + ':' + moduleNameAttribute, moduleName)
                    attributes[moduleNameAttribute] = moduleName
                }
            }
        }
    }

    private enum class TestFramework {
        TESTNG,
        JUNIT,
    }

}
