package name.remal.building.gradle_plugins

import name.remal.building.gradle_plugins.dsl.*
import name.remal.building.gradle_plugins.utils.*
import org.gradle.api.DefaultTask
import org.gradle.api.Project
import org.gradle.api.Project.DEFAULT_BUILD_DIR_NAME
import org.gradle.api.plugins.JavaBasePlugin.BUILD_TASK_NAME
import org.gradle.api.plugins.ReportingBasePlugin
import org.gradle.api.reporting.ReportingExtension
import org.gradle.api.reporting.ReportingExtension.DEFAULT_REPORTS_DIR_NAME
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.OutputFile
import org.gradle.api.tasks.SkipWhenEmpty
import org.gradle.api.tasks.TaskAction
import org.gradle.initialization.DefaultSettings.DEFAULT_BUILD_SRC_DIR
import org.gradle.testing.jacoco.tasks.JacocoReport
import org.gradle.testing.jacoco.tasks.JacocoReportBase
import java.io.File
import java.util.*
import javax.xml.parsers.DocumentBuilder
import javax.xml.parsers.DocumentBuilderFactory
import javax.xml.parsers.ParserConfigurationException

@API
class MergedCoverageMetricsPlugin : ProjectPlugin() {

    companion object {
        const val REPORT_FILE_RELATIVE_PATH = "merged-coverage-metrics.properties"
        const val COLLECT_MERGED_COVERAGE_METRICS_TASK_NAME = "collectMergedCoverageMetrics"
        const val DISPLAY_MERGED_COVERAGE_METRICS_TASK_NAME = "displayMergedCoverageMetrics"
    }

    override fun apply(project: Project) {
        if (!project.isRootProject) return

        project.pluginManager.apply(ReportingBasePlugin::class.java)
        val mergedCoverageMetricsPropsFile = project[ReportingExtension::class.java].baseDir.resolve(REPORT_FILE_RELATIVE_PATH)


        val collectTask = project.tasks.create(COLLECT_MERGED_COVERAGE_METRICS_TASK_NAME, CollectMergedCoverageMetricsTask::class.java) {
            it.mergedCoverageMetricsPropsFile = mergedCoverageMetricsPropsFile

            if (project.isBuildSrcProject) {
                project.tasks[BUILD_TASK_NAME].dependsOn(it)
            }
        }


        val tempExecutionDataFile = createTempFile(project.id + ".", ".empty-jacoco-execution").apply(File::deleteOnExit)
        project.allprojects { proj ->
            proj.applyPlugin(CoverageMetricsPlugin::class.java)

            proj.afterEvaluateOrdered(Int.MAX_VALUE) {
                proj.tasks[JacocoReportBase::class.java].all { task ->
                    /*
                     * JacocoReportBase skips execution if all executionData files exist.
                     * So let's add one executionData file that exists.
                     */
                    task.executionData = (task.executionData + proj.files(tempExecutionDataFile)).filter(File::exists)

                    if (task is JacocoReport) {
                        task.reports.xml.isEnabled = true // We will parse XML reports, so enable it.
                        collectTask.jacocoXmlReportFiles.add(task.reports.xml.destination)
                        collectTask.dependsOn(task)
                    }
                }
            }
        }
        project.evaluationDependsOnChildren()


        project.tasks.create(DISPLAY_MERGED_COVERAGE_METRICS_TASK_NAME, DisplayMergedCoverageMetricsTask::class.java) {
            it.dependsOn(collectTask)

            it.mergedCoverageMetricsPropsFiles.add(mergedCoverageMetricsPropsFile)

            if (!project.isBuildSrcProject) {
                val buildSrcPropsFile = project.rootDir.resolve("$DEFAULT_BUILD_SRC_DIR/$DEFAULT_BUILD_DIR_NAME/$DEFAULT_REPORTS_DIR_NAME/$REPORT_FILE_RELATIVE_PATH")
                it.mergedCoverageMetricsPropsFiles.add(buildSrcPropsFile)
            }
        }
    }

}


private const val COVERED_LINES_PROPERTY_NAME = "covered"
private const val MISSED_LINES_PROPERTY_NAME = "missed"


@API
class CollectMergedCoverageMetricsTask : DefaultTask() {

    init {
        description = "Collect merged coverage metrics"
    }

    @InputFiles
    @SkipWhenEmpty
    val jacocoXmlReportFiles = mutableListOf<File>()

    @OutputFile
    lateinit var mergedCoverageMetricsPropsFile: File

    @TaskAction
    fun doCollect() {
        var coveredLines = 0L
        var missedLines = 0L
        jacocoXmlReportFiles.forEach forEachReportFile@ { jacocoXmlReportFile ->
            if (!jacocoXmlReportFile.isFile) return@forEachReportFile
            logger.debug("Processing Jacoco XML report file: {}", jacocoXmlReportFile)
            documentBuilder.parse(jacocoXmlReportFile)
                .documentElement
                ?.getChildElements("counter")
                ?.forEach forEachCounter@ { counterNode ->
                    if (true != counterNode.getAttribute("type")?.equals("LINE", true)) return@forEachCounter
                    val curCoveredLines = counterNode.getAttribute("covered")?.toLongOrNull()
                    val curMissedLines = counterNode.getAttribute("missed")?.toLongOrNull()
                    if (null != curCoveredLines && null != curMissedLines) {
                        coveredLines += curCoveredLines
                        missedLines += curMissedLines
                    } else {
                        logger.warn("Unsupported report node: {}", counterNode.asString(true))
                    }
                }
        }

        Properties().apply {
            put(COVERED_LINES_PROPERTY_NAME, coveredLines.toString())
            put(MISSED_LINES_PROPERTY_NAME, missedLines.toString())
            store(mergedCoverageMetricsPropsFile.forceCreate().createParentDirectories())
        }
    }

}


@API
class DisplayMergedCoverageMetricsTask : DefaultTask() {

    init {
        description = "Display merged coverage metrics"
    }

    @InputFiles
    @SkipWhenEmpty
    val mergedCoverageMetricsPropsFiles = mutableListOf<File>()

    @TaskAction
    fun doDisplay() {
        var coveredLines = 0L
        var missedLines = 0L
        mergedCoverageMetricsPropsFiles.forEach { propsFile ->
            if (!propsFile.exists()) return@forEach
            logger.debug("Processing merged coverage metrics properties file: {}", propsFile)
            loadProperties(propsFile).apply {
                coveredLines += this[COVERED_LINES_PROPERTY_NAME]?.toString()?.toLongOrNull() ?: 0L
                missedLines += this[MISSED_LINES_PROPERTY_NAME]?.toString()?.toLongOrNull() ?: 0L
            }
        }


        val totalLines = coveredLines + missedLines
        if (0L == totalLines) return
        logger.warn(
            "Merged code coverage: {}% ({} of {} lines)",
            100 * coveredLines / totalLines,
            coveredLines,
            totalLines
        )
    }

}


private val documentBuilder: DocumentBuilder = DocumentBuilderFactory.newInstance()
    .apply {
        isValidating = false
        isXIncludeAware = false
        isNamespaceAware = false
        isCoalescing = true
        isIgnoringComments = true

        try {
            setFeature("http://apache.org/xml/features/nonvalidating/load-dtd-grammar", false)
        } catch (e: ParserConfigurationException) {
            // do nothing
        }
        try {
            setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false)
        } catch (e: ParserConfigurationException) {
            // do nothing
        }
    }
    .newDocumentBuilder()
    .apply {
        setEntityResolver { _, _ -> null }
    }
