package name.remal.building.gradle_plugins

import name.remal.building.gradle_plugins.dsl.default
import name.remal.building.gradle_plugins.dsl.getOrCreateSourcesJarTask
import name.remal.building.gradle_plugins.dsl.java
import name.remal.building.gradle_plugins.utils.*
import name.remal.building.gradle_plugins.utils.PluginIds.JAVA_PLUGIN_ID
import name.remal.building.gradle_plugins.utils.PluginIds.MAVEN_PUBLISH_PLUGIN_ID
import org.gradle.api.Project
import org.gradle.api.artifacts.Dependency.ARCHIVES_CONFIGURATION
import org.gradle.api.plugins.BasePlugin.BUILD_GROUP
import org.gradle.api.plugins.JavaBasePlugin.BUILD_TASK_NAME
import org.gradle.api.plugins.JavaPlugin.JAR_TASK_NAME
import org.gradle.api.publish.PublishingExtension
import org.gradle.api.publish.maven.MavenPublication
import org.gradle.api.publish.maven.plugins.MavenPublishPlugin.PUBLISH_LOCAL_LIFECYCLE_TASK_NAME
import org.gradle.api.tasks.SourceSet.MAIN_SOURCE_SET_NAME
import org.gradle.api.tasks.SourceSet.TEST_SOURCE_SET_NAME
import org.gradle.api.tasks.bundling.Jar
import org.gradle.internal.component.external.model.ModuleComponentArtifactIdentifier
import org.w3c.dom.Element

@API
class PublishSettingsPlugin : ProjectPlugin() {

    companion object {
        const val BUILD_INSTALL_TASK_NAME = "buildInstall"
        const val MAVEN_PUBLICATION_NAME = "mavenArchivesAuto"
        const val PUBLISH_SETTINGS_EXTENSION_NAME = "publishSettings"

        const val MAVEN_CENTRAL_RELEASES_DEPLOY_REPOSITORY_URL = "https://oss.sonatype.org/service/local/staging/deploy/maven2/"
        const val MAVEN_CENTRAL_SNAPSHOTS_DEPLOY_REPOSITORY_URL = "https://oss.sonatype.org/content/repositories/snapshots/"
    }

    override fun apply(project: Project) {
        project.extensions.create(PUBLISH_SETTINGS_EXTENSION_NAME, PublishSettingsExtension::class.java)

        project.withPlugin(MAVEN_PUBLISH_PLUGIN_ID) {
            project.applyPlugin(JAVA_PLUGIN_ID)

            project.artifacts.add(ARCHIVES_CONFIGURATION, project.tasks[JAR_TASK_NAME])
            project.artifacts.add(ARCHIVES_CONFIGURATION, project.getOrCreateSourcesJarTask())

            project.applyPlugin(SigningSettingsPlugin::class.java)

            setupMavenPublish(project)
        }
    }

    private fun setupMavenPublish(project: Project) {
        project.tasks[PUBLISH_LOCAL_LIFECYCLE_TASK_NAME].dependsOn(project.tasks[BUILD_TASK_NAME])

        project.tasks.create(BUILD_INSTALL_TASK_NAME) {
            it.group = BUILD_GROUP
            it.dependsOn(project.tasks[BUILD_TASK_NAME])
            it.dependsOn(project.tasks[PUBLISH_LOCAL_LIFECYCLE_TASK_NAME])
        }

        project[PublishingExtension::class.java].apply {
            publications.configure(MavenPublication::class.java) { task ->
                task.pom.withXml { setupPomRequiredFields(project, it.asElement()) }
                task.pom.withXml { setupPomDependencies(project, it.asElement()) }
            }

            project.afterEvaluateOrdered {
                if (repositories.isNotEmpty()) return@afterEvaluateOrdered
                repositories.maven { maven ->
                    maven.setUrl(if (project.version.toString().contains("snapshot", true)) {
                        MAVEN_CENTRAL_SNAPSHOTS_DEPLOY_REPOSITORY_URL
                    } else {
                        MAVEN_CENTRAL_RELEASES_DEPLOY_REPOSITORY_URL
                    })
                    maven.credentials {
                        it.username = getEnvOrNull("OSS_USER")
                        it.password = getEnvOrNull("OSS_PASSWORD")
                    }
                }
            }
        }

        project[PublishingExtension::class.java].apply {
            publications.create(MAVEN_PUBLICATION_NAME, MavenPublication::class.java).let { publication ->
                project.afterEvaluateOrdered {
                    project.components.all { publication.from(it) }

                    val archivesConf = project.configurations[ARCHIVES_CONFIGURATION]
                    archivesConf.allArtifacts.all { artifact ->
                        if (publication.artifacts.any { it.extension.default() == artifact.extension.default() && it.classifier.default() == artifact.classifier.default() }) return@all
                        publication.artifact(artifact)
                    }

                    if (publication.artifacts.none { "javadoc" == it.classifier }) {
                        publication.artifact(project.tasks.create("emptyJavadocJar", Jar::class.java) {
                            it.classifier = "javadoc"
                        })
                    }
                }
            }
        }
    }

    private fun setupPomRequiredFields(project: Project, root: Element) {
        project[PublishSettingsExtension::class.java].run {
            root.findOrAppendElement("name") { it += displayName ?: project.displayName ?: project.name }
            root.findOrAppendElement("description") { it += description ?: "" }
            root.findOrAppendElement("url") { it += url ?: "" }
            root.findOrAppendElement("scm") {
                it.appendElement("url") += scmUrl ?: ""
            }
            root.findOrAppendElement("licenses") { licensesNode ->
                val licenses = licenses.toMutableList()
                if (licenses.isEmpty()) {
                    licenses += PublishSettingsExtension.LicenseInfo("Unlicense", "http://unlicense.org/")
                }
                licenses.forEach { license ->
                    licensesNode.appendElement("license").apply {
                        appendElement("name") += license.name
                        appendElement("url") += license.url
                        appendElement("distribution") += license.distribution
                    }
                }
            }
            root.findOrAppendElement("developers") { developersNode ->
                developers.forEach { developer ->
                    developersNode.appendElement("developer").apply {
                        appendElement("id") += developer.id
                        appendElement("name") += developer.name
                        appendElement("email") += developer.email
                    }
                }
            }
        }
    }

    private fun setupPomDependencies(project: Project, root: Element) {
        compileXPath("//dependencies/dependency[not(version) or not(version/text()) or contains(version/text(), '+')]")
            .evaluateAsNodeList(root)
            .forEach forEachDependency@ { dependency ->
                dependency as Element

                val groupId = compileXPath("./groupId/text()").evaluateAsString(dependency)
                val artifactId = compileXPath("./artifactId/text()").evaluateAsString(dependency)

                val scope = compileXPath("./scope/text()").evaluateAsString(dependency)
                val configuration = when (scope) {
                    "system" -> return@forEachDependency
                    "import" -> return@forEachDependency
                    "test" -> project.configurations[project.java.sourceSets[TEST_SOURCE_SET_NAME].compileClasspathConfigurationName]
                    "provided" -> project.configurations[project.java.sourceSets[MAIN_SOURCE_SET_NAME].compileClasspathConfigurationName]
                    "runtime" -> project.configurations[project.java.sourceSets[MAIN_SOURCE_SET_NAME].runtimeClasspathConfigurationName]
                    else -> project.configurations[project.java.sourceSets[MAIN_SOURCE_SET_NAME].compileClasspathConfigurationName]
                }

                configuration.incoming.artifacts.forEach forEachArtifact@ { artifact ->
                    val id = artifact.id as? ModuleComponentArtifactIdentifier ?: return@forEachArtifact
                    val componentId = id.componentIdentifier
                    if (componentId.group == groupId && componentId.module == artifactId) {
                        dependency.getElementsByTagName("version").forEach { it.parentNode.removeChild(it) }
                        dependency.appendChild(
                            root.ownerDocument.createElement("version").apply {
                                appendChild(
                                    root.ownerDocument.createTextNode(componentId.version)
                                )
                            }
                        )
                        return@forEachDependency
                    }
                }
            }
    }

}


@API
class PublishSettingsExtension {

    var displayName: String? = null

    var description: String? = null

    var url: String? = null
        get() = field ?: scmUrl

    var scmUrl: String? = null

    val licenses: MutableSet<LicenseInfo> = mutableSetOf()

    fun addLicense(name: String, url: String) {
        licenses += LicenseInfo(name, url)
    }

    fun addLicense(name: String, url: String, distribution: String) {
        licenses += LicenseInfo(name, url, distribution)
    }

    data class LicenseInfo(
        var name: String,
        var url: String,
        var distribution: String = "repo"
    )

    val developers: MutableSet<DeveloperInfo> = mutableSetOf()

    fun addDeveloper(name: String, email: String) {
        developers += DeveloperInfo(name, email)
    }

    fun addDeveloper(id: String, name: String, email: String) {
        developers += DeveloperInfo(name, email, id)
    }

    data class DeveloperInfo(
        var name: String,
        var email: String,
        var id: String = email
    )

}
