package name.remal.building.gradle_plugins

import name.remal.building.gradle_plugins.dsl.default
import name.remal.building.gradle_plugins.dsl.escapeRegex
import name.remal.building.gradle_plugins.utils.Ordered
import name.remal.building.gradle_plugins.utils.ProjectPlugin
import name.remal.building.gradle_plugins.utils.RESOURCES_CLASS_LOADER
import name.remal.building.gradle_plugins.utils.configure
import org.gradle.api.DefaultTask
import org.gradle.api.Project
import org.gradle.api.internal.plugins.PluginRegistry
import org.gradle.api.plugins.HelpTasksPlugin.HELP_GROUP
import org.gradle.api.tasks.TaskAction
import org.gradle.internal.logging.text.StyledTextOutputFactory
import java.util.*
import javax.inject.Inject
import kotlin.text.RegexOption.IGNORE_CASE

private val versionSelectors: List<VersionSelector> by lazy { ServiceLoader.load(VersionSelector::class.java, RESOURCES_CLASS_LOADER).toList().sorted() }

@API
class VersionSelectionsPlugin : ProjectPlugin() {

    companion object {
        const val VERSION_SELECTIONS_HELP_TASK_NAME = "versionSelectionsPluginHelp"
    }

    override fun apply(project: Project) {
        project.configurations.configure {
            it.resolutionStrategy {
                it.eachDependency {
                    with(it) {

                        val version = requested.version
                        if (!version.isNullOrEmpty() && Project.DEFAULT_VERSION != version) return@eachDependency

                        val group = requested.group.default()
                        val name = requested.name.default()
                        versionSelectors.forEach {
                            it.getVersion(project, group, name)?.let { versionToUse ->
                                useVersion(versionToUse)
                                return@eachDependency
                            }
                        }

                    }
                }
            }
        }


        project.tasks.create(VERSION_SELECTIONS_HELP_TASK_NAME, VersionSelectionsPluginHelpTask::class.java)
    }

}


interface VersionSelector : Ordered {
    fun getVersion(project: Project, group: String, name: String): String?
}

interface DescribedVersionSelector : VersionSelector {
    val description: String
}

abstract class VersionSelectorByPattern : VersionSelector {

    abstract val dependencyPatterns: List<String>

    init {
        if (dependencyPatterns.isEmpty()) throw IllegalStateException("dependencyPatterns is empty")
    }

    private val dependencyRegexps = dependencyPatterns.map { Regex("^${it.splitToSequence('*').map(String::escapeRegex).joinToString(".*")}$", IGNORE_CASE) }

    final override fun getVersion(project: Project, group: String, name: String): String? {
        dependencyRegexps.forEach {
            if (it.matches("$group:$name")) {
                return getVersion(project)
            }
        }
        return null
    }

    protected abstract fun getVersion(project: Project): String?

}

abstract class VersionSelectorByPatternAndProperty : VersionSelectorByPattern(), DescribedVersionSelector {

    abstract val propertyNames: List<String>

    init {
        if (propertyNames.isEmpty()) throw IllegalStateException("propertyNames is empty")
    }

    open val fallbackValue: String? = null

    final override fun getVersion(project: Project): String? {
        propertyNames.forEach {
            project.findProperty(it)?.toString()?.apply { return this }
        }
        return fallbackValue
    }

    final override val description
        get() = buildString {
            append("'")
            dependencyPatterns.joinTo(this, "', '")
            append("'")

            if (1 == propertyNames.size) {
                append(" - project property '").append(propertyNames.first()).append("'")
            } else {
                append(" - project properties '")
                propertyNames.joinTo(this, "', '")
                append("'")
            }

            fallbackValue?.apply {
                append(" (fallback: ").append(this).append(")")
            }
        }

}


@API
class VersionSelectionsPluginHelpTask : DefaultTask() {

    @get:Inject
    protected val styledTextOutputFactory: StyledTextOutputFactory
        get() = throw AbstractMethodError()
    @get:Inject
    protected val pluginRegistry: PluginRegistry
        get() = throw AbstractMethodError()

    protected val pluginId = pluginRegistry.inspect(VersionSelectionsPlugin::class.java)?.pluginId?.id

    init {
        group = HELP_GROUP
        description = "Display help for $pluginId plugin"
    }

    @TaskAction
    fun displayHelp() {
        with(styledTextOutputFactory.create(VersionSelectionsPlugin::class.java)) {
            println()

            val describedVersionSelectors = versionSelectors.filterIsInstance(DescribedVersionSelector::class.java)
            if (describedVersionSelectors.isEmpty()) {
                text("No version selection rules are configured!\n")
            } else {
                text("This version selection rules are configured:\n")
                describedVersionSelectors.forEach { text("  * ${it.description}\n") }
            }

            println()
        }
    }

}
