package name.remal.building.gradle_plugins.classes_relocation

import name.remal.building.gradle_plugins.artifact.Artifact
import name.remal.building.gradle_plugins.artifact.CachedArtifactsCollection
import name.remal.building.gradle_plugins.artifact.HasEntries
import name.remal.building.gradle_plugins.dsl.createParentDirectories
import name.remal.building.gradle_plugins.utils.ASM_API
import name.remal.building.gradle_plugins.utils.classToResourcePath
import name.remal.building.gradle_plugins.utils.forEachClassFile
import org.gradle.api.artifacts.Configuration
import org.objectweb.asm.ClassReader
import org.objectweb.asm.ClassVisitor
import org.objectweb.asm.ClassWriter
import org.objectweb.asm.commons.ClassRemapper
import org.objectweb.asm.commons.Remapper
import org.slf4j.LoggerFactory
import java.io.File
import java.util.*

internal class ClassesRelocator(
    private val classesDir: File,
    private val basePackage: String,
    private val relocateAllConf: Configuration,
    private val relocateUsedConf: Configuration
) {

    companion object {
        @JvmStatic private val logger = LoggerFactory.getLogger(ClassesRelocator::class.java)
    }

    val remappedClassNames: Map<String, String> get() = remappedClassNamesMutable
    private val remappedClassNamesMutable = mutableMapOf<String, String>()

    fun doRelocateClasses() {
        extractAll()
        extractUsed()
        remapClasses()
    }

    private fun extractAll() {
        val resolvedArtifactFiles = relocateAllConf.resolve()
        if (resolvedArtifactFiles.isEmpty()) return
        logger.info("Extracting all classes from {}", resolvedArtifactFiles.map { it.name })

        val outputArtifact = Artifact(classesDir)
        val artifacts = CachedArtifactsCollection(resolvedArtifactFiles)
        artifacts.classNames.forEach { className ->
            if (className in outputArtifact.classNames) {
                logger.error("Already exists in sources {} - do not extract", className)
                return@forEach
            } else {
                logger.debug("Extracting {}", className)
            }

            remappedClassNamesMutable.computeIfAbsent(className) { artifacts.extractClass(className) }
        }
    }

    private fun extractUsed() {
        val resolvedArtifactFiles = relocateUsedConf.resolve()
        if (resolvedArtifactFiles.isEmpty()) return
        logger.info("Extracting used only classes from {}", resolvedArtifactFiles.map { it.name })

        val artifacts = CachedArtifactsCollection(resolvedArtifactFiles)

        val outputArtifact = Artifact(classesDir)
        val extractedClasses = mutableSetOf<String>()
        val classesToExtract: Queue<String> = ArrayDeque()
        val remapper = object : Remapper() {
            override fun map(internalClassName: String): String? {
                internalClassName.replace('/', '.').let { className ->
                    if (className in artifacts.classNames) {
                        if (className in outputArtifact.classNames) {
                            logger.error("Already exists in sources {} - do not extract", className)
                        } else {
                            val resultClassName = basePackage + '.' + className
                            if (extractedClasses.add(className)) {
                                if (null == remappedClassNamesMutable.put(className, resultClassName)) {
                                    logger.debug("Extracting {}", className)
                                    classesToExtract += className
                                } else {
                                    logger.debug("Already relocated {} - do not extract", className)
                                }
                            }
                            return resultClassName.replace('.', '/');
                        }
                    }
                }
                return null
            }
        }

        remapClasses(remapper)

        while (true) {
            val classToExtract = classesToExtract.poll() ?: break
            logger.debug("Processing {}", classToExtract)
            artifacts.extractClass(classToExtract, remapper)
        }
    }

    private fun remapClasses() {
        if (remappedClassNamesMutable.isEmpty()) return
        remapClasses(object : Remapper() {
            override fun map(internalClassName: String): String? {
                return remappedClassNamesMutable[internalClassName.replace('/', '.')]?.replace('.', '/')
            }
        })
    }

    private fun HasEntries.extractClass(className: String, remapper: Remapper? = null): String {
        val resultClassName = basePackage + '.' + className
        val resultFile = File(classesDir, classToResourcePath(resultClassName))

        val classWriter = ClassWriter(0)
        var classVisitor: ClassVisitor = classWriter
        classVisitor = DoNotProcessAnnotationAdder(classVisitor)
        classVisitor = RelocatedClassesTransformer(classVisitor)
        if (null != remapper) classVisitor = ClassRemapper(classVisitor, remapper)
        this.openStreamForClass(className).use { inputStream ->
            ClassReader(inputStream.readBytes()).accept(classVisitor, 0)
        }
        resultFile.createParentDirectories().writeBytes(classWriter.toByteArray())

        return resultClassName;
    }

    private fun remapClasses(remapper: Remapper) {
        forEachClassFile(classesDir) { classFile ->
            val classWriter = ClassWriter(0)
            var classVisitor: ClassVisitor = classWriter
            classVisitor = ClassRemapper(classVisitor, remapper)
            val logVisitor = object : ClassVisitor(ASM_API, classVisitor) {
                override fun visit(version: Int, access: Int, name: String?, signature: String?, superName: String?, interfaces: Array<out String>?) {
                    logger.debug("Processing {}", name?.replace('/', '.'))
                    super.visit(version, access, name, signature, superName, interfaces)
                }
            }
            ClassReader(classFile.readBytes()).accept(logVisitor, 0)
            classFile.writeBytes(classWriter.toByteArray())
        }
    }

}
