package name.remal.building.gradle_plugins.coverage_metrics

import com.google.common.collect.ImmutableMultimap
import com.google.common.collect.MultimapBuilder
import groovy.json.JsonSlurper

internal object CoverageMetricsAnnotationParser {

    fun parse(text: String): CoverageMetricsAnnotationInfo {
        if (text.isEmpty()) throw IllegalArgumentException("Coverage metrics annotation info can't be an empty string")
        val className = text.substringBefore('{')
        val checkValues: ImmutableMultimap<String, String> = run {
            val pos = text.indexOf('{')
            if (pos < 0) return@run ImmutableMultimap.of<String, String>()

            val valuesJson = text.substring(pos)
            val checkValues = MultimapBuilder.treeKeys().linkedHashSetValues().build<String, String>()
            try {
                @Suppress("UNCHECKED_CAST")
                val json = JsonSlurper().parseText(valuesJson) as Map<String, Any?>
                json.forEach { name, value ->
                    if (name.isEmpty()) throw IllegalArgumentException("Coverage metrics annotation info value name can't be empty")
                    if (null == value) return@forEach
                    if (value is Iterable<*>) {
                        value.asSequence().filterNotNull().forEach { checkValues.put(name, it.toString()) }
                    } else if (value is Map<*, *>) {
                        throw IllegalArgumentException("Coverage metrics annotation info value can't be an map")
                    } else {
                        checkValues.put(name, value.toString())
                    }
                }
            } catch (e: Exception) {
                throw IllegalStateException("Error processing coverage metrics annotation info: $text (values json: $valuesJson)", e)
            }
            return@run ImmutableMultimap.copyOf(checkValues)
        }
        return CoverageMetricsAnnotationInfo(className, checkValues)
    }

}

