package name.remal.building.gradle_plugins.dsl

import java.time.Instant
import java.time.LocalDateTime
import java.time.ZoneId
import java.time.temporal.ChronoField.*
import java.util.*
import java.util.regex.Pattern

data class Version(
    val major: Int = 0,
    val minor: Int? = null,
    val patch: Int? = null,
    val build: Int? = null,
    val suffix: String? = null
) : Comparable<Version> {

    companion object {
        @JvmStatic
        fun parse(string: String): Version {
            val matcher = PATTERN.matcher(string)
            if (!matcher.matches()) throw IllegalArgumentException("version string '$string' isn't matched /$PATTERN/")
            return Version(
                matcher.group("major")!!.let(Integer::parseInt),
                matcher.group("minor")?.let(Integer::parseInt),
                matcher.group("patch")?.let(Integer::parseInt),
                matcher.group("build")?.let(Integer::parseInt),
                matcher.group("suffix")
            )
        }

        @JvmStatic
        fun parseOrNull(string: String) = try {
            parse(string)
        } catch (ignored: Exception) {
            null
        }

        @JvmStatic
        fun parseInaccurate(string: String): Version {
            val matcher = INACCURATE_PATTERN.matcher(string)
            if (!matcher.matches()) throw IllegalArgumentException("version string '$string' isn't matched /$INACCURATE_PATTERN/")
            return Version(
                matcher.group("major")!!.let(Integer::parseInt),
                matcher.group("minor")?.let(Integer::parseInt),
                matcher.group("patch")?.let(Integer::parseInt),
                matcher.group("build")?.let(Integer::parseInt),
                matcher.group("suffix")
            )
        }

        @JvmStatic
        fun parseInaccurateOrNull(string: String) = try {
            parseInaccurate(string)
        } catch (ignored: Exception) {
            null
        }

        @JvmStatic
        fun ofLocalDateTime(dateTime: LocalDateTime): Version {
            return dateTime.atZone(ZoneId.systemDefault()).withZoneSameInstant(ZoneId.of("UTC")).let {
                Version(
                    it.get(YEAR),
                    it.get(MONTH_OF_YEAR),
                    it.get(DAY_OF_MONTH),
                    null,
                    "%02d%02d%02d".format(it.get(HOUR_OF_DAY), it.get(MINUTE_OF_HOUR), it.get(SECOND_OF_MINUTE))
                )
            }
        }

        @JvmStatic
        fun ofTimestamp(timestamp: Long) = ofLocalDateTime(Instant.ofEpochMilli(timestamp).atZone(ZoneId.systemDefault()).toLocalDateTime())

        @JvmStatic
        fun ofDate(date: Date) = ofLocalDateTime(LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault()))
    }

    init {
        if (major < 0) throw IllegalArgumentException("major < 0")
        if (null != minor && minor < 0) throw IllegalArgumentException("minor < 0")
        if (null != patch && patch < 0) throw IllegalArgumentException("patch < 0")
        if (null != build && build < 0) throw IllegalArgumentException("build < 0")
        if (null != suffix && suffix.isEmpty()) throw IllegalArgumentException("suffix is empty")
    }

    constructor(major: Int, minor: Int?, patch: Int?, suffix: String?) : this(major, minor, patch, null, suffix)
    constructor(major: Int, minor: Int?, suffix: String?) : this(major, minor, null, suffix)
    constructor(major: Int, suffix: String?) : this(major, null, suffix)

    override fun compareTo(other: Version): Int {
        var result = major.compareTo(other.major)
        if (0 == result) result = compareNullsFirst(minor, other.minor)
        if (0 == result) result = compareNullsFirst(patch, other.patch)
        if (0 == result) result = compareNullsFirst(build, other.build)
        if (0 == result) result = compareNullsFirst(suffix, other.suffix)
        return result
    }

    private val stringValue = buildString {
        append(major)
        val minor = minor ?: if (null != patch || null != build) 0 else null
        if (null != minor) append(MAIN_DELIM).append(minor)
        val patch = patch ?: if (null != build) 0 else null
        if (null != patch) append(MAIN_DELIM).append(patch)
        if (null != build) append(MAIN_DELIM).append(build)
        if (null != suffix) append(SUFFIX_DELIM).append(suffix)
    }

    override fun toString() = stringValue

    fun withMajor(major: Int): Version {
        if (this.major == major) return this
        return Version(major, minor, patch, build, suffix)
    }

    fun incrementMajor(increment: Int = 1): Version {
        if (increment < 0) throw IllegalArgumentException("increment < 0")
        return Version(major + increment, minor, patch, build, suffix)
    }

    val hasMinor: Boolean get() = null != minor

    fun withMinor(minor: Int?): Version {
        if (this.minor == minor) return this
        return Version(major, minor, patch, build, suffix)
    }

    fun incrementMinor(increment: Int = 1): Version {
        if (increment < 0) throw IllegalArgumentException("increment < 0")
        return Version(major, minor + increment, patch, build, suffix)
    }

    val hasPatch: Boolean get() = null != patch

    fun withPatch(patch: Int?): Version {
        if (this.patch == patch) return this
        return Version(major, if (null != patch) minor + 0 else minor, patch, build, suffix)
    }

    fun incrementPatch(increment: Int = 1): Version {
        if (increment < 0) throw IllegalArgumentException("increment < 0")
        return Version(major, minor + 0, patch + increment, build, suffix)
    }

    val hasBuild: Boolean get() = null != build

    fun withBuild(build: Int?): Version {
        if (this.build == build) return this
        return Version(major, if (null != build) minor + 0 else minor, if (null != build) patch + 0 else patch, build, suffix)
    }

    fun incrementBuild(increment: Int = 1): Version {
        if (increment < 0) throw IllegalArgumentException("increment < 0")
        return Version(major, minor + 0, patch + 0, build + increment, suffix)
    }

    val hasSuffix: Boolean get() = null != suffix

    fun withSuffix(suffix: String?): Version {
        if (this.suffix == suffix) return this
        return Version(major, minor, patch, build, suffix)
    }

}


private const val MAIN_DELIM = '.'
private const val SUFFIX_DELIM = '-'


private fun Char.escapeRegex(): String {
    if ('\\' == this
        || '^' == this || '$' == this
        || '{' == this || '}' == this
        || '[' == this || ']' == this
        || '(' == this || ')' == this
        || '.' == this
        || '*' == this || '+' == this || '?' == this
        || '|' == this
        || '<' == this || '>' == this
        || '-' == this
        || '&' == this
        ) {
        return "\\" + this
    }

    return "$this"
}

private val PATTERN = Pattern.compile(buildString {
    append('^')
    append("(?<major>\\d+)")
    append("(${MAIN_DELIM.escapeRegex()}(?<minor>\\d+))?")
    append("(${MAIN_DELIM.escapeRegex()}(?<patch>\\d+))?")
    append("(${MAIN_DELIM.escapeRegex()}(?<build>\\d+))?")
    append("(${SUFFIX_DELIM.escapeRegex()}(?<suffix>.+))?")
    append('$')
})

private val INACCURATE_PATTERN = Pattern.compile(buildString {
    append('^')
    append("(?<major>\\d+)")
    append("(${MAIN_DELIM.escapeRegex()}(?<minor>\\d+))?")
    append("(${MAIN_DELIM.escapeRegex()}(?<patch>\\d+))?")
    append("(${MAIN_DELIM.escapeRegex()}(?<build>\\d+))?")
    append("(${MAIN_DELIM.escapeRegex()}\\d+)*")
    append("(${SUFFIX_DELIM.escapeRegex()}(?<suffix>.+))?")
    append('$')
})

private fun <T : Comparable<T>> compareNullsFirst(obj1: T?, obj2: T?): Int {
    if (obj1 == obj2) return 0
    if (null == obj1) return -1
    if (null == obj2) return 1
    return obj1.compareTo(obj2)
}

private operator fun Int?.plus(increment: Int) = if (null == this) increment else this + increment
