package name.remal.building.gradle_plugins.dynamic_dependencies

import groovy.lang.Closure
import groovy.lang.Closure.DELEGATE_FIRST
import groovy.lang.DelegatesTo
import name.remal.building.gradle_plugins.API
import name.remal.building.gradle_plugins.utils.concurrentSetOf
import org.gradle.api.Action
import org.gradle.api.artifacts.Dependency
import org.gradle.api.artifacts.dsl.DependencyHandler
import org.gradle.util.Configurable
import org.gradle.util.ConfigureUtil.configure
import org.gradle.util.ConfigureUtil.configureSelf

@API
class DynamicDependenciesExtension(private val dependencyHandler: DynamicDependenciesDependencyHandler) : Configurable<DynamicDependenciesExtension>, DynamicDependenciesDependencyHandler by dependencyHandler {

    constructor(dependencyHandler: DependencyHandler) : this(DynamicDependenciesDependencyHandlerImpl(dependencyHandler))

    internal val dynamicDependencies: MutableSet<DynamicDependency> = concurrentSetOf()

    override fun configure(@DelegatesTo(value = Check::class, strategy = DELEGATE_FIRST) closure: Closure<*>) = configureSelf(closure, this)

    fun check(dependencyNotation: Any?): Check {
        val checkInfos = ((dependencyNotation as? Iterable<*>)?.asSequence() ?: sequenceOf(dependencyNotation))
            .filterNotNull()
            .map { it as? Dependency ?: dependencyHandler.create(it) }
            .map { CheckInfo(it.group, it.name) }
            .toSet()
        return Check(checkInfos)
    }

    fun check(dependencyNotation: Any?, @DelegatesTo(value = Check::class, strategy = DELEGATE_FIRST) closure: Closure<*>): Check {
        return check(dependencyNotation).apply { configure(closure, this) }
    }

    fun check(dependencyNotation: Any?, action: Action<Check>): Check {
        return check(dependencyNotation).apply { action.execute(this) }
    }

    fun check(dependencyNotation: Any?, action: Check.() -> Unit): Check {
        return check(dependencyNotation).apply { action(this) }
    }

    inner class Check internal constructor(internal val checkInfos: Set<CheckInfo>) : Configurable<Check>, DynamicDependenciesDependencyHandler by dependencyHandler {

        override fun configure(@DelegatesTo(value = Check::class, strategy = DELEGATE_FIRST) closure: Closure<*>) = configureSelf(closure, this)

        fun check(dependencyNotation: Any?): Check {
            return Check(checkInfos + this@DynamicDependenciesExtension.check(dependencyNotation).checkInfos)
        }

        fun check(dependencyNotation: Any?, @DelegatesTo(value = Check::class, strategy = DELEGATE_FIRST) closure: Closure<*>): Check {
            return check(dependencyNotation).apply { configure(closure, this) }
        }

        fun check(dependencyNotation: Any?, action: Action<Check>): Check {
            return check(dependencyNotation).apply { action.execute(this) }
        }

        fun check(dependencyNotation: Any?, action: Check.() -> Unit): Check {
            return check(dependencyNotation).apply { action(this) }
        }

        fun add(dependencyNotation: Any?) {
            ((dependencyNotation as? Iterable<*>)?.asSequence() ?: sequenceOf(dependencyNotation))
                .filterNotNull()
                .map { it as? Dependency ?: dependencyHandler.create(it) }
                .forEach { dynamicDependencies += DynamicDependency(checkInfos, it) }
        }

    }

}

