package name.remal.building.gradle_plugins.generate_sources

import name.remal.building.gradle_plugins.dsl.escapeJava
import org.gradle.api.Project
import java.io.StringWriter
import java.io.Writer

abstract class BaseSourceGenerator() {

    abstract fun accept(project: Project, writer: Writer)

    lateinit var packageName: String
    lateinit var fileName: String

    constructor(packageName: String, fileName: String) : this() {
        this.packageName = packageName
        this.fileName = fileName
    }

    protected open fun Writer.writePackage() {
        if (packageName.isNotEmpty()) {
            append("package ").append(packageName).append(";\n")
        }
    }

    protected open fun Writer.writeImport(canonicalClassName: String) {
        append("import ").append(canonicalClassName).append(";\n")
    }

    open fun Writer.writeImport(clazz: Class<*>) = writeImport(clazz.canonicalName)

    protected open fun Writer.writeStaticImport(canonicalClassName: String, member: String) {
        append("import static ").append(canonicalClassName).append('.').append(member).append(";\n")
    }

    open fun Writer.writeStaticImport(clazz: Class<*>, member: String) = writeStaticImport(clazz.canonicalName, member)

    protected open fun Writer.writeSuppressWarnings(vararg warnings: String = arrayOf("all")) {
        append("@SuppressWarnings({")
        warnings.forEachIndexed { index, warning ->
            if (1 < index) append(", ")
            append('"').append(warning.escapeJava()).append('"')
        }
        append("})\n")
    }

    protected open val indentSize = 4

    protected fun Writer.writeBlock(expression: String = "", blockWriterHandler: Writer.() -> Unit) {
        if (expression.isNotEmpty()) {
            append(expression)
            append(' ')
        }

        append('{')

        StringWriter().use { blockWriter ->
            blockWriterHandler(blockWriter)
            val content = blockWriter.toString().replace("\r", "").let {
                if (it.endsWith('\n')) {
                    it.substringBeforeLast('\n')
                } else {
                    it
                }
            }
            if (content.isNotEmpty()) {
                append('\n')
                val indent = " ".repeat(indentSize)
                content.split('\n').forEach { append(indent).append(it).append('\n') }
            }
        }

        append('}')
    }

}
