package name.remal.building.gradle_plugins.gitlab_tools

import groovy.json.JsonSlurper
import name.remal.building.gradle_plugins.API
import name.remal.building.gradle_plugins.dsl.DEFAULT_IO_TIMEOUT
import name.remal.building.gradle_plugins.dsl.contentCharset
import name.remal.building.gradle_plugins.dsl.encodeURIComponent
import name.remal.building.gradle_plugins.dsl.use
import name.remal.building.gradle_plugins.utils.get
import org.gradle.api.DefaultTask
import java.net.HttpURLConnection
import java.net.URI
import java.net.URL

@API
abstract class AbstractGitlabTask : DefaultTask() {

    init {
        onlyIf { !project.gradle.startParameter.isOffline }
    }

    protected val settings: GitlabToolsExtension by lazy { project[GitlabToolsExtension::class.java] }

    protected enum class Method {
        GET, POST, PUT, DELETE
    }

    protected object PROJECT_ID

    protected fun <T> executeApiRequest(method: Method = Method.GET, path: String, params: Map<String, Any?> = mapOf()): T? {
        if (project.gradle.startParameter.isOffline) throw IllegalStateException("Gradle is running in offline mode")

        val projectURI = URI(settings.projectUrl ?: throw IllegalStateException("Gitlab project URL is not set"))
        val apiToken = settings.apiToken ?: throw IllegalStateException("Gitlab API token is not set")

        fun CharSequence.makeSafe(): String = this.toString()
            .replace(apiToken, "x".repeat(apiToken.length))
            .replace(apiToken.encodeURIComponent(), "x".repeat(apiToken.length))

        val location = buildString {
            append("https://${projectURI.authority}/api/v4/")

            val pathParams = mutableSetOf<String>()
            append(path.trimStart('/').split('/')
                .map {
                    if (it.startsWith(':')) {
                        val paramName = it.substring(1)
                        pathParams.add(paramName)
                        val param = params[paramName] ?: throw IllegalArgumentException("Required parameter $paramName for path $path is not set")
                        if (PROJECT_ID === param) {
                            return@map projectURI.path.trim('/').encodeURIComponent()
                        }
                        return@map param.toString().encodeURIComponent()
                    }
                    return@map it
                }
                .joinToString("/"))

            append('?')

            params.forEach { name, value ->
                if (null == value) return@forEach
                if (name in pathParams) return@forEach
                append('&')
                append(name.encodeURIComponent())
                append('=')
                append(value.toString().encodeURIComponent())
            }

            append("&private_token=").append(apiToken.encodeURIComponent())
        }

        try {
            val connection = URL(location).openConnection() as HttpURLConnection
            connection.connectTimeout = DEFAULT_IO_TIMEOUT.toMillis().toInt()
            connection.readTimeout = 10 * DEFAULT_IO_TIMEOUT.toMillis().toInt()
            connection.useCaches = false
            connection.instanceFollowRedirects = true
            connection.requestMethod = method.name
            connection.connect()
            try {
                val responseCode = connection.responseCode
                if (204 == responseCode) return null

                val body = (connection.errorStream ?: connection.inputStream).reader(connection.contentCharset).use { it.readText() }
                val result = if (true == connection.contentType?.toLowerCase()?.let { it.contains("javascript") || it.contains("json") }) {
                    JsonSlurper().parseText(body)
                } else {
                    body
                }

                if (400 <= responseCode) {
                    throw GitlabApiException("Error requesting $method ${location.makeSafe()}: $responseCode ${connection.responseMessage}: " +
                        if (result is Map<*, *>) {
                            result["message"]
                        } else {
                            result
                        }
                    )
                }

                @Suppress("UNCHECKED_CAST")
                return result as T

            } finally {
                connection.disconnect()
            }

        } catch (e: GitlabApiException) {
            throw e
        } catch (throwable: Throwable) {
            throw GitlabApiException("Error requesting $method ${location.makeSafe()}: ${throwable.message?.makeSafe()}")
        }
    }

}
