package name.remal.building.gradle_plugins.packed_dependencies

import name.remal.building.gradle_plugins.dsl.decodeURIComponent
import org.gradle.api.Project
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ResolvedArtifact
import org.gradle.api.artifacts.component.ModuleComponentIdentifier
import org.gradle.api.artifacts.component.ProjectComponentIdentifier
import java.io.File

internal data class PackedDependencyInfo(
    val resourceName: String,
    val artifactFile: File
) {

    constructor(project: Project, resolvedArtifact: ResolvedArtifact) : this(
        resolvedArtifact.let {
            val id = it.id.componentIdentifier
            if (id is ModuleComponentIdentifier) {
                return@let "META-INF/packed-deps/${id.group.toString().decodeURIComponent()}.${id.module.decodeURIComponent()}.${it.extension}"
            } else if (id is ProjectComponentIdentifier) {
                val projectPath = id.projectPath
                val dependencyProject = project.rootProject.evaluationDependsOn(projectPath)
                return@let "META-INF/packed-deps/${dependencyProject.group.toString().decodeURIComponent()}.${dependencyProject.name.decodeURIComponent()}.${it.extension}"
            }
            throw IllegalArgumentException("Unsupported artifact: $it")
        },
        resolvedArtifact.file
    )

    override fun hashCode(): Int {
        return resourceName.hashCode()
    }

    override fun equals(other: Any?): Boolean {
        if (other !is PackedDependencyInfo) return false
        return resourceName == other.resourceName
    }

}

internal fun Configuration.createPackedDependencyInfos(project: Project): Collection<PackedDependencyInfo> {
    val conf = this
    return mutableSetOf<PackedDependencyInfo>().apply {
        conf.resolvedConfiguration.resolvedArtifacts.forEach {
            add(PackedDependencyInfo(project, it))
        }
    }
}
