package name.remal.building.gradle_plugins.utils

import org.gradle.api.*
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ConfigurationContainer
import org.gradle.api.artifacts.ExcludeRule.GROUP_KEY
import org.gradle.api.artifacts.ExcludeRule.MODULE_KEY
import org.gradle.api.artifacts.ModuleDependency
import org.gradle.api.artifacts.result.ResolvedDependencyResult
import org.gradle.api.attributes.Attribute
import org.gradle.api.internal.artifacts.configurations.ConfigurationInternal
import org.gradle.api.internal.project.ProjectInternal
import org.gradle.api.internal.project.ProjectStateInternal
import org.gradle.api.plugins.Convention
import org.gradle.api.plugins.ExtensionContainer
import org.gradle.api.plugins.UnknownPluginException
import org.gradle.api.tasks.TaskCollection
import org.gradle.api.tasks.TaskContainer
import org.gradle.api.tasks.TaskState
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicInteger

// Project:

internal val Project.id: String
    get() {
        var result = buildString {
            append(group)
            append('.')
            append(name)
        }
        result = result.replace(':', '.')
        while (result.contains("..")) result = result.replace("..", ".")
        result = result.trim('.')
        return result
    }

internal val Project.isExecuting: Boolean get() = (this.state as ProjectStateInternal).executing
internal val Project.isExecuted: Boolean get() = this.state.executed
internal val Project.hasFailure: Boolean get() = null != this.state.failure
internal val Project.isEvaluated: Boolean get() = isExecuting || isExecuted || hasFailure

internal operator fun <T : Any> Project.get(type: Class<T>): T {
    try {
        return this.convention[type]

    } catch (conventionException: Throwable) {
        try {
            return this.extensions[type]

        } catch (e: Throwable) {
            e.addSuppressed(conventionException)
            throw e
        }
    }
}

internal fun Project.applyPlugin(id: String, action: () -> Unit = {}) {
    this.pluginManager.apply(id)
    action()
}

internal fun Project.applyPlugin(ids: Iterable<String>, action: () -> Unit = {}) {
    val idsList = ids.toList()
    if (idsList.isEmpty()) throw IllegalArgumentException("empty ids")

    val exceptions = ArrayList<UnknownPluginException>(idsList.size)
    idsList.forEach { id ->
        try {
            this.pluginManager.apply(id);
            action()
            return

        } catch (e: UnknownPluginException) {
            exceptions.add(e)
        }
    }

    if (idsList.size == exceptions.size) {
        throw UnknownPluginException("Plugins with ids $idsList not found").apply {
            exceptions.forEach(this::addSuppressed)
        }
    }
}

internal fun Project.applyPluginIfExists(id: String, action: () -> Unit = {}) = this.applyPluginIfExists(listOf(id), action)

internal fun Project.applyPluginIfExists(ids: Iterable<String>, action: () -> Unit = {}) {
    ids.forEach { id ->
        try {
            this.pluginManager.apply(id);
            action()
            return

        } catch (ignored: UnknownPluginException) {
            // do nothing
        }
    }
}

internal fun <T : Plugin<Project>> Project.applyPlugin(type: Class<T>, action: () -> Unit = {}) {
    this.pluginManager.apply(type)
    action()
}

internal fun <T : Task> Project.configureTasks(type: Class<T>, configurer: (T) -> Unit) {
    this.tasks.configure(type, configurer)
}

internal fun Project.withPlugin(id: String, action: (Project) -> Unit) {
    this.pluginManager.withPlugin(id, { action(this) })
}

internal fun Project.withPlugin(id: Iterable<String>, action: (Project) -> Unit) = this.withOneOfPlugin(ids = id, action = action)

internal fun Project.withPlugins(vararg ids: Any, action: (Project) -> Unit) {
    if (ids.isEmpty()) throw IllegalArgumentException("empty ids")
    val counter = AtomicInteger(ids.size)
    for (id in ids) {
        this.withOneOfPlugin(id) {
            if (0 == counter.decrementAndGet()) {
                action(this)
            }
        }
    }
}

internal fun Project.withOneOfPlugin(vararg ids: Any, action: (Project) -> Unit) {
    val flattenIds = ids.asSequence()
        .flatMap {
            if (it is Iterable<*>) {
                it.asSequence()
            } else {
                sequenceOf(it)
            }
        }
        .filterNotNull()
        .map(Any::toString)
        .toList()

    if (flattenIds.isEmpty()) throw IllegalArgumentException("empty ids")

    val isExecuted = AtomicBoolean(false)
    for (id in flattenIds) {
        this.pluginManager.withPlugin(id) {
            if (isExecuted.compareAndSet(false, true)) {
                action(this)
            }
        }
    }
}

internal fun Project.afterEvaluate(action: ProjectAction) {
    val ext = this.extensions.getOrCreate(
        AfterEvaluateProjectActionsExtension::class.java,
        onCreate = { ext ->
            this.afterEvaluate {
                ext.actions.sorted().forEach { it.execute(this) }
            }
        }
    )
    ext.actions.add(action)
}

internal fun Project.afterEvaluateOrdered(order: Int = 0, action: (Project) -> Unit) {
    afterEvaluate(object : ProjectAction {
        override fun execute(project: Project) {
            action(project)
        }

        override fun getOrder(): Int {
            return order
        }
    })
}

internal fun Project.evaluate() {
    val projectInternal = this as ProjectInternal
    projectInternal.evaluate()
}


// TaskContainer:

internal fun TaskContainer.getOrCreate(name: String, configurer: (Task) -> Unit = {}): Task {
    return findByName(name) ?: create(name, configurer)
}

internal fun <T : Task> TaskContainer.getOrCreate(name: String, type: Class<T>, configurer: (T) -> Unit = {}): Task {
    return findByName(name) ?: create(name, type, configurer)
}


// Task:

@Suppress("UNCHECKED_CAST")
internal fun <TaskType : Task> TaskType.doFirst(action: TaskAction<TaskType>) {
    val ext = this.extensions.getOrCreate(
        DoFirstTaskActionsExtension::class.java,
        onCreate = { ext ->
            ext as DoFirstTaskActionsExtension<TaskType>
            this.doFirst {
                ext.actions.sorted().forEach { it.execute(this) }
            }
        }
    )
    ext as DoFirstTaskActionsExtension<TaskType>
    ext.actions.add(action)
}

internal fun <TaskType : Task> TaskType.doFirstOrdered(order: Int = 0, action: (TaskType) -> Unit) {
    doFirst(object : TaskAction<TaskType>() {
        override fun execute(task: TaskType) {
            action(task)
        }

        override fun getOrder(): Int {
            return order
        }
    })
}

@Suppress("UNCHECKED_CAST")
internal fun <TaskType : Task> TaskType.doLast(action: TaskAction<TaskType>) {
    val ext = this.extensions.getOrCreate(
        DoLastTaskActionsExtension::class.java,
        onCreate = { ext ->
            ext as DoLastTaskActionsExtension<TaskType>
            this.doLast {
                ext.actions.sorted().forEach { it.execute(this) }
            }
        }
    )
    ext as DoLastTaskActionsExtension<TaskType>
    ext.actions.add(action)
}

internal fun <TaskType : Task> TaskType.doLastOrdered(order: Int = 0, action: (TaskType) -> Unit) {
    doLast(object : TaskAction<TaskType>() {
        override fun execute(task: TaskType) {
            action(task)
        }

        override fun getOrder(): Int {
            return order
        }
    })
}


// TaskState:

internal val TaskState.failed: Boolean get() = null != this.failure


// ConfigurationContainer:

internal fun ConfigurationContainer.findRootConfigurations(): NamedDomainObjectSet<Configuration> {
    return this.matching { it.extendsFrom.isEmpty() }
}

internal fun ConfigurationContainer.resolveCopyOf(vararg configurations: Configuration) = resolveCopyOf(configurations.toList())

internal fun ConfigurationContainer.resolveCopyOf(configurations: Collection<Configuration>): Set<ResolvedDependencyResult> {
    if (configurations.isEmpty()) return setOf()

    fun Configuration.extendsFromCopy(conf: Configuration) {
        extendsFrom(detachedConfiguration().apply {
            attributes.let { attrs ->
                conf.attributes.let { confAttrs ->
                    confAttrs.keySet().forEach { key ->
                        val value = confAttrs.getAttribute(key) ?: return@forEach
                        @Suppress("UNCHECKED_CAST")
                        val attrKey = key as Attribute<Any?>
                        attrs.attribute(attrKey, value)
                        Unit
                    }
                }
            }
            javaClass.setField(this, "resolutionStrategy", resolutionStrategy)
            isTransitive = conf.isTransitive
            dependencies.addAll(conf.dependencies)
            conf.excludeRules.forEach {
                val excludeProps = mutableMapOf<String, String?>()
                excludeProps[GROUP_KEY] = it.group
                excludeProps[MODULE_KEY] = it.module
                exclude(excludeProps)
            }
            javaClass.setField(this, "defaultDependencyActions", conf.javaClass.getField(conf, "defaultDependencyActions"))
            conf.extendsFrom.forEach { extendsFromCopy(it) }
        })
    }

    val rootConf = this.detachedConfiguration()
    configurations.forEach { rootConf.extendsFromCopy(it) }

    rootConf as ConfigurationInternal
    rootConf.triggerWhenEmptyActionsIfNecessary()

    if (rootConf.allDependencies.isEmpty()) return emptySet()

    return rootConf.incoming.resolutionResult.allDependencies.asSequence()
        .filterIsInstance(ResolvedDependencyResult::class.java)
        .toSet()
}


// Configuration:

internal fun Configuration.makeNotTransitive() {
    this.isTransitive = false
    this.dependencies.configure { dep ->
        if (dep is ModuleDependency) dep.isTransitive = false
    }
}


// ExtensionContainer:

internal operator fun ExtensionContainer.contains(type: Class<*>): Boolean {
    if (this is Convention) {
        if (null != this.findPlugin(type)) return true
    }

    if (null != this.findByType(type)) return true

    return false
}

internal operator fun <T : Any> ExtensionContainer.get(type: Class<T>): T {
    var supressedException: Throwable? = null
    try {
        if (this is Convention) {
            return this.getPlugin(type)
        }
    } catch (e: Throwable) {
        supressedException = e
    }

    try {
        return this.getByType(type)
    } catch (e: Throwable) {
        e.addSuppressed(supressedException)
        throw e
    }
}

@Suppress("NULLABILITY_MISMATCH_BASED_ON_JAVA_ANNOTATIONS")
internal fun <Type : Any, ImplType : Type> ExtensionContainer.getOrCreate(type: Class<Type>, name: String? = null, implType: Class<ImplType>? = null, vararg arguments: Any?, onCreate: (Type) -> Unit = {}): Type {
    this.findByType(type)?.let { return it }
    return this.create(type, name ?: type.name.replace('.', '_'), implType ?: type, *arguments).apply {
        onCreate(this)
    }
}

internal fun ExtensionContainer.addIfNotContains(name: String, extension: Any) {
    try {
        this.add(name, extension)
    } catch (e: IllegalArgumentException) {
        // do nothing
    }
}


// DomainObjectCollection:

internal operator fun <T, R : T> DomainObjectCollection<T>.get(type: Class<R>): DomainObjectCollection<R> {
    return this.withType(type)
}

internal fun <T> DomainObjectCollection<T>.configure(configurer: (T) -> Unit) {
    this.all(configurer)
}

internal fun <T, R : T> DomainObjectCollection<T>.configure(type: Class<R>, configurer: (R) -> Unit) {
    this[type].all(configurer)
}


// NamedDomainObjectCollection:

internal operator fun NamedDomainObjectCollection<*>.contains(name: String) = null != this.findByName(name)

internal operator fun <T> NamedDomainObjectCollection<T>.get(name: String): T {
    return this.getByName(name)
}

internal fun <T> NamedDomainObjectContainer<T>.getOrCreate(name: String, configurer: (T) -> Unit = {}): T {
    return findByName(name) ?: create(name, configurer)
}

internal fun <T> NamedDomainObjectContainer<T>.createIfNotExists(name: String, configurer: (T) -> Unit = {}) {
    getOrCreate(name, configurer)
}

internal fun <T> NamedDomainObjectCollection<T>.configure(name: String, configurer: (T) -> Unit) {
    val hasObjectWithName = AtomicBoolean(false)
    this.all {
        this.findByName(name)?.let {
            if (hasObjectWithName.compareAndSet(false, true)) {
                configurer(it)
            }
        }
    }
    this.whenObjectRemoved {
        if (null == this.findByName(name)) {
            hasObjectWithName.set(false)
        }
    }
}

internal operator fun <T, R : T> NamedDomainObjectCollection<T>.get(type: Class<R>): NamedDomainObjectCollection<R> {
    return this.withType(type)
}


// Convention:

internal operator fun <T : Any> Convention.get(name: String): T {
    @Suppress("UNCHECKED_CAST")
    return this.getByName(name) as T
}

internal operator fun <T : Any> Convention.get(type: Class<T>): T {
    val supressedException: Throwable
    try {
        return this.getPlugin(type)
    } catch (e: Throwable) {
        supressedException = e
    }

    try {
        return this.getByType(type)
    } catch (e: Throwable) {
        e.addSuppressed(supressedException)
        throw e
    }
}


// other:

internal operator fun <T : Task, R : T> TaskCollection<T>.get(type: Class<R>): TaskCollection<R> {
    return this.withType(type)
}

