package name.remal.building.gradle_plugins.utils

import org.w3c.dom.CDATASection
import org.w3c.dom.Comment
import org.w3c.dom.Document
import org.w3c.dom.Node

private val INDENT = ' '.repeat(2)

internal fun Node.writeTo(appendable: Appendable, doIndent: Boolean = false, baseIndent: Int = 0) {
    when (this) {
        is CDATASection -> this.writeTo(appendable)
        is Comment -> this.writeTo(appendable)
        is Document -> this.writeTo(appendable, doIndent, baseIndent)
        else -> throw UnsupportedOperationException("Unsupported node type: ${this::class.java}")
    }
}

internal fun CDATASection.writeTo(appendable: Appendable) {
    val text = this.textContent
    if (text.contains("]]>")) {
        text.splitToSequence("]]>")
            .forEachIndexed { index, part ->
                if (1 <= index) appendable.append("]]>")
                appendable.append("<![CDATA[").append(part).append("]]>")
            }
    } else {
        appendable.append("<![CDATA[").append(text).append("]]>")
    }
}

internal fun Comment.writeTo(appendable: Appendable) {
    val text = this.textContent
    if (text.contains("--")) throw IllegalStateException("'--' string (double-hyphen) must not occur within comments")
    appendable.append("<!--").append(text).append("-->")
}

@Suppress("UNUSED_PARAMETER")
internal fun Document.writeTo(appendable: Appendable, doIndent: Boolean = false, baseIndent: Int = 0) {
    appendable.append("<?xml version=\"").append(escapeXml(xmlVersion)).append("\" encoding=\"UTF-8\"?>")
    this.xmlEncoding
}


private fun Document.escapeXml(text: String?): String {
    if (null == text) return ""
    if (this.xmlVersion.isNullOrEmpty() || "1.0" == this.xmlVersion) {
        @Suppress("DEPRECATION")
        return org.apache.commons.lang3.StringEscapeUtils.escapeXml10(text)
    } else {
        @Suppress("DEPRECATION")
        return org.apache.commons.lang3.StringEscapeUtils.escapeXml11(text)
    }
}
