package name.remal.building.gradle_plugins.vcs

import name.remal.building.gradle_plugins.VCSAutoVersionExtension
import name.remal.building.gradle_plugins.dsl.Version
import name.remal.building.gradle_plugins.dsl.use
import org.eclipse.jgit.api.Git
import org.eclipse.jgit.lib.Constants
import org.eclipse.jgit.lib.Ref
import org.eclipse.jgit.storage.file.FileRepositoryBuilder
import java.io.File
import java.time.Instant
import java.time.LocalDateTime
import java.time.ZoneId

class GitVCSService(override val repositoryRootDir: File) : VCSService {

    override fun getCurrentCommitDateTime(): LocalDateTime? {
        FileRepositoryBuilder.create(repositoryRootDir.resolve(Constants.DOT_GIT)).use useRepository@ { repository ->
            Git(repository).use { git ->
                val refCommit = git.log().setMaxCount(1).call().firstOrNull() ?: return null
                return Instant.ofEpochSecond(refCommit.commitTime.toLong()).atZone(ZoneId.systemDefault()).toLocalDateTime()
            }
        }
    }

    override fun calculateVersion(vcsAutoVersion: VCSAutoVersionExtension): VCSVersion {
        FileRepositoryBuilder.create(repositoryRootDir.resolve(Constants.DOT_GIT)).use useRepository@ { repository ->
            Git(repository).use { git ->
                val versionTagPrefixes = vcsAutoVersion.versionTagPrefixes
                    .asSequence()
                    .flatMap { sequenceOf(it, "$it ", "$it-", "$it:", "$it/") }
                    .sortedDescending()
                val tags: Map<Version, Ref> = git.tagList().call().asSequence()
                    .mapNotNull { tagRef ->
                        val name = tagRef.name.substringAfter("refs/tags/")
                        versionTagPrefixes
                            .firstOrNull { prefix -> name.startsWith(prefix) }
                            ?.let { prefix -> name.substring(prefix.length).trim() }
                            ?.let { Version.parseOrNull(it) }
                            ?.let { it to tagRef }
                    }
                    .toMap()

                var isSpecified = true
                var commitsAfterBaseVersion = 0
                var baseVersion = Version()
                run {
                    git.log().call().forEach { revCommit ->
                        tags.entries
                            .firstOrNull { (_, tagRef) -> tagRef.peeledObjectId == revCommit.id || tagRef.objectId == revCommit.id }
                            ?.let { (tagVersion, _) ->
                                baseVersion = tagVersion
                                return@run
                            }
                        ++commitsAfterBaseVersion
                    }
                    isSpecified = false
                }

                baseVersion = baseVersion.withSuffix(null)

                val branchName = vcsAutoVersion.overwriteBranchName ?: repository.branch

                return VCSVersion(
                    isSpecified,
                    baseVersion,
                    branchName.let { if (vcsAutoVersion.mainBranchName != it) it else null },
                    commitsAfterBaseVersion,
                    git.status().call().hasUncommittedChanges()
                )
            }
        }
    }

}
