package name.remal.gradle_plugins

import name.remal.gradle_plugins.RemalGradlePlugins.AptPluginDisabledID
import name.remal.gradle_plugins.dsl.aptConfigurationName
import name.remal.gradle_plugins.dsl.createDirectories
import name.remal.gradle_plugins.dsl.default
import name.remal.gradle_plugins.dsl.java
import name.remal.gradle_plugins.utils.*
import name.remal.gradle_plugins.utils.PluginIds.JAVA_PLUGIN_ID
import org.gradle.api.Project
import org.gradle.api.tasks.compile.JavaCompile

@API
class AptPlugin : ProjectPlugin() {

    companion object {
        const val APT_ALL_CONFIGURATION_NAME = "aptAll"
        const val APT_OPTIONS_EXTENSION_NAME = "aptOptions"
    }

    override fun apply(project: Project) {
        if (project.isDisabledBy(AptPluginDisabledID)) return

        project.withPlugin(JAVA_PLUGIN_ID) {
            val aptAllConf = project.configurations.create(APT_ALL_CONFIGURATION_NAME) {
                it.description = "All apt configurations extends this configuration"
            }

            project.java.sourceSets.configure { sourceSet ->
                val compileClasspathConf = project.configurations.findByName(sourceSet.compileClasspathConfigurationName) ?: return@configure
                project.configurations.create(sourceSet.aptConfigurationName) {
                    it.description = "Configuration for annotation-processing dependencies for ${sourceSet.name} source set"
                    it.extendsFrom(aptAllConf)
                    compileClasspathConf.extendsFrom(it)
                }

                project.tasks[JavaCompile::class.java].configure(sourceSet.compileJavaTaskName) { task ->
                    val generatedSourcesDir = project.buildDir.resolve("generated/apt/${sourceSet.name}").createDirectories().absoluteFile
                    task.options.compilerArgs.addAll(listOf("-s", generatedSourcesDir.path))
                    sourceSet.allJava.srcDir(generatedSourcesDir)

                    task.doFirstOrdered(Int.MIN_VALUE) { generatedSourcesDir.createDirectories() }
                }
            }

            val aptOptions = project.extensions.create(APT_OPTIONS_EXTENSION_NAME, AptOptionsExtension::class.java)
            project.afterEvaluate {
                project.tasks.configure(JavaCompile::class.java) { task ->
                    aptOptions.processorArgs.forEach { name, value ->
                        task.options.compilerArgs.add("-A$name=${value?.toString().default()}")
                    }
                }
            }
        }
    }

}


@API
class AptOptionsExtension {

    var processorArgs: MutableMap<String, Any?> = LinkedHashMap()
        set(value) {
            field = LinkedHashMap(value)
        }

}
