package name.remal.gradle_plugins

import name.remal.gradle_plugins.RemalGradlePlugins.CheckDependencyUpdatesPluginDisabledID
import name.remal.gradle_plugins.dsl.concatWith
import name.remal.gradle_plugins.dsl.filterIsInstance
import name.remal.gradle_plugins.utils.*
import org.gradle.api.DefaultTask
import org.gradle.api.Project
import org.gradle.api.artifacts.*
import org.gradle.api.internal.artifacts.dsl.dependencies.DependencyFactory
import org.gradle.api.plugins.HelpTasksPlugin
import org.gradle.api.tasks.TaskAction
import java.util.*
import java.util.Collections.newSetFromMap
import java.util.concurrent.ConcurrentHashMap
import java.util.stream.Collectors.toSet
import java.util.stream.Stream
import javax.inject.Inject
import kotlin.collections.set

@API
class CheckDependencyUpdatesPlugin : ProjectPlugin() {

    companion object {
        const val CHECK_DEPENDENCY_UPDATES_TASK_NAME = "checkDependencyUpdates"
        const val CHECK_DEPENDENCY_UPDATES_CONFIGURATION_NAME = "checkDependencyUpdates"
    }

    override fun apply(project: Project) {
        if (project.isDisabledBy(CheckDependencyUpdatesPluginDisabledID)) return

        project.allprojects { it.getOrCreateDependencyUpdatesConfiguration() }
        project.tasks.create(CHECK_DEPENDENCY_UPDATES_TASK_NAME, DependencyUpdatesTask::class.java)
    }

    private fun Project.getOrCreateDependencyUpdatesConfiguration(): Configuration {
        return configurations.findByName(CHECK_DEPENDENCY_UPDATES_CONFIGURATION_NAME) ?: configurations.create(CHECK_DEPENDENCY_UPDATES_CONFIGURATION_NAME) {
            it.isCanBeConsumed = false
            it.isVisible = false
            it.description = "Configuration for check dependency updates"

            it.makeNotTransitive()
        }
    }

}


@API
class DependencyUpdatesTask : DefaultTask() {

    @get:Inject
    protected val dependencyFactory: DependencyFactory
        get() = throw AbstractMethodError()

    init {
        group = HelpTasksPlugin.HELP_GROUP
        description = "Displays the dependency updates for the project"

        onlyIf { !project.gradle.startParameter.isOffline }
    }

    var notCheckedDependencies: MutableSet<String> = mutableSetOf()
    var checkAllVersionsDependencies: MutableSet<String> = mutableSetOf()
    var invalidVersionTokens: MutableSet<String> = mutableSetOf("alpha", "beta", "rc", "cr", "m", "redhat", "b", "pr", "dev")

    @TaskAction
    fun doCheckDependencyUpdates() {
        val embeddedPluginsDependencies = ProjectInfo.EMBEDDED_PLUGINS.toSet()

        val notCheckedDependencies = this.notCheckedDependencies.toSet()
        val checkAllVersionsDependencies = this.checkAllVersionsDependencies.toSet()
        val invalidVersionTokens = this.invalidVersionTokens.toSet()

        val notResolvedSelectors: MutableSet<ModuleVersionSelector> = newSetFromMap(ConcurrentHashMap())
        val usedDependencyIds: MutableMap<DependencyKey, ModuleVersionIdentifier> = ConcurrentHashMap()
        val latestDependencyIds: MutableMap<DependencyKey, ModuleVersionIdentifier> = ConcurrentHashMap()

        project.allprojects.parallelStream()
            .flatMap {
                Stream.of(
                    it.buildscript.configurations,
                    it.configurations
                )
            }
            .peek forEachConfigurationContainer@ { configurations ->
                val dependenciesToResolve = configurations.parallelStream()
                    .filter { it.dependencies.isNotEmpty() }
                    .map {
                        it.copy().apply {
                            isCanBeResolved = true
                            dependencies.retainAll { it.doCheckForNewVersions }
                            makeNotTransitive()
                        }
                    }
                    .filter { it.dependencies.isNotEmpty() }
                    .flatMap { it.resolvedConfiguration.lenientConfiguration.getFirstLevelModuleDependencies { it.doCheckForNewVersions }.stream() }
                    .map { it.module.id }
                    .peek { usedDependencyIds[it.dependencyKey] = it }
                    .map {
                        mapOf(
                            "group" to it.group,
                            "name" to it.name,
                            "version" to "+"
                        )
                    }
                    .concatWith(
                        embeddedPluginsDependencies.stream().map {
                            val (group, name) = it.split(':')
                            mapOf(
                                "group" to group,
                                "name" to name,
                                "version" to "+"
                            )
                        }
                    )
                    .distinct()
                    .map { dependencyFactory.createDependency(it) }
                    .filterIsInstance(ExternalModuleDependency::class.java)
                    .peek { it.isTransitive = false }
                    .collect(toSet())
                if (dependenciesToResolve.isEmpty()) return@forEachConfigurationContainer

                val resolveConf = configurations.detachedConfiguration(*dependenciesToResolve.toTypedArray()).apply {
                    makeNotTransitive()
                    resolutionStrategy {
                        it.componentSelection {
                            it.all { selection ->
                                with(selection) {
                                    with(candidate) {
                                        val dependency = "$group:$module"

                                        if (dependency in notCheckedDependencies) {
                                            reject(dependency)
                                            return@all
                                        }

                                        if (dependency in checkAllVersionsDependencies) return@all

                                        invalidVersionTokens.firstOrNull { Regex(".*[._-]$it[._-]?\\d*(\\.\\d*)*([._-].*)?", RegexOption.IGNORE_CASE).matches(version) }?.let { token ->
                                            reject(token)
                                            return@all
                                        }
                                    }
                                }
                            }
                        }
                    }
                }

                with(resolveConf.resolvedConfiguration.lenientConfiguration) {
                    unresolvedModuleDependencies.forEach { notResolvedSelectors.add(it.selector) }
                    firstLevelModuleDependencies.forEach { it.module.id.apply { latestDependencyIds[dependencyKey] = this } }
                }
            }
            .forEach {}

        notResolvedSelectors.stream()
            .flatMap { sel ->
                if (embeddedPluginsDependencies.any { it.startsWith("${sel.group}:${sel.name}:") }) {
                    Stream.of()
                } else {
                    Stream.of(sel)
                }
            }
            .map { "${it.group}:${it.name}:${it.version}" }
            .distinct()
            .sorted()
            .forEach { logger.warn("Dependency not resolved: $it") }

        TreeMap(latestDependencyIds).forEach { key, latestId ->
            val usedId = usedDependencyIds[key] ?: return@forEach
            if (latestId.version != usedId.version) {
                val messageSuffix = if (embeddedPluginsDependencies.any { it.startsWith("${usedId.group}:${usedId.name}:") }) {
                    " (embedded into ${ProjectInfo.PROJECT_GROUP}:${ProjectInfo.PROJECT_NAME})"
                } else {
                    ""
                }
                logger.warn("New dependency version: ${usedId.group}:${usedId.name}: ${usedId.version} -> ${latestId.version}$messageSuffix")
            }
        }

        didWork = true
    }

    private val Dependency.doCheckForNewVersions: Boolean get() = this is ExternalModuleDependency && !isForce && "+" != version

}
